/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *      By Shawn Hargreaves,
 *      1 Salisbury Road,
 *      Market Drayton,
 *      Shropshire,
 *      England, TF9 1AJ.
 *
 *      Text drawing routines.
 *
 *      textout_justify() by Seymour Shlien.
 *
 *      See readme.txt for copyright information.
 */


#include <stdlib.h>
#include <stdio.h>
#include <stddef.h>
#include <stdarg.h>
#include <string.h>

#include "allegro.h"
#include "internal.h"



static FONT_8x8 _font_8x8 =                     /* the default 8x8 font */
{
   {
      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
      { 0x18, 0x3C, 0x3C, 0x18, 0x18, 0x00, 0x18, 0x00 },
      { 0x6C, 0x6C, 0x6C, 0x00, 0x00, 0x00, 0x00, 0x00 },
      { 0x6C, 0x6C, 0xFE, 0x6C, 0xFE, 0x6C, 0x6C, 0x00 },
      { 0x18, 0x7E, 0xC0, 0x7C, 0x06, 0xFC, 0x18, 0x00 },
      { 0x00, 0xC6, 0xCC, 0x18, 0x30, 0x66, 0xC6, 0x00 },
      { 0x38, 0x6C, 0x38, 0x76, 0xDC, 0xCC, 0x76, 0x00 },
      { 0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00 },
      { 0x18, 0x30, 0x60, 0x60, 0x60, 0x30, 0x18, 0x00 },
      { 0x60, 0x30, 0x18, 0x18, 0x18, 0x30, 0x60, 0x00 },
      { 0x00, 0x66, 0x3C, 0xFF, 0x3C, 0x66, 0x00, 0x00 },
      { 0x00, 0x18, 0x18, 0x7E, 0x18, 0x18, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x30 },
      { 0x00, 0x00, 0x00, 0x7E, 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00 },
      { 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0x80, 0x00 },
      { 0x7C, 0xCE, 0xDE, 0xF6, 0xE6, 0xC6, 0x7C, 0x00 },
      { 0x30, 0x70, 0x30, 0x30, 0x30, 0x30, 0xFC, 0x00 },
      { 0x78, 0xCC, 0x0C, 0x38, 0x60, 0xCC, 0xFC, 0x00 },
      { 0x78, 0xCC, 0x0C, 0x38, 0x0C, 0xCC, 0x78, 0x00 },
      { 0x1C, 0x3C, 0x6C, 0xCC, 0xFE, 0x0C, 0x1E, 0x00 },
      { 0xFC, 0xC0, 0xF8, 0x0C, 0x0C, 0xCC, 0x78, 0x00 },
      { 0x38, 0x60, 0xC0, 0xF8, 0xCC, 0xCC, 0x78, 0x00 },
      { 0xFC, 0xCC, 0x0C, 0x18, 0x30, 0x30, 0x30, 0x00 },
      { 0x78, 0xCC, 0xCC, 0x78, 0xCC, 0xCC, 0x78, 0x00 },
      { 0x78, 0xCC, 0xCC, 0x7C, 0x0C, 0x18, 0x70, 0x00 },
      { 0x00, 0x18, 0x18, 0x00, 0x00, 0x18, 0x18, 0x00 },
      { 0x00, 0x18, 0x18, 0x00, 0x00, 0x18, 0x18, 0x30 },
      { 0x18, 0x30, 0x60, 0xC0, 0x60, 0x30, 0x18, 0x00 },
      { 0x00, 0x00, 0x7E, 0x00, 0x7E, 0x00, 0x00, 0x00 },
      { 0x60, 0x30, 0x18, 0x0C, 0x18, 0x30, 0x60, 0x00 },
      { 0x3C, 0x66, 0x0C, 0x18, 0x18, 0x00, 0x18, 0x00 },
      { 0x7C, 0xC6, 0xDE, 0xDE, 0xDC, 0xC0, 0x7C, 0x00 },
      { 0x30, 0x78, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0x00 },
      { 0xFC, 0x66, 0x66, 0x7C, 0x66, 0x66, 0xFC, 0x00 },
      { 0x3C, 0x66, 0xC0, 0xC0, 0xC0, 0x66, 0x3C, 0x00 },
      { 0xF8, 0x6C, 0x66, 0x66, 0x66, 0x6C, 0xF8, 0x00 },
      { 0xFE, 0x62, 0x68, 0x78, 0x68, 0x62, 0xFE, 0x00 },
      { 0xFE, 0x62, 0x68, 0x78, 0x68, 0x60, 0xF0, 0x00 },
      { 0x3C, 0x66, 0xC0, 0xC0, 0xCE, 0x66, 0x3A, 0x00 },
      { 0xCC, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0xCC, 0x00 },
      { 0x78, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00 },
      { 0x1E, 0x0C, 0x0C, 0x0C, 0xCC, 0xCC, 0x78, 0x00 },
      { 0xE6, 0x66, 0x6C, 0x78, 0x6C, 0x66, 0xE6, 0x00 },
      { 0xF0, 0x60, 0x60, 0x60, 0x62, 0x66, 0xFE, 0x00 },
      { 0xC6, 0xEE, 0xFE, 0xFE, 0xD6, 0xC6, 0xC6, 0x00 },
      { 0xC6, 0xE6, 0xF6, 0xDE, 0xCE, 0xC6, 0xC6, 0x00 },
      { 0x38, 0x6C, 0xC6, 0xC6, 0xC6, 0x6C, 0x38, 0x00 },
      { 0xFC, 0x66, 0x66, 0x7C, 0x60, 0x60, 0xF0, 0x00 },
      { 0x7C, 0xC6, 0xC6, 0xC6, 0xD6, 0x7C, 0x0E, 0x00 },
      { 0xFC, 0x66, 0x66, 0x7C, 0x6C, 0x66, 0xE6, 0x00 },
      { 0x7C, 0xC6, 0xE0, 0x78, 0x0E, 0xC6, 0x7C, 0x00 },
      { 0xFC, 0xB4, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00 },
      { 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xFC, 0x00 },
      { 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x00 },
      { 0xC6, 0xC6, 0xC6, 0xC6, 0xD6, 0xFE, 0x6C, 0x00 },
      { 0xC6, 0xC6, 0x6C, 0x38, 0x6C, 0xC6, 0xC6, 0x00 },
      { 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x30, 0x78, 0x00 },
      { 0xFE, 0xC6, 0x8C, 0x18, 0x32, 0x66, 0xFE, 0x00 },
      { 0x78, 0x60, 0x60, 0x60, 0x60, 0x60, 0x78, 0x00 },
      { 0xC0, 0x60, 0x30, 0x18, 0x0C, 0x06, 0x02, 0x00 },
      { 0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x78, 0x00 },
      { 0x10, 0x38, 0x6C, 0xC6, 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF },
      { 0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x76, 0x00 },
      { 0xE0, 0x60, 0x60, 0x7C, 0x66, 0x66, 0xDC, 0x00 },
      { 0x00, 0x00, 0x78, 0xCC, 0xC0, 0xCC, 0x78, 0x00 },
      { 0x1C, 0x0C, 0x0C, 0x7C, 0xCC, 0xCC, 0x76, 0x00 },
      { 0x00, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00 },
      { 0x38, 0x6C, 0x64, 0xF0, 0x60, 0x60, 0xF0, 0x00 },
      { 0x00, 0x00, 0x76, 0xCC, 0xCC, 0x7C, 0x0C, 0xF8 },
      { 0xE0, 0x60, 0x6C, 0x76, 0x66, 0x66, 0xE6, 0x00 },
      { 0x30, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00 },
      { 0x0C, 0x00, 0x1C, 0x0C, 0x0C, 0xCC, 0xCC, 0x78 },
      { 0xE0, 0x60, 0x66, 0x6C, 0x78, 0x6C, 0xE6, 0x00 },
      { 0x70, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00 },
      { 0x00, 0x00, 0xCC, 0xFE, 0xFE, 0xD6, 0xD6, 0x00 },
      { 0x00, 0x00, 0xB8, 0xCC, 0xCC, 0xCC, 0xCC, 0x00 },
      { 0x00, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0x78, 0x00 },
      { 0x00, 0x00, 0xDC, 0x66, 0x66, 0x7C, 0x60, 0xF0 },
      { 0x00, 0x00, 0x76, 0xCC, 0xCC, 0x7C, 0x0C, 0x1E },
      { 0x00, 0x00, 0xDC, 0x76, 0x62, 0x60, 0xF0, 0x00 },
      { 0x00, 0x00, 0x7C, 0xC0, 0x70, 0x1C, 0xF8, 0x00 },
      { 0x10, 0x30, 0xFC, 0x30, 0x30, 0x34, 0x18, 0x00 },
      { 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x76, 0x00 },
      { 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x00 },
      { 0x00, 0x00, 0xC6, 0xC6, 0xD6, 0xFE, 0x6C, 0x00 },
      { 0x00, 0x00, 0xC6, 0x6C, 0x38, 0x6C, 0xC6, 0x00 },
      { 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x7C, 0x0C, 0xF8 },
      { 0x00, 0x00, 0xFC, 0x98, 0x30, 0x64, 0xFC, 0x00 },
      { 0x1C, 0x30, 0x30, 0xE0, 0x30, 0x30, 0x1C, 0x00 },
      { 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00 },
      { 0xE0, 0x30, 0x30, 0x1C, 0x30, 0x30, 0xE0, 0x00 },
      { 0x76, 0xDC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x00, 0x10, 0x38, 0x6C, 0xC6, 0xC6, 0xFE, 0x00 },
      { 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00 },
      { 0x18, 0x18, 0x7E, 0xC0, 0xC0, 0x7E, 0x18, 0x18 },
      { 0x38, 0x6C, 0x64, 0xF0, 0x60, 0xE6, 0xFC, 0x00 },
      { 0x00, 0xC6, 0x7C, 0xC6, 0xC6, 0x7C, 0xC6, 0x00 },
      { 0xCC, 0xCC, 0x78, 0xFC, 0x30, 0xFC, 0x30, 0x30 },
      { 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00 },
      { 0x3E, 0x61, 0x3C, 0x66, 0x66, 0x3C, 0x86, 0x7C },
      { 0x00, 0xC6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
      { 0x7E, 0x81, 0x9D, 0xA1, 0xA1, 0x9D, 0x81, 0x7E },
      { 0x3C, 0x6C, 0x6C, 0x3E, 0x00, 0x7E, 0x00, 0x00 },
      { 0x00, 0x33, 0x66, 0xCC, 0x66, 0x33, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0xFC, 0x0C, 0x0C, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x7E, 0x00, 0x00, 0x00, 0x00 },
      { 0x7E, 0x81, 0xB9, 0xA5, 0xB9, 0xA5, 0x81, 0x7E },
      { 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
      { 0x38, 0x6C, 0x6C, 0x38, 0x00, 0x00, 0x00, 0x00 },
      { 0x30, 0x30, 0xFC, 0x30, 0x30, 0x00, 0xFC, 0x00 },
      { 0x70, 0x18, 0x30, 0x60, 0x78, 0x00, 0x00, 0x00 },
      { 0x78, 0x0C, 0x38, 0x0C, 0x78, 0x00, 0x00, 0x00 },
      { 0x0C, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x33, 0x33, 0x66, 0x7E, 0xC0, 0x80 },
      { 0x7F, 0xDB, 0xDB, 0x7B, 0x1B, 0x1B, 0x1B, 0x00 },
      { 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x0C, 0x38 },
      { 0x18, 0x38, 0x18, 0x18, 0x3C, 0x00, 0x00, 0x00 },
      { 0x38, 0x6C, 0x6C, 0x38, 0x00, 0x7C, 0x00, 0x00 },
      { 0x00, 0xCC, 0x66, 0x33, 0x66, 0xCC, 0x00, 0x00 },
      { 0xC3, 0xC6, 0xCC, 0xDB, 0x37, 0x6F, 0xCF, 0x03 },
      { 0xC3, 0xC6, 0xCC, 0xDE, 0x33, 0x66, 0xCC, 0x0F },
      { 0xE1, 0x32, 0xE4, 0x3A, 0xF6, 0x2A, 0x5F, 0x86 },
      { 0x30, 0x00, 0x30, 0x60, 0xC0, 0xCC, 0x78, 0x00 },
      { 0x18, 0x0C, 0x38, 0x6C, 0xC6, 0xFE, 0xC6, 0x00 },
      { 0x30, 0x60, 0x38, 0x6C, 0xC6, 0xFE, 0xC6, 0x00 },
      { 0x7C, 0x82, 0x38, 0x6C, 0xC6, 0xFE, 0xC6, 0x00 },
      { 0x76, 0xDC, 0x38, 0x6C, 0xC6, 0xFE, 0xC6, 0x00 },
      { 0xC6, 0x00, 0x38, 0x6C, 0xC6, 0xFE, 0xC6, 0x00 },
      { 0x10, 0x28, 0x38, 0x6C, 0xC6, 0xFE, 0xC6, 0x00 },
      { 0x3E, 0x6C, 0xCC, 0xFE, 0xCC, 0xCC, 0xCE, 0x00 },
      { 0x78, 0xCC, 0xC0, 0xCC, 0x78, 0x18, 0x0C, 0x78 },
      { 0x30, 0x18, 0xFE, 0xC0, 0xFC, 0xC0, 0xFE, 0x00 },
      { 0x0C, 0x18, 0xFE, 0xC0, 0xFC, 0xC0, 0xFE, 0x00 },
      { 0x7C, 0x82, 0xFE, 0xC0, 0xFC, 0xC0, 0xFE, 0x00 },
      { 0xC6, 0x00, 0xFE, 0xC0, 0xFC, 0xC0, 0xFE, 0x00 },
      { 0x30, 0x18, 0x3C, 0x18, 0x18, 0x18, 0x3C, 0x00 },
      { 0x0C, 0x18, 0x3C, 0x18, 0x18, 0x18, 0x3C, 0x00 },
      { 0x3C, 0x42, 0x3C, 0x18, 0x18, 0x18, 0x3C, 0x00 },
      { 0x66, 0x00, 0x3C, 0x18, 0x18, 0x18, 0x3C, 0x00 },
      { 0xF8, 0x6C, 0x66, 0xF6, 0x66, 0x6C, 0xF8, 0x00 },
      { 0xFC, 0x00, 0xCC, 0xEC, 0xFC, 0xDC, 0xCC, 0x00 },
      { 0x30, 0x18, 0x7C, 0xC6, 0xC6, 0xC6, 0x7C, 0x00 },
      { 0x18, 0x30, 0x7C, 0xC6, 0xC6, 0xC6, 0x7C, 0x00 },
      { 0x7C, 0x82, 0x7C, 0xC6, 0xC6, 0xC6, 0x7C, 0x00 },
      { 0x76, 0xDC, 0x7C, 0xC6, 0xC6, 0xC6, 0x7C, 0x00 },
      { 0xC6, 0x00, 0x7C, 0xC6, 0xC6, 0xC6, 0x7C, 0x00 },
      { 0x00, 0xC6, 0x6C, 0x38, 0x6C, 0xC6, 0x00, 0x00 },
      { 0x3A, 0x6C, 0xCE, 0xD6, 0xE6, 0x6C, 0xB8, 0x00 },
      { 0x60, 0x30, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00 },
      { 0x18, 0x30, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00 },
      { 0x7C, 0x82, 0x00, 0xC6, 0xC6, 0xC6, 0x7C, 0x00 },
      { 0xC6, 0x00, 0xC6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00 },
      { 0x0C, 0x18, 0x66, 0x66, 0x3C, 0x18, 0x3C, 0x00 },
      { 0xE0, 0x60, 0x7C, 0x66, 0x66, 0x7C, 0x60, 0xF0 },
      { 0x78, 0xCC, 0xCC, 0xD8, 0xCC, 0xC6, 0xCC, 0x00 },
      { 0xE0, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00 },
      { 0x1C, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00 },
      { 0x7E, 0xC3, 0x3C, 0x06, 0x3E, 0x66, 0x3F, 0x00 },
      { 0x76, 0xDC, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00 },
      { 0xCC, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00 },
      { 0x30, 0x30, 0x78, 0x0C, 0x7C, 0xCC, 0x7E, 0x00 },
      { 0x00, 0x00, 0x7F, 0x0C, 0x7F, 0xCC, 0x7F, 0x00 },
      { 0x00, 0x00, 0x78, 0xC0, 0xC0, 0x78, 0x0C, 0x38 },
      { 0xE0, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00 },
      { 0x1C, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00 },
      { 0x7E, 0xC3, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00 },
      { 0xCC, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00 },
      { 0xE0, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00 },
      { 0x38, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00 },
      { 0x7C, 0xC6, 0x38, 0x18, 0x18, 0x18, 0x3C, 0x00 },
      { 0xCC, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00 },
      { 0x08, 0x3C, 0x08, 0x7C, 0xCC, 0xCC, 0x78, 0x00 },
      { 0x00, 0xF8, 0x00, 0xF8, 0xCC, 0xCC, 0xCC, 0x00 },
      { 0x00, 0xE0, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00 },
      { 0x00, 0x1C, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00 },
      { 0x78, 0xCC, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00 },
      { 0x76, 0xDC, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00 },
      { 0x00, 0xCC, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00 },
      { 0x30, 0x30, 0x00, 0xFC, 0x00, 0x30, 0x30, 0x00 },
      { 0x00, 0x02, 0x7C, 0xCE, 0xD6, 0xE6, 0x7C, 0x80 },
      { 0x00, 0xE0, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00 },
      { 0x00, 0x1C, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00 },
      { 0x78, 0xCC, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00 },
      { 0x00, 0xCC, 0x00, 0xCC, 0xCC, 0xCC, 0x7E, 0x00 },
      { 0x18, 0x30, 0xCC, 0xCC, 0xCC, 0x7C, 0x0C, 0xF8 },
      { 0xF0, 0x60, 0x7C, 0x66, 0x7C, 0x60, 0xF0, 0x00 },
      { 0x00, 0xCC, 0x00, 0xCC, 0xCC, 0x7C, 0x0C, 0xF8 }
   }
};


static FONT _font =
{
   8,
   {
      &_font_8x8
   }
};


FONT *font = &_font;

int _textmode = 0;



/* text_mode:
 *  Sets the mode in which text will be drawn. If mode is positive, text
 *  output will be opaque and the background will be set to mode. If mode
 *  is negative, text will be drawn transparently (ie. the background will
 *  not be altered). The default is a mode of zero.
 */
void text_mode(int mode)
{
   if (mode < 0)
      _textmode = -1;
   else
      _textmode = mode;
}



/* blit_character:
 *  Helper routine for opaque multicolor output of proportional fonts.
 */
static void blit_character(BITMAP *bmp, BITMAP *sprite, int x, int y)
{
   blit(sprite, bmp, 0, 0, x, y, sprite->w, sprite->h);
}



/* textout:
 *  Writes the null terminated string str onto bmp at position x, y, using 
 *  the current text mode and the specified font and foreground color.
 *  If color is -1 and a proportional font is in use, it will be drawn
 *  using the colors from the original font bitmap (the one imported into
 *  the grabber program), which allows multicolored text output.
 */
void textout(BITMAP *bmp, FONT *f, unsigned char *str, int x, int y, int color)
{
   FONT_PROP *fp;
   BITMAP *b;
   int c;
   void (*putter)();

   if (f->height == 8) {
      bmp->vtable->textout_fixed(bmp, f->dat.dat_8x8, 3, str, x, y, color);
      return;
   }
   else if (f->height == 16) {
      bmp->vtable->textout_fixed(bmp, f->dat.dat_8x16, 4, str, x, y, color);
      return;
   }

   fp = f->dat.dat_prop;

   if (color < 0) {
      if (_textmode < 0)
	 putter = bmp->vtable->draw_256_sprite;
      else
	 putter = blit_character; 
   }
   else
      putter = bmp->vtable->draw_character; 

   while (*str) {
      c = (int)*str - ' ';
      if ((c < 0) || (c >= FONT_SIZE))
	 c = 0;
      b = fp->dat[c];
      if (b) {
	 putter(bmp, b, x, y, color);
	 x += b->w;
	 if (x >= bmp->cr)
	    return;
      }
      str++;
   }
}



/* textout_centre:
 *  Like textout(), but uses the x coordinate as the centre rather than 
 *  the left of the string.
 */
void textout_centre(BITMAP *bmp, FONT *f, unsigned char *str, int x, int y, int color)
{
   int len;

   len = text_length(f, str);
   textout(bmp, f, str, x - len/2, y, color);
}



/* textout_justify:
 *  Like textout(), but right justifies the string to the specified area.
 */
void textout_justify(BITMAP *bmp, FONT *f, unsigned char *str, int x1, int x2, int y, int diff, int color)
{
   char *tok[128];
   char strbuf[256];
   int i, minlen, last, space;
   float fleft, finc;

   /* count words and measure min length (without spaces) */ 
   minlen = 0;
   strcpy(strbuf, str);
   last = 0;
   for (tok[last] = strtok(strbuf, " \t\n\r"); tok[last]; tok[last] = strtok(0, " \t\n\r")) {
      minlen += text_length(f, tok[last]);
      last++; 
   }

   /* amount of room for space between words */
   space = x2 - x1 - minlen;

   if ((space <= 0) || (space > diff) || (last < 2)) {
      /* can't justify */
      textout(bmp, f, str, x1, y, color);
      return; 
   }

   /* distribute space left evenly between words */
   fleft = (float)x1;
   finc = (float)space / (float)(last-1);
   for (i=0; i<last; i++) {
      textout(bmp, f, tok[i], (int)fleft, y, color);
      fleft += (float)text_length(f, tok[i]) + finc;
   }
}



/* textprintf:
 *  Formatted text output, using a printf() style format string.
 */
void textprintf(BITMAP *bmp, FONT *f, int x, int y, int color, char *format, ...)
{
   char buf[256];

   va_list ap;
   va_start(ap, format);
   vsprintf(buf, format, ap);
   va_end(ap);

   textout(bmp, f, buf, x, y, color);
}



/* textprintf_centre:
 *  Like textprintf(), but uses the x coordinate as the centre rather than 
 *  the left of the string.
 */
void textprintf_centre(BITMAP *bmp, FONT *f, int x, int y, int color, char *format, ...)
{
   char buf[256];

   va_list ap;
   va_start(ap, format);
   vsprintf(buf, format, ap);
   va_end(ap);

   textout_centre(bmp, f, buf, x, y, color);
}



/* text_length:
 *  Calculates the length of a string in a particular font.
 */
int text_length(FONT *f, unsigned char *str)
{
   FONT_PROP *fp;
   int c;
   int len;

   if (f->height > 0)
      return strlen(str) * 8;

   fp = f->dat.dat_prop;
   len = 0;

   while (*str) {
      c = (int)*str - ' ';
      if ((c < 0) || (c >= FONT_SIZE))
	 c = 0;
      if (fp->dat[c])
	 len += fp->dat[c]->w;
      str++;
   }

   return len;
}



/* text_height:
 *  Returns the height of a character in the specified font.
 */
int text_height(FONT *f)
{
   return (f->height > 0) ? f->height : f->dat.dat_prop->dat[0]->h;
}



/* destroy_font:
 *  Frees the memory being used by a font structure.
 */
void destroy_font(FONT *f)
{
   FONT_PROP *fp;
   int c;

   if (f) {
      if (f->height == 8) {
	 /* free 8x8 font */
	 if (f->dat.dat_8x8)
	    free(f->dat.dat_8x8);
      }
      else if (f->height == 16) {
	 /* free 8x16 font */
	 if (f->dat.dat_8x16)
	    free(f->dat.dat_8x16);
      }
      else {
	 /* free proportional font */
	 fp = f->dat.dat_prop;
	 if (fp) {
	    for (c=0; c<FONT_SIZE; c++) {
	       if (fp->dat[c])
		  destroy_bitmap(fp->dat[c]);
	    }
	   free(fp); 
	 }
      }
      free(f);
   }
}

