/**
 * Copyright 2023 Google Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { EventEmitter } from '../common/EventEmitter.js';
import { ElementHandle, ClickOptions } from './ElementHandle.js';
import type { Page } from './Page.js';
/**
 * @internal
 */
export interface LocatorOptions {
    /**
     * Whether to wait for the element to be `visible` or `hidden`.
     */
    visibility: 'hidden' | 'visible';
    /**
     * Total timeout for the entire locator operation.
     */
    timeout: number;
}
/**
 * @internal
 */
type ActionCondition = (element: ElementHandle, signal: AbortSignal) => Promise<void>;
/**
 * @internal
 */
export interface ActionOptions {
    signal?: AbortSignal;
    conditions: ActionCondition[];
}
/**
 * All the events that a locator instance may emit.
 *
 * @internal
 */
export declare enum LocatorEmittedEvents {
    /**
     * Emitted every time before the locator performs an action on the located element(s).
     */
    Action = "action"
}
/**
 * @internal
 */
export interface LocatorEventObject {
    [LocatorEmittedEvents.Action]: never;
}
/**
 * Locators describe a strategy of locating elements and performing an action on
 * them. If the action fails because the element are not ready for the action,
 * the whole operation is retried.
 *
 * @internal
 */
export declare class Locator extends EventEmitter {
    #private;
    constructor(page: Page, selector: string, options?: LocatorOptions);
    on<K extends keyof LocatorEventObject>(eventName: K, handler: (event: LocatorEventObject[K]) => void): Locator;
    once<K extends keyof LocatorEventObject>(eventName: K, handler: (event: LocatorEventObject[K]) => void): Locator;
    off<K extends keyof LocatorEventObject>(eventName: K, handler: (event: LocatorEventObject[K]) => void): Locator;
    click(clickOptions?: ClickOptions & {
        signal?: AbortSignal;
    }): Promise<void>;
}
export {};
//# sourceMappingURL=Locator.d.ts.map