/**
 * Copyright 2023 Google Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __classPrivateFieldSet = (this && this.__classPrivateFieldSet) || function (receiver, state, value, kind, f) {
    if (kind === "m") throw new TypeError("Private method is not writable");
    if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a setter");
    if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot write private member to an object whose class did not declare it");
    return (kind === "a" ? f.call(receiver, value) : f ? f.value = value : state.set(receiver, value)), value;
};
var __classPrivateFieldGet = (this && this.__classPrivateFieldGet) || function (receiver, state, kind, f) {
    if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a getter");
    if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot read private member from an object whose class did not declare it");
    return kind === "m" ? f : kind === "a" ? f.call(receiver) : f ? f.value : state.get(receiver);
};
var _Locator_instances, _Locator_page, _Locator_selector, _Locator_options, _Locator_waitForFunction, _Locator_ensureElementIsInTheViewport, _Locator_waitForVisibility, _Locator_waitForEnabled, _Locator_waitForStableBoundingBox, _Locator_run;
import { TimeoutError } from '../common/Errors.js';
import { EventEmitter } from '../common/EventEmitter.js';
import { debugError } from '../common/util.js';
import { isErrorLike } from '../util/ErrorLike.js';
/**
 * Timeout for individual operations inside the locator. On errors the
 * operation is retried as long as {@link LocatorOptions.timeout} is not
 * exceeded. This timeout should be generally much lower as locating an
 * element means multiple asynchronious operations.
 */
const CONDITION_TIMEOUT = 1000;
const WAIT_FOR_FUNCTION_DELAY = 100;
/**
 * All the events that a locator instance may emit.
 *
 * @internal
 */
export var LocatorEmittedEvents;
(function (LocatorEmittedEvents) {
    /**
     * Emitted every time before the locator performs an action on the located element(s).
     */
    LocatorEmittedEvents["Action"] = "action";
})(LocatorEmittedEvents || (LocatorEmittedEvents = {}));
/**
 * Locators describe a strategy of locating elements and performing an action on
 * them. If the action fails because the element are not ready for the action,
 * the whole operation is retried.
 *
 * @internal
 */
export class Locator extends EventEmitter {
    constructor(page, selector, options = {
        visibility: 'visible',
        timeout: page.getDefaultTimeout(),
    }) {
        super();
        _Locator_instances.add(this);
        _Locator_page.set(this, void 0);
        _Locator_selector.set(this, void 0);
        _Locator_options.set(this, void 0);
        /**
         * Checks if the element is in the viewport and auto-scrolls it if it is not.
         */
        _Locator_ensureElementIsInTheViewport.set(this, async (element, signal) => {
            // Side-effect: this also checks if it is connected.
            const isIntersectingViewport = await element.isIntersectingViewport({
                threshold: 0,
            });
            signal === null || signal === void 0 ? void 0 : signal.throwIfAborted();
            if (!isIntersectingViewport) {
                await element.scrollIntoView();
                signal === null || signal === void 0 ? void 0 : signal.throwIfAborted();
                await __classPrivateFieldGet(this, _Locator_instances, "m", _Locator_waitForFunction).call(this, async () => {
                    return await element.isIntersectingViewport({
                        threshold: 0,
                    });
                }, signal);
                signal === null || signal === void 0 ? void 0 : signal.throwIfAborted();
            }
        });
        /**
         * Waits for the element to become visible or hidden. visibility === 'visible'
         * means that the element has a computed style, the visibility property other
         * than 'hidden' or 'collapse' and non-empty bounding box. visibility ===
         * 'hidden' means the opposite of that.
         */
        _Locator_waitForVisibility.set(this, async (element, signal) => {
            if (__classPrivateFieldGet(this, _Locator_options, "f").visibility === 'hidden') {
                await __classPrivateFieldGet(this, _Locator_instances, "m", _Locator_waitForFunction).call(this, async () => {
                    return element.isHidden();
                }, signal);
            }
            await __classPrivateFieldGet(this, _Locator_instances, "m", _Locator_waitForFunction).call(this, async () => {
                return element.isVisible();
            }, signal);
        });
        /**
         * If the element is a button, textarea, input or select, wait till the
         * element becomes enabled.
         */
        _Locator_waitForEnabled.set(this, async (element, signal) => {
            await __classPrivateFieldGet(this, _Locator_page, "f").waitForFunction(el => {
                if (['button', 'textarea', 'input', 'select'].includes(el.tagName)) {
                    return !el.disabled;
                }
                return true;
            }, {
                timeout: CONDITION_TIMEOUT,
                signal,
            }, element);
        });
        /**
         * Compares the bounding box of the element for two consecutive animation
         * frames and waits till they are the same.
         */
        _Locator_waitForStableBoundingBox.set(this, async (element, signal) => {
            function getClientRect() {
                return element.evaluate(el => {
                    return new Promise(resolve => {
                        window.requestAnimationFrame(() => {
                            const rect1 = el.getBoundingClientRect();
                            window.requestAnimationFrame(() => {
                                const rect2 = el.getBoundingClientRect();
                                resolve([
                                    {
                                        x: rect1.x,
                                        y: rect1.y,
                                        width: rect1.width,
                                        height: rect1.height,
                                    },
                                    {
                                        x: rect2.x,
                                        y: rect2.y,
                                        width: rect2.width,
                                        height: rect2.height,
                                    },
                                ]);
                            });
                        });
                    });
                });
            }
            await __classPrivateFieldGet(this, _Locator_instances, "m", _Locator_waitForFunction).call(this, async () => {
                const [rect1, rect2] = await getClientRect();
                return (rect1.x === rect2.x &&
                    rect1.y === rect2.y &&
                    rect1.width === rect2.width &&
                    rect1.height === rect2.height);
            }, signal);
        });
        __classPrivateFieldSet(this, _Locator_page, page, "f");
        __classPrivateFieldSet(this, _Locator_selector, selector, "f");
        __classPrivateFieldSet(this, _Locator_options, options, "f");
    }
    on(eventName, handler) {
        return super.on(eventName, handler);
    }
    once(eventName, handler) {
        return super.once(eventName, handler);
    }
    off(eventName, handler) {
        return super.off(eventName, handler);
    }
    async click(clickOptions) {
        await __classPrivateFieldGet(this, _Locator_instances, "m", _Locator_run).call(this, async (element) => {
            await element.click(clickOptions);
        }, {
            signal: clickOptions === null || clickOptions === void 0 ? void 0 : clickOptions.signal,
            conditions: [
                __classPrivateFieldGet(this, _Locator_ensureElementIsInTheViewport, "f"),
                __classPrivateFieldGet(this, _Locator_waitForVisibility, "f"),
                __classPrivateFieldGet(this, _Locator_waitForEnabled, "f"),
                __classPrivateFieldGet(this, _Locator_waitForStableBoundingBox, "f"),
            ],
        });
    }
}
_Locator_page = new WeakMap(), _Locator_selector = new WeakMap(), _Locator_options = new WeakMap(), _Locator_ensureElementIsInTheViewport = new WeakMap(), _Locator_waitForVisibility = new WeakMap(), _Locator_waitForEnabled = new WeakMap(), _Locator_waitForStableBoundingBox = new WeakMap(), _Locator_instances = new WeakSet(), _Locator_waitForFunction = 
/**
 * Retries the `fn` until a truthy result is returned.
 */
async function _Locator_waitForFunction(fn, signal, timeout = CONDITION_TIMEOUT) {
    let isActive = true;
    let controller;
    // If the loop times out, we abort only the last iteration's controller.
    const timeoutId = setTimeout(() => {
        isActive = false;
        controller === null || controller === void 0 ? void 0 : controller.abort();
    }, timeout);
    // If the user's signal aborts, we abort the last iteration and the loop.
    signal === null || signal === void 0 ? void 0 : signal.addEventListener('abort', () => {
        controller === null || controller === void 0 ? void 0 : controller.abort();
        isActive = false;
    }, { once: true });
    while (isActive) {
        controller = new AbortController();
        try {
            const result = await fn(controller.signal);
            if (result) {
                clearTimeout(timeoutId);
                return;
            }
        }
        catch (err) {
            if (isErrorLike(err)) {
                debugError(err);
                // Retry on all timeouts.
                if (err instanceof TimeoutError) {
                    continue;
                }
                // Abort error are ignored as they only affect one iteration.
                if (err.name === 'AbortError') {
                    continue;
                }
            }
            throw err;
        }
        finally {
            // We abort any operations that might have been started by `fn`, because
            // the iteration is now over.
            controller.abort();
        }
        await new Promise(resolve => {
            return setTimeout(resolve, WAIT_FOR_FUNCTION_DELAY);
        });
    }
    signal === null || signal === void 0 ? void 0 : signal.throwIfAborted();
    throw new TimeoutError(`waitForFunction timed out. The timeout is ${timeout}ms.`);
}, _Locator_run = async function _Locator_run(action, options) {
    await __classPrivateFieldGet(this, _Locator_instances, "m", _Locator_waitForFunction).call(this, async (signal) => {
        // 1. Select the element without visibility checks.
        const element = await __classPrivateFieldGet(this, _Locator_page, "f").waitForSelector(__classPrivateFieldGet(this, _Locator_selector, "f"), {
            visible: false,
            timeout: __classPrivateFieldGet(this, _Locator_options, "f").timeout,
            signal,
        });
        // Retry if no element is found.
        if (!element) {
            return false;
        }
        try {
            signal === null || signal === void 0 ? void 0 : signal.throwIfAborted();
            // 2. Perform action specific checks.
            await Promise.all((options === null || options === void 0 ? void 0 : options.conditions.map(check => {
                return check(element, signal);
            })) || []);
            signal === null || signal === void 0 ? void 0 : signal.throwIfAborted();
            // 3. Perform the action
            this.emit(LocatorEmittedEvents.Action);
            await action(element);
            return true;
        }
        finally {
            void element.dispose().catch(debugError);
        }
    }, options === null || options === void 0 ? void 0 : options.signal, __classPrivateFieldGet(this, _Locator_options, "f").timeout);
};
//# sourceMappingURL=Locator.js.map