/**
 * Copyright 2023 Google Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __classPrivateFieldSet = (this && this.__classPrivateFieldSet) || function (receiver, state, value, kind, f) {
    if (kind === "m") throw new TypeError("Private method is not writable");
    if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a setter");
    if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot write private member to an object whose class did not declare it");
    return (kind === "a" ? f.call(receiver, value) : f ? f.value = value : state.set(receiver, value)), value;
};
var __classPrivateFieldGet = (this && this.__classPrivateFieldGet) || function (receiver, state, kind, f) {
    if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a getter");
    if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot read private member from an object whose class did not declare it");
    return kind === "m" ? f : kind === "a" ? f.call(receiver) : f ? f.value : state.get(receiver);
};
var _Context_instances, _Context_connection, _Context_url, _Context_id, _Context_parentId, _Context_evaluate;
import { assert } from '../../util/assert.js';
import { stringifyFunction } from '../../util/Function.js';
import { ProtocolError, TimeoutError } from '../Errors.js';
import { EventEmitter } from '../EventEmitter.js';
import { getSourcePuppeteerURLIfAvailable, isString, PuppeteerURL, setPageContent, waitWithTimeout, } from '../util.js';
import { ElementHandle } from './ElementHandle.js';
import { JSHandle } from './JSHandle.js';
import { BidiSerializer } from './Serializer.js';
import { createEvaluationError } from './utils.js';
const SOURCE_URL_REGEX = /^[\040\t]*\/\/[@#] sourceURL=\s*(\S*?)\s*$/m;
const getSourceUrlComment = (url) => {
    return `//# sourceURL=${url}`;
};
/**
 * @internal
 */
const lifeCycleToReadinessState = new Map([
    ['load', 'complete'],
    ['domcontentloaded', 'interactive'],
]);
/**
 * @internal
 */
const lifeCycleToSubscribedEvent = new Map([
    ['load', 'browsingContext.load'],
    ['domcontentloaded', 'browsingContext.domContentLoaded'],
]);
/**
 * @internal
 */
export class Context extends EventEmitter {
    constructor(connection, frameManager, result) {
        super();
        _Context_instances.add(this);
        _Context_connection.set(this, void 0);
        _Context_url.set(this, void 0);
        _Context_id.set(this, void 0);
        _Context_parentId.set(this, void 0);
        __classPrivateFieldSet(this, _Context_connection, connection, "f");
        this._frameManager = frameManager;
        __classPrivateFieldSet(this, _Context_id, result.context, "f");
        __classPrivateFieldSet(this, _Context_parentId, result.parent, "f");
        __classPrivateFieldSet(this, _Context_url, result.url, "f");
        this.on('browsingContext.fragmentNavigated', (info) => {
            __classPrivateFieldSet(this, _Context_url, info.url, "f");
        });
    }
    get connection() {
        return __classPrivateFieldGet(this, _Context_connection, "f");
    }
    get id() {
        return __classPrivateFieldGet(this, _Context_id, "f");
    }
    get parentId() {
        return __classPrivateFieldGet(this, _Context_parentId, "f");
    }
    async evaluateHandle(pageFunction, ...args) {
        return __classPrivateFieldGet(this, _Context_instances, "m", _Context_evaluate).call(this, false, pageFunction, ...args);
    }
    async evaluate(pageFunction, ...args) {
        return __classPrivateFieldGet(this, _Context_instances, "m", _Context_evaluate).call(this, true, pageFunction, ...args);
    }
    async goto(url, options = {}) {
        const { waitUntil = 'load', timeout = this._frameManager._timeoutSettings.navigationTimeout(), } = options;
        const readinessState = lifeCycleToReadinessState.get(getWaitUntilSingle(waitUntil));
        try {
            const response = await waitWithTimeout(this.connection.send('browsingContext.navigate', {
                url: url,
                context: this.id,
                wait: readinessState,
            }), 'Navigation', timeout);
            __classPrivateFieldSet(this, _Context_url, response.result.url, "f");
            return null;
        }
        catch (error) {
            if (error instanceof ProtocolError) {
                error.message += ` at ${url}`;
            }
            else if (error instanceof TimeoutError) {
                error.message = 'Navigation timeout of ' + timeout + ' ms exceeded';
            }
            throw error;
        }
    }
    url() {
        return __classPrivateFieldGet(this, _Context_url, "f");
    }
    async setContent(html, options = {}) {
        const { waitUntil = 'load', timeout = this._frameManager._timeoutSettings.navigationTimeout(), } = options;
        const waitUntilCommand = lifeCycleToSubscribedEvent.get(getWaitUntilSingle(waitUntil));
        await Promise.all([
            setPageContent(this, html),
            waitWithTimeout(new Promise(resolve => {
                this.once(waitUntilCommand, () => {
                    resolve();
                });
            }), waitUntilCommand, timeout),
        ]);
    }
    async content() {
        return await this.evaluate(() => {
            let retVal = '';
            if (document.doctype) {
                retVal = new XMLSerializer().serializeToString(document.doctype);
            }
            if (document.documentElement) {
                retVal += document.documentElement.outerHTML;
            }
            return retVal;
        });
    }
    async sendCDPCommand(method, params = {}) {
        const session = await __classPrivateFieldGet(this, _Context_connection, "f").send('cdp.getSession', {
            context: this.id,
        });
        // TODO: remove any once chromium-bidi types are updated.
        const sessionId = session.result.cdpSession;
        return await __classPrivateFieldGet(this, _Context_connection, "f").send('cdp.sendCommand', {
            cdpMethod: method,
            cdpParams: params,
            cdpSession: sessionId,
        });
    }
}
_Context_connection = new WeakMap(), _Context_url = new WeakMap(), _Context_id = new WeakMap(), _Context_parentId = new WeakMap(), _Context_instances = new WeakSet(), _Context_evaluate = async function _Context_evaluate(returnByValue, pageFunction, ...args) {
    var _a, _b;
    const sourceUrlComment = getSourceUrlComment((_b = (_a = getSourcePuppeteerURLIfAvailable(pageFunction)) === null || _a === void 0 ? void 0 : _a.toString()) !== null && _b !== void 0 ? _b : PuppeteerURL.INTERNAL_URL);
    let responsePromise;
    const resultOwnership = returnByValue ? 'none' : 'root';
    if (isString(pageFunction)) {
        const expression = SOURCE_URL_REGEX.test(pageFunction)
            ? pageFunction
            : `${pageFunction}\n${sourceUrlComment}\n`;
        responsePromise = __classPrivateFieldGet(this, _Context_connection, "f").send('script.evaluate', {
            expression,
            target: { context: __classPrivateFieldGet(this, _Context_id, "f") },
            resultOwnership,
            awaitPromise: true,
        });
    }
    else {
        let functionDeclaration = stringifyFunction(pageFunction);
        functionDeclaration = SOURCE_URL_REGEX.test(functionDeclaration)
            ? functionDeclaration
            : `${functionDeclaration}\n${sourceUrlComment}\n`;
        responsePromise = __classPrivateFieldGet(this, _Context_connection, "f").send('script.callFunction', {
            functionDeclaration,
            arguments: await Promise.all(args.map(arg => {
                return BidiSerializer.serialize(arg, this);
            })),
            target: { context: __classPrivateFieldGet(this, _Context_id, "f") },
            resultOwnership,
            awaitPromise: true,
        });
    }
    const { result } = await responsePromise;
    if ('type' in result && result.type === 'exception') {
        throw createEvaluationError(result.exceptionDetails);
    }
    return returnByValue
        ? BidiSerializer.deserialize(result.result)
        : getBidiHandle(this, result.result);
};
/**
 * @internal
 */
function getWaitUntilSingle(event) {
    if (Array.isArray(event) && event.length > 1) {
        throw new Error('BiDi support only single `waitUntil` argument');
    }
    const waitUntilSingle = Array.isArray(event)
        ? event.find(lifecycle => {
            return lifecycle === 'domcontentloaded' || lifecycle === 'load';
        })
        : event;
    if (waitUntilSingle === 'networkidle0' ||
        waitUntilSingle === 'networkidle2') {
        throw new Error(`BiDi does not support 'waitUntil' ${waitUntilSingle}`);
    }
    assert(waitUntilSingle, `Invalid waitUntil option ${waitUntilSingle}`);
    return waitUntilSingle;
}
/**
 * @internal
 */
export function getBidiHandle(context, result) {
    if (result.type === 'node' || result.type === 'window') {
        return new ElementHandle(context, result);
    }
    return new JSHandle(context, result);
}
//# sourceMappingURL=Context.js.map