/*=========================================================================

     board.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Revision history:

1993 July 14 [Don Cross]
     Fixed a bug in FormatChessMove().  It was using 'dest'
     expecting it to contain a pawn promotion piece in a situation
     after it had assigned the destination offset of the promotion
     to 'dest'.

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1993 October 19 [Don Cross]
     Adding ChessBoard::IsDefiniteDraw()

1993 October 27 [Don Cross]
     Adding ChessBoard::GetPastMove() and
     ChessBoard::GetCurrentPlyNumber().

1993 December 31 [Don Cross]
     Making the following changes to FormatChessMove:
     (1) If non-pawn is being moved, include piece name in source.
     (2) If capture of non-pawn, include piece name in dest.

1994 January 30 [Don Cross]
     Added ChessBoard::ClearEverythingButKings().
     Added ChessBoard::EditCommand(Move).

1994 March 1 [Don Cross]
     Added hash function and code to help find repeated board positions.

1995 December 25 [Don Cross]
     Fixed a bug: ChessBoard needed copy constructor and 
     assignment operator.

1997 January 25 [Don Cross]
     A user reported to me that it really is possible to force a
	 checkmate with B+N+K versus a lone K.  Therefore, I have
	 modified ChessBoard::IsDefiniteDraw to reflect this.

=========================================================================*/

#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "chess.h"

ChessBoard::ChessBoard():
   gameHistory ( new Move [MAX_GAME_HISTORY] ),
   iddCallCount ( 0 ),
   iddHashMatchCount ( 0 )
{
   if ( !gameHistory )
   {
      ChessFatal ( "Out of memory in ChessBoard::ChessBoard()" );
   }

   for ( int i=0; i < MAX_GAME_HISTORY; i++ )
   {
      gameHistory[i].source = 0;
      gameHistory[i].dest   = 0;
   }

   Init();
}


ChessBoard::ChessBoard ( const ChessBoard &other ):
    gameHistory ( new Move [MAX_GAME_HISTORY] )
{
    if ( !gameHistory )
    {
        ChessFatal ( "Out of memory in ChessBoard copy constructor" );
    }

    *this = other;
}


ChessBoard & ChessBoard::operator= ( const ChessBoard &other )
{
    if ( this != &other )
    {
        for ( int i=0; i < MAX_GAME_HISTORY; i++ )
        {
            gameHistory[i] = other.gameHistory[i];
        }

        for ( i=0; i<144; i++ )
        {
            board[i] = other.board[i];
        }

        flags           =   other.flags;
        wmaterial       =   other.wmaterial;
        bmaterial       =   other.bmaterial;
        wk_offset       =   other.wk_offset;
        bk_offset       =   other.bk_offset;
        white_to_move   =   other.white_to_move;

        for ( i=0; i < PIECE_ARRAY_SIZE; i++ )
        {
            inventory[i] = other.inventory[i];
        }

        prev_move       =   other.prev_move;
        ply_number      =   other.ply_number;
        editedFlag      =   other.editedFlag;

        hashCode        =   other.hashCode;
        for ( i=0; i < HASH_COUNT_TABLE_SIZE; i++ )
        {
            hashCount[i] = other.hashCount[i];
        }

        lastCapOrPawn       =   other.lastCapOrPawn;
        iddCallCount        =   other.iddCallCount;
        iddHashMatchCount   =   other.iddHashMatchCount;
    }

    return *this;
}


ChessBoard::~ChessBoard()
{
   if ( gameHistory )
   {
      delete[] gameHistory;
      gameHistory = 0;
   }
}

void ChessBoard::Init()
{
   // Make every square off the board...

   for ( int x=0; x < 144; x++ )
   {
      board[x] = OFFBOARD;
   }

   // Now go back and make the interior squares EMPTY...
   // put the pawns on the board at the same time...

   for ( x=2; x <= 9; x++ )
   {
      board [ OFFSET(x,3) ] = WPAWN;
      board [ OFFSET(x,8) ] = BPAWN;

      for ( int y=4; y <= 7; y++ )
      {
         board [ OFFSET(x,y) ] = EMPTY;
      }
   }

#if WACKY_GAME
   board [ OFFSET(5,5) ] = OFFBOARD;
   board [ OFFSET(5,6) ] = OFFBOARD;
   board [ OFFSET(6,5) ] = OFFBOARD;
   board [ OFFSET(6,6) ] = OFFBOARD;
#endif

   // Put the white pieces on the board...

   board [ OFFSET(2,2) ] = board [ OFFSET(9,2) ] = WROOK;
   board [ OFFSET(3,2) ] = board [ OFFSET(8,2) ] = WKNIGHT;
   board [ OFFSET(4,2) ] = board [ OFFSET(7,2) ] = WBISHOP;
   board [ OFFSET(5,2) ] = WQUEEN;
   board [ wk_offset = OFFSET(6,2) ] = WKING;

   // Put the black pieces on the board...
   board [ OFFSET(2,9) ] = board [ OFFSET(9,9) ] = BROOK;
   board [ OFFSET(3,9) ] = board [ OFFSET(8,9) ] = BKNIGHT;
   board [ OFFSET(4,9) ] = board [ OFFSET(7,9) ] = BBISHOP;
   board [ OFFSET(5,9) ] = BQUEEN;
   board [ bk_offset = OFFSET(6,9) ] = BKING;

   // Now we are done with the board squares.
   // We just need to initialize all the other things in the ChessBoard.

   flags = 0;

   white_to_move = cTRUE;

   prev_move.dest   = SPECIAL_MOVE_NULL;
   prev_move.source = 0;
   prev_move.score  = 0;

   for ( x=0; x < PIECE_ARRAY_SIZE; x++ )
   {
      inventory[x] = 0;
   }

   inventory [WP_INDEX] = inventory [BP_INDEX] = 8;
   inventory [WN_INDEX] = inventory [BN_INDEX] = 2;
   inventory [WB_INDEX] = inventory [BB_INDEX] = 2;
   inventory [WR_INDEX] = inventory [BR_INDEX] = 2;
   inventory [WQ_INDEX] = inventory [BQ_INDEX] = 1;
   inventory [WK_INDEX] = inventory [BK_INDEX] = 1;

   wmaterial = bmaterial = 8*PAWN_VAL + 2*KNIGHT_VAL + 2*BISHOP_VAL +
                           2*ROOK_VAL + QUEEN_VAL + KING_VAL;

   ply_number = 0;

   editedFlag = cFALSE;

   hashCode = 0;
   lastCapOrPawn = -1;
   for ( x=0; x < HASH_COUNT_TABLE_SIZE; x++ )
   {
      hashCount[x] = 0;
   }
}



cBOOLEAN ChessBoard::IsDefiniteDraw ( int maxRepCount )
{
   ++iddCallCount;

   if ( hashCount[hashCode] > maxRepCount &&
        ply_number < MAX_GAME_HISTORY )
   {
      ++iddHashMatchCount;

      INT8 level [64];
      int parity = 0;
      int repCount = 0;

      // Zero out the entire level array...

      ((INT32 *)level) [0] = 0;
      ((INT32 *)level) [1] = 0;
      ((INT32 *)level) [2] = 0;
      ((INT32 *)level) [3] = 0;
      ((INT32 *)level) [4] = 0;
      ((INT32 *)level) [5] = 0;
      ((INT32 *)level) [6] = 0;
      ((INT32 *)level) [7] = 0;
      ((INT32 *)level) [8] = 0;
      ((INT32 *)level) [9] = 0;
      ((INT32 *)level) [10] = 0;
      ((INT32 *)level) [11] = 0;
      ((INT32 *)level) [12] = 0;
      ((INT32 *)level) [13] = 0;
      ((INT32 *)level) [14] = 0;
      ((INT32 *)level) [15] = 0;

      for ( int p = ply_number-1; p > lastCapOrPawn; p-- )
      {
         int ofs1, ofs2;

         if ( p & 1 )
         {
            gameHistory[p].blackOffsets ( ofs1, ofs2 );
         }
         else
         {
            gameHistory[p].whiteOffsets ( ofs1, ofs2 );
         }

         if ( --level[ofs1] == 0 )
         {
            --parity;
         }
         else
         {
            ++parity;
         }

         if ( ++level[ofs2] == 0 )
         {
            --parity;
         }
         else
         {
            ++parity;
         }

         if ( parity == 0 )
         {
            if ( ++repCount >= maxRepCount )
            {
               return cTRUE;
            }
         }
      }
   }

   if ( inventory[WR_INDEX] == 0 &&
        inventory[BR_INDEX] == 0 &&
        inventory[WQ_INDEX] == 0 &&
        inventory[BQ_INDEX] == 0 )
   {
      if ( inventory[WP_INDEX] == 0 &&
           inventory[BP_INDEX] == 0 )
      {
         // Declare it a draw if both sides have knights and/or bishops.
         // If either side has just a king, then see if the other side
         // has enough force for a theoretical win.

         if ( inventory[WN_INDEX] == 0 && inventory[WB_INDEX] == 0 )
         {
            // See if black has enough for a theoretical win.
			if ( inventory[BB_INDEX]>=1 && inventory[BN_INDEX]>=1 )
				return cFALSE;

            return (inventory[BB_INDEX] < 2) ? cTRUE : cFALSE;
         }
         else if ( inventory[BN_INDEX] == 0 && inventory[BB_INDEX] == 0 )
         {
            // See if white has enough for a theoretical win.
			if ( inventory[WB_INDEX]>=1 && inventory[WN_INDEX]>=1 )
				return cFALSE;

            return (inventory[WB_INDEX] < 2) ? cTRUE : cFALSE;
         }
         else
         {
            return cTRUE;
         }
      }
   }

   return cFALSE;
}


cBOOLEAN ChessBoard::WhiteToMove() const
{
   return white_to_move;
}


cBOOLEAN ChessBoard::BlackToMove() const
{
   return !white_to_move;
}


cBOOLEAN ChessBoard::WhiteInCheck() const
{
   return (flags & SF_WCHECK) ? cTRUE : cFALSE;
}


cBOOLEAN ChessBoard::BlackInCheck() const
{
   return (flags & SF_BCHECK) ? cTRUE : cFALSE;
}


SCORE RawPieceValues[] = {PAWN_VAL, KNIGHT_VAL, BISHOP_VAL,
                          ROOK_VAL, QUEEN_VAL,  KING_VAL};


PieceLookup::PieceLookup()
{
   for ( int i=0; i < PIECE_ARRAY_SIZE; i++ )
   {
      sval[i] = 0;
   }

   sval [WP_INDEX] = WPAWN;
   sval [WN_INDEX] = WKNIGHT;
   sval [WB_INDEX] = WBISHOP;
   sval [WR_INDEX] = WROOK;
   sval [WQ_INDEX] = WQUEEN;
   sval [WK_INDEX] = WKING;

   sval [BP_INDEX] = BPAWN;
   sval [BN_INDEX] = BKNIGHT;
   sval [BB_INDEX] = BBISHOP;
   sval [BR_INDEX] = BROOK;
   sval [BQ_INDEX] = BQUEEN;
   sval [BK_INDEX] = BKING;
}


SQUARE ChessBoard::GetSquareContents ( int x, int y ) const
{
   if ( x < 0 || x > 7 || y < 0 || y > 8 )
   {
      return OFFBOARD;
   }
   else
   {
      return board [ OFFSET ( x+2, y+2 ) ];
   }
}


SQUARE ChessBoard::GetSquareContents ( int offset ) const
{
   if ( offset < 0 || offset > 143 )
   {
      return OFFBOARD;
   }
   else
   {
      return board [ offset ];
   }
}


void ChessBoard::SetSquareContents ( SQUARE s, int x, int y )
{
   if ( x >= 0 && x <= 7 && y >= 0 && y <= 7 )
   {
      SetSquareContents ( s, OFFSET(x+2,y+2) );
   }
}


void ChessBoard::SetSquareContents ( SQUARE s, int offset )
{
   if ( offset >= OFFSET(2,2) && offset <= OFFSET(9,9) )
   {
      if ( (board[offset] & OFFBOARD) == 0 )
      {
         if ( board[offset] & (BK_MASK | WK_MASK) )
         {
            return;  // Can't mess with king!  Can only move it elsewhere.
         }

         if ( (s & (WP_MASK | BP_MASK)) &&
              (YPART(offset) == 2 || YPART(offset) == 9) )
         {
            return;  // Can't put pawns on first or last ranks.
         }

         if ( s == WKING && offset != wk_offset )
         {
            board [wk_offset] = EMPTY;
            wk_offset = offset;
            flags |= SF_WKMOVED;
         }
         else if ( s == BKING && offset != bk_offset )
         {
            board [bk_offset] = EMPTY;
            bk_offset = offset;
            flags |= SF_BKMOVED;
         }
         else if ( board [offset] == WROOK && s != WROOK )
         {
            // See if we need to set the rook-moved flag...
            if ( offset == OFFSET(2,2) )
            {
               flags |= SF_WQRMOVED;
            }
            else if ( offset == OFFSET(9,2) )
            {
               flags |= SF_WKRMOVED;
            }
         }
         else if ( board [offset] == BROOK && s != BROOK )
         {
            // See if we need to set the rook-moved flag...
            if ( offset == OFFSET(2,9) )
            {
               flags |= SF_BQRMOVED;
            }
            else if ( offset == OFFSET(9,9) )
            {
               flags |= SF_BKRMOVED;
            }
         }

         board [offset] = s;
         editedFlag = cTRUE;

         Update();
      }
   }
}


void ChessBoard::ClearEverythingButKings()
{
   for ( int x=0; x < 144; x++ )
   {
      SQUARE s = GetSquareContents(x);

      if ( (s & (OFFBOARD | WK_MASK | BK_MASK)) == 0 )
      {
         SetSquareContents ( EMPTY, x );
      }
   }
}


void ChessBoard::EditCommand ( Move edit )
{
   int offset = edit.source;
   if ( offset == 0 )
   {
      // This is a special case...

      ClearEverythingButKings();
   }
   else
   {
      // The offset given is the square to be changed.
      // We need to figure out what to put in it...

      int pindex = edit.dest & 0x0F;
      SQUARE s = ConvertNybbleToSquare ( pindex );
      SetSquareContents ( s, offset );
   }
}


void ChessBoard::Update()
{
   int i;

   for ( i=0; i < PIECE_ARRAY_SIZE; i++ )
   {
      inventory[i] = 0;
   }

   hashCode = 0;
   lastCapOrPawn = -1;
   for ( i=0; i < HASH_COUNT_TABLE_SIZE; i++ )
   {
      hashCount[i] = 0;
   }

   wmaterial = 0;
   bmaterial = 0;

   for ( i=OFFSET(2,2); i <= OFFSET(9,9); i++ )
   {
      SQUARE s = board[i];
      if ( s & (WHITE_MASK | BLACK_MASK) )
      {
         ++inventory [ SPIECE_INDEX(s) ];

         switch ( s )
         {
            case WKING:    wmaterial += KING_VAL;    break;
            case WQUEEN:   wmaterial += QUEEN_VAL;   break;
            case WROOK:    wmaterial += ROOK_VAL;    break;
            case WBISHOP:  wmaterial += BISHOP_VAL;  break;
            case WKNIGHT:  wmaterial += KNIGHT_VAL;  break;
            case WPAWN:    wmaterial += PAWN_VAL;    break;

            case BKING:    bmaterial += KING_VAL;    break;
            case BQUEEN:   bmaterial += QUEEN_VAL;   break;
            case BROOK:    bmaterial += ROOK_VAL;    break;
            case BBISHOP:  bmaterial += BISHOP_VAL;  break;
            case BKNIGHT:  bmaterial += KNIGHT_VAL;  break;
            case BPAWN:    bmaterial += PAWN_VAL;    break;
         }
      }
   }

   if ( IsAttackedByBlack(wk_offset) )
       flags |= SF_WCHECK;
   else
       flags &= ~SF_WCHECK;

   if ( IsAttackedByWhite(bk_offset) )
       flags |= SF_BCHECK;
   else
       flags &= ~SF_BCHECK;
}


char PieceRepresentation ( SQUARE square )
{
   switch ( square )
   {
      case OFFBOARD:     return '*';

      case EMPTY:        return '.';

      case WPAWN:        return 'P';
      case WKNIGHT:      return 'N';
      case WBISHOP:      return 'B';
      case WROOK:        return 'R';
      case WQUEEN:       return 'Q';
      case WKING:        return 'K';

      case BPAWN:        return 'p';
      case BKNIGHT:      return 'n';
      case BBISHOP:      return 'b';
      case BROOK:        return 'r';
      case BQUEEN:       return 'q';
      case BKING:        return 'k';
   }

   return '?';
}


char PromPieceRepresentation ( int prom_piece_index )
{
   switch ( prom_piece_index )
   {
      case N_INDEX:   return 'N';
      case B_INDEX:   return 'B';
      case R_INDEX:   return 'R';
      case Q_INDEX:   return 'Q';
   }

   return '?';
}


static cBOOLEAN OnlyPieceThatCanReach ( SQUARE piece, const SQUARE *dest )
{
   int count = 0;
   const SQUARE *p;

   if ( piece & N_MASK )
   {
      if ( SAME_PIECE ( dest[OFFSET(1,2)], piece ) )    ++count;
      if ( SAME_PIECE ( dest[OFFSET(-1,2)], piece ) )   ++count;
      if ( SAME_PIECE ( dest[OFFSET(1,-2)], piece ) )   ++count;
      if ( SAME_PIECE ( dest[OFFSET(-1,-2)], piece ) )  ++count;
      if ( SAME_PIECE ( dest[OFFSET(2,1)], piece ) )    ++count;
      if ( SAME_PIECE ( dest[OFFSET(-2,1)], piece ) )   ++count;
      if ( SAME_PIECE ( dest[OFFSET(2,-1)], piece ) )   ++count;
      if ( SAME_PIECE ( dest[OFFSET(-2,-1)], piece ) )  ++count;
   }

   if ( piece & (B_MASK | Q_MASK) )
   {
      for ( p = dest + NORTHEAST; *p == EMPTY; p += NORTHEAST );
      if ( SAME_PIECE ( *p, piece ) )  ++count;

      for ( p = dest + NORTHWEST; *p == EMPTY; p += NORTHWEST );
      if ( SAME_PIECE ( *p, piece ) )  ++count;

      for ( p = dest + SOUTHEAST; *p == EMPTY; p += SOUTHEAST );
      if ( SAME_PIECE ( *p, piece ) )  ++count;

      for ( p = dest + SOUTHWEST; *p == EMPTY; p += SOUTHWEST );
      if ( SAME_PIECE ( *p, piece ) )  ++count;
   }

   if ( piece & (R_MASK | Q_MASK) )
   {
      for ( p = dest + NORTH; *p == EMPTY; p += NORTH );
      if ( SAME_PIECE ( *p, piece ) )  ++count;

      for ( p = dest + WEST; *p == EMPTY; p += WEST );
      if ( SAME_PIECE ( *p, piece ) )  ++count;

      for ( p = dest + SOUTH; *p == EMPTY; p += SOUTH );
      if ( SAME_PIECE ( *p, piece ) )  ++count;

      for ( p = dest + EAST; *p == EMPTY; p += EAST );
      if ( SAME_PIECE ( *p, piece ) )  ++count;
   }

   return (count == 1) ? cTRUE : cFALSE;
}


// Must be called before move is made on the given board!!!

void FormatChessMove ( const ChessBoard &board,
                       Move              move,
                       char              movestr [MAX_MOVE_STRLEN + 1] )
{
   int source = move.source & BOARD_OFFSET_MASK;
   int dest = move.dest;
   cBOOLEAN validMove = cTRUE;

   cBOOLEAN white_to_move = board.WhiteToMove();

   int pawn_dir = white_to_move ? NORTH : SOUTH;

   char source_str [16];
   char dest_str [16];

   char *ptr = source_str;

   SQUARE pieceMoving = board.GetSquareContents ( source );
   SQUARE pieceCaptured = EMPTY;

   if ( !(pieceMoving & (WP_MASK | BP_MASK)) )
   {
      *ptr++ = toupper ( PieceRepresentation(pieceMoving) );
   }

   if ( (pieceMoving & (WP_MASK | BP_MASK)) ||
        board.inventory [ SPIECE_INDEX(pieceMoving) ] > 1 )
   {
      *ptr++ = XPART(source) - 2 + 'a';
      *ptr++ = YPART(source) - 2 + '1';
   }

   *ptr = '\0';

   if ( dest > OFFSET(9,9) )
   {
      char prom_piece;

      switch ( dest & 0xf0 )
      {
         case SPECIAL_MOVE_PROMOTE_NORM:
              prom_piece = PromPieceRepresentation ( dest & PIECE_MASK );
              dest = source + pawn_dir;
              dest_str[0] = XPART(dest) - 2 + 'a';
              dest_str[1] = YPART(dest) - 2 + '1';
              dest_str[2] = '\0';
              sprintf ( movestr, "%s-%s/%c", source_str, dest_str, prom_piece );
              break;

         case SPECIAL_MOVE_PROMOTE_CAP_EAST:
              prom_piece = PromPieceRepresentation ( dest & PIECE_MASK );
              dest = source + pawn_dir + EAST;
              pieceCaptured = board.GetSquareContents ( dest );
              dest_str[0] = toupper ( PieceRepresentation ( pieceCaptured ) );
              dest_str[1] = XPART(dest) - 2 + 'a';
              dest_str[2] = YPART(dest) - 2 + '1';
              dest_str[3] = '\0';
              sprintf ( movestr, "%sx%s/%c", source_str, dest_str, prom_piece );
              break;

         case SPECIAL_MOVE_PROMOTE_CAP_WEST:
              prom_piece = PromPieceRepresentation ( dest & PIECE_MASK );
              dest = source + pawn_dir + WEST;
              pieceCaptured = board.GetSquareContents ( dest );
              dest_str[0] = toupper ( PieceRepresentation ( pieceCaptured ) );
              dest_str[1] = XPART(dest) - 2 + 'a';
              dest_str[2] = YPART(dest) - 2 + '1';
              dest_str[3] = '\0';
              sprintf ( movestr, "%sx%s/%c", source_str, dest_str, prom_piece );
              break;

         case SPECIAL_MOVE_KCASTLE:
              strcpy ( movestr, "O-O" );
              break;

         case SPECIAL_MOVE_QCASTLE:
              strcpy ( movestr, "O-O-O" );
              break;

         case SPECIAL_MOVE_EP_EAST:
              dest = source + pawn_dir + EAST;
              dest_str[0] = XPART(dest) - 2 + 'a';
              dest_str[1] = YPART(dest) - 2 + '1';
              dest_str[2] = '\0';
              sprintf ( movestr, "%sx%s e.p.", source_str, dest_str );
              break;

         case SPECIAL_MOVE_EP_WEST:
              dest = source + pawn_dir + WEST;
              dest_str[0] = XPART(dest) - 2 + 'a';
              dest_str[1] = YPART(dest) - 2 + '1';
              dest_str[2] = '\0';
              sprintf ( movestr, "%sx%s e.p.", source_str, dest_str );
              break;

         default:
              validMove = cFALSE;
              strcpy ( movestr, "??-??" );
              break;
      }
   }
   else  // it's a normal move
   {
      pieceCaptured = board.GetSquareContents(dest);
      cBOOLEAN capture = (pieceCaptured != EMPTY);

      ptr = dest_str;

      if ( capture && !(pieceCaptured & (WP_MASK | BP_MASK)) )
      {
         *ptr++ = toupper ( PieceRepresentation ( pieceCaptured ) );
      }

      if ( !capture ||
           (pieceCaptured & (WP_MASK | BP_MASK)) ||
           board.inventory [SPIECE_INDEX(pieceCaptured)] > 1 )
      {
         *ptr++ = XPART(dest) - 2 + 'a';
         *ptr++ = YPART(dest) - 2 + '1';
      }

      *ptr = '\0';

      if ( isupper(source_str[0]) &&
           OnlyPieceThatCanReach ( pieceMoving, board.board + dest ) )
      {
         source_str[1] = '\0';    // don't need source square
      }

      sprintf ( movestr, "%s%c%s",
                source_str,
                (capture ? 'x' : '-'),
                dest_str );
   }

   ChessBoard &temp = (ChessBoard &)board;   // cheating a little here

   if ( validMove )
   {
       cBOOLEAN      enemyInCheck;
       MoveList      enemyMovelist;
       UnmoveInfo    unmove;

       if ( white_to_move )
       {
          temp.MakeWhiteMove ( move, unmove, cTRUE, cTRUE );
          enemyInCheck = temp.BlackInCheck();
          temp.GenBlackMoves ( enemyMovelist );
          temp.UnmakeWhiteMove ( move, unmove );
       }
       else
       {
          temp.MakeBlackMove ( move, unmove, cTRUE, cTRUE );
          enemyInCheck = temp.WhiteInCheck();
          temp.GenWhiteMoves ( enemyMovelist );
          temp.UnmakeBlackMove ( move, unmove );
       }

       if ( enemyInCheck )
       {
          if ( enemyMovelist.num == 0 )
          {
             strcat ( movestr, " mate" );
          }
          else
          {
             strcat ( movestr, "+" );
          }
       }
       else if ( enemyMovelist.num == 0 )
       {
          strcat ( movestr, " stale" );
       }
   }
}


void GetGameListing (
    const ChessBoard &board,
    char *buffer,
    unsigned bufSize )
{
    sprintf ( buffer, "     %-15s   %-15s\r\n", "White", "Black" );
    unsigned len = strlen(buffer);
    char *p = buffer + len;
    ChessBoard temp;
    int numPlies = board.GetCurrentPlyNumber();
    for ( int i=0; i<numPlies && len<bufSize; i++ )
    {
	char moveString [64];
	char s [64];
	int thisLen = 0;
	Move move = board.GetPastMove(i);
	FormatChessMove ( temp, move, moveString );
	if ( temp.WhiteToMove() )
	    thisLen = sprintf ( s, "%3d. %-15s", 1+(i/2), moveString );
	else
	    thisLen = sprintf ( s, "   %s\r\n", moveString );

	if ( len + thisLen + 1 > bufSize )
	    break;

	strcat ( p, s );
	p += thisLen;
	len += thisLen;
	UnmoveInfo unmove;
	temp.MakeMove ( move, unmove );
    }
}



int ChessBoard::GetCurrentPlyNumber() const
{
   return int(ply_number);
}


Move ChessBoard::GetPastMove ( int p ) const
{
   Move move;

   if ( p < 0 || p >= ply_number || p >= MAX_GAME_HISTORY || !gameHistory )
   {
      move.source = move.dest = BYTE(0);
      move.score = SCORE(0);
   }
   else
   {
      move = gameHistory[p];
   }

   return move;
}


void ChessBoard::SaveSpecialMove ( Move special )
{
   if ( ply_number < MAX_GAME_HISTORY )
   {
      gameHistory [ply_number] = special;
   }

   ++ply_number;
}


UINT32 ChessBoard::Hash() const
{
    UINT32 h = 0;

    for ( int y=OFFSET(2,2); y <= OFFSET(2,9); y += NORTH )
    {
        for ( int x=0; x<8; x++ )
        {
            int ofs = x+y;
            UINT32 piece = SPIECE_INDEX(board[ofs]);
            h += (piece+2) << (ofs % (32-5));
        }
    }

    if ( h == 0 )
        h = 0xFFFFFFFF;   // This way we know 0 will never match any hash code

    return h;
}


int ChessBoard::operator== ( const ChessBoard &other ) const
{
    if ( flags != other.flags )
        return 0;

    if ( white_to_move != other.white_to_move )
        return 0;

    for ( int y=OFFSET(2,2); y <= OFFSET(2,9); y += NORTH )
    {
		for ( int x=0; x<8; x++ )
		{
			int ofs = x+y;
			if ( board[ofs] != other.board[ofs] )
					return 0;
		}
    }

    return 1;
}


PieceLookup PieceLookupTable;   // Initialized by constructor before main()


/*--- end of file board.cpp ---*/
