/*===========================================================================

     chess.h  -  Copyright (C) 1993-1996 by Don Cross

     email:  dcross@intersrv.com
     WWW:    http://www.intersrv.com/~dcross/

     Revision history:

1993 April ?? [Don Cross]
     Started writing

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1993 October 22 [Don Cross]
     Moving all operating system specific stuff into separate
     header files.

1994 January 15 [Don Cross]
     Added ComputerChessPlayer::AbortSearch().  I did this to
     facilitate certain features of the OS/2 Presentation Manager
     version of this chess program.

1994 January 22 [Don Cross]
     Going to change the search so that it uses a pointer to
     member function to call the eval function.  The eval function
     will be changed according to the situation.  This will replace
     my older endgame heuristics.

1994 February 2 [Don Cross]
     Added BestPath stuff.

1994 February 10 [Don Cross]
     Starting implementing keeping of piece offsets.
     This should speed eval greatly!

1995 March 26 [Don Cross]
     Added new constructor ChessGame::ChessGame ( cBOOLEAN, cBOOLEAN ).
     This constructor refrains from creating the players, but is told what
     kind of players to expect.

     Added simple learning algorithm.
     The ComputerChessPlayer class remembers every move it makes in
     every game it plays.  The scores assigned to each position is
     based on the time the number of seconds evaluating it was maximized.
     An entire "game tree" is saved to disk in binary form after each
     move generated by a timed search.

1995 July 13 [Don Cross]
     Removed UI dependencies from global functions LoadGame and SaveGame.
     Moved these functions out of UISTDIO into the portable code space
     (game.cpp).
     Added ChessGame::AutoSaveToFile.

1995 December 25 [Don Cross]
     Fixed a bug!  ChessBoard needed a copy constructor and
     an assignment operator.

1996 January 1 [Don Cross]
     Adding the function ParseFancyMove.
     This function recognizes more natural move notation like
     B-c6, PxP, etc.

1996 January 23 [Don Cross]
     Starting to add code to experiment with transposition tables.

1997 January 3 [Don Cross]
     Added ComputerChessPlayer::QueryTimeLimit().

===========================================================================*/
#ifndef __cplusplus
#error Must compile as C++
#endif

#ifndef __DDC_CHESS_32_H
#define __DDC_CHESS_32_H

#pragma pack(1)     // for Microsoft C++

class ChessBoard;
class ChessPlayer;
class ChessGame;
class ChessUI;

// Define the following symbol to be 0 for normal chess.
// Define it to be 1 for a wacky game where the center of the board is missing!
#define WACKY_GAME   0

#define NODES_ARRAY_SIZE    100

// Define portable integer types...

#ifdef __BORLANDC__

      #if sizeof(int) == 4
          typedef unsigned int    UINT32;
          typedef          int    INT32;
      #elif sizeof(long) == 4
          typedef unsigned long   UINT32;
          typedef          long   INT32;
      #else
          #error Error finding a 32-bit integer type!
      #endif

      #if sizeof(int) == 2
          typedef int       INT16;
          typedef unsigned  UINT16;
      #elif sizeof(short) == 2
          typedef short            INT16;
          typedef unsigned short   UINT16;
      #else
          #error Error finding a 16-bit integer type!
      #endif

#else  // __BORLANDC__ is not defined (assume 2-pass compiler)

     // Just fake it...if it's not right, it will be caught at runtime.
     typedef unsigned int     UINT32;
     typedef int              INT32;
     typedef unsigned short   UINT16;
     typedef short            INT16;

#endif  // __BORLANDC__

typedef UINT32          SQUARE;    //holds contents of square on chess board
typedef unsigned char   BYTE;
typedef signed char     INT8;
typedef unsigned char   cBOOLEAN;
typedef INT16           SCORE;     //holds move-ordering and eval scores

#define  cTRUE   1
#define  cFALSE  0

/*-------------------------------------------------------------------------

    This is a map of the bits used in a SQUARE:

                bit7 bit6 bit5 bit4 bit3 bit2 bit1 bit0
    BYTE 0:        0    0   WK   WQ   WR   WB   WN   WP
    BYTE 1:     offb    0   BK   BQ   BR   BB   BN   BP
    BYTE 2:        0    0    0 Bbit Wbit  pc2  pc1  pc0
    BYTE 3:        0 ind6 ind5 ind4 ind3 ind2 ind1 ind0

-------------------------------------------------------------------------*/



//--------------------------- BYTE 0, 1 piece masks...

#define  WP_MASK   (1uL <<  0)
#define  WN_MASK   (1uL <<  1)
#define  WB_MASK   (1uL <<  2)
#define  WR_MASK   (1uL <<  3)
#define  WQ_MASK   (1uL <<  4)
#define  WK_MASK   (1uL <<  5)

#define  BP_MASK   (1uL <<  8)
#define  BN_MASK   (1uL <<  9)
#define  BB_MASK   (1uL << 10)
#define  BR_MASK   (1uL << 11)
#define  BQ_MASK   (1uL << 12)
#define  BK_MASK   (1uL << 13)

#define  P_MASK    (WP_MASK | BP_MASK)
#define  N_MASK    (WN_MASK | BN_MASK)
#define  B_MASK    (WB_MASK | BB_MASK)
#define  R_MASK    (WR_MASK | BR_MASK)
#define  Q_MASK    (WQ_MASK | BQ_MASK)
#define  K_MASK    (WK_MASK | BK_MASK)

#define  OFFBOARD  (1uL << 15)

#define  WHITE_MASK  (WP_MASK|WN_MASK|WB_MASK|WR_MASK|WQ_MASK|WK_MASK)
#define  BLACK_MASK  (BP_MASK|BN_MASK|BB_MASK|BR_MASK|BQ_MASK|BK_MASK)
#define  ALL_MASK    (WHITE_MASK | BLACK_MASK)

#define  SAME_PIECE(square1,square2)   (((square1)&(square2)&(ALL_MASK))!=0)

//--------------------------- BYTE 2 piece indices and side masks
#define  P_INDEX   0
#define  N_INDEX   1
#define  B_INDEX   2
#define  R_INDEX   3
#define  Q_INDEX   4
#define  K_INDEX   5

#define  WHITE_IND    (1 << 3)
#define  BLACK_IND    (1 << 4)

#define  WP_INDEX  (P_INDEX | WHITE_IND)
#define  WN_INDEX  (N_INDEX | WHITE_IND)
#define  WB_INDEX  (B_INDEX | WHITE_IND)
#define  WR_INDEX  (R_INDEX | WHITE_IND)
#define  WQ_INDEX  (Q_INDEX | WHITE_IND)
#define  WK_INDEX  (K_INDEX | WHITE_IND)

#define  BP_INDEX  (P_INDEX | BLACK_IND)
#define  BN_INDEX  (N_INDEX | BLACK_IND)
#define  BB_INDEX  (B_INDEX | BLACK_IND)
#define  BR_INDEX  (R_INDEX | BLACK_IND)
#define  BQ_INDEX  (Q_INDEX | BLACK_IND)
#define  BK_INDEX  (K_INDEX | BLACK_IND)

#define  PIECE_MASK   7
#define  SIDE_MASK    (WHITE_IND | BLACK_IND)
#define  INDEX_MASK   (SIDE_MASK | PIECE_MASK)

#define  PIECE_ARRAY_SIZE     (BLACK_IND + K_INDEX + 1)

#define  SPIECE_INDEX(square)   (((square) >> 16) & INDEX_MASK)
#define  UPIECE_INDEX(square)   (((square) >> 16) & PIECE_MASK)

#define  PROM_PIECE(fakedest,side_index,pawn)  \
   ( PieceLookupTable.sval [ ((fakedest) & PIECE_MASK) | (side_index) ] )

#define  RAW_PIECE_VALUE(square)  \
   RawPieceValues [ UPIECE_INDEX(square) ]

//---------- The following piece definitions are used
// to allow a board edit command to fit inside a Move.

#define EMPTY_NYBBLE  0      // 0000
#define WP_NYBBLE     1      // 0001
#define WN_NYBBLE     2      // 0010
#define WB_NYBBLE     3      // 0011
#define WR_NYBBLE     4      // 0100
#define WQ_NYBBLE     5      // 0101
#define WK_NYBBLE     6      // 0110
#define BP_NYBBLE     7      // 0111
#define BN_NYBBLE     8      // 1000
#define BB_NYBBLE     9      // 1001
#define BR_NYBBLE    10      // 1010
#define BQ_NYBBLE    11      // 1011
#define BK_NYBBLE    12      // 1100

SQUARE ConvertNybbleToSquare ( int nybble );
int    ConvertSquareToNybble ( SQUARE );

//--------------------------- definitions for the 32-bit pieces

#define  WPAWN     (WP_MASK | (SQUARE(WP_INDEX) << 16))
#define  WKNIGHT   (WN_MASK | (SQUARE(WN_INDEX) << 16))
#define  WBISHOP   (WB_MASK | (SQUARE(WB_INDEX) << 16))
#define  WROOK     (WR_MASK | (SQUARE(WR_INDEX) << 16))
#define  WQUEEN    (WQ_MASK | (SQUARE(WQ_INDEX) << 16))
#define  WKING     (WK_MASK | (SQUARE(WK_INDEX) << 16))

#define  BPAWN     (BP_MASK | (SQUARE(BP_INDEX) << 16))
#define  BKNIGHT   (BN_MASK | (SQUARE(BN_INDEX) << 16))
#define  BBISHOP   (BB_MASK | (SQUARE(BB_INDEX) << 16))
#define  BROOK     (BR_MASK | (SQUARE(BR_INDEX) << 16))
#define  BQUEEN    (BQ_MASK | (SQUARE(BQ_INDEX) << 16))
#define  BKING     (BK_MASK | (SQUARE(BK_INDEX) << 16))

#define  EMPTY     SQUARE(0)

//------------------------------------------------------------------

// Raw piece values...

#define   PAWN_VAL      10
#define   KNIGHT_VAL    31
#define   BISHOP_VAL    33
#define   ROOK_VAL      50
#define   QUEEN_VAL     90
#define   KING_VAL      35

// Lookup table for piece values...
extern SCORE  RawPieceValues[];


SCORE MaterialEval ( SCORE wmaterial, SCORE bmaterial );


struct PieceLookup
{
    SQUARE   sval [PIECE_ARRAY_SIZE];

    PieceLookup();
};

extern PieceLookup PieceLookupTable;

//------------------------------------------------------------------


#define  OFFSET(x,y)     ((x) + 12*(y))    // offset in board
#define  XPART(ofs)      ((ofs) % 12)      // x part of offset
#define  YPART(ofs)      ((ofs) / 12)      // y part of offset

#define  NORTH         OFFSET(0,1)
#define  NORTHEAST     OFFSET(1,1)
#define  EAST          OFFSET(1,0)
#define  SOUTHEAST     OFFSET(1,-1)
#define  SOUTH         OFFSET(0,-1)
#define  SOUTHWEST     OFFSET(-1,-1)
#define  WEST          OFFSET(-1,0)
#define  NORTHWEST     OFFSET(-1,1)


#define  WON_FOR_BLACK  (-29000)   // position is winnable for black
#define  WON_FOR_WHITE   (29000)   // position is winnable for white

#define  BLACK_WINS     (-30000)   // black has checkmate
#define  WHITE_WINS      (30000)   // white has checkmate
#define  DRAW                (0)   // (yawn)

#define  WHITE_RESIGN_THRESHOLD   (-1200)
#define  BLACK_RESIGN_THRESHOLD    (1200)

#define  WIN_DELAY_PENALTY   10
#define  WIN_POSTPONEMENT(depth)   (WIN_DELAY_PENALTY*(depth))

#define  MIN_WINDOW     (-31000)   // alpha-beta search window
#define  MAX_WINDOW      (31000)

#define  NEGINF         (-32000)   // negative pseudo-infinity
#define  POSINF          (32000)   // positive pseudo-infinity


/*-------------------------------------------------------------------
     The 'Move' structure contains the source offset in the board,
     the destination offset in the board, and a place to put the
     score of this move (for move sorting, etc).

     *** NOTE 1 ***
     For "special" moves (casting, en passant, and pawn promotion),
     'dest' will contain one of the SPECIAL_MOVE_xxx indicators,
     all of which are invalid offsets (because they are bigger than
     143).  In this case, the source is still valid.

     SPECIAL_MOVE_PROMOTE_xxx use the upper nybble as a
     flag that the move is a pawn promotion.  In this case, the
     lower nybble should be checked for one of the following
     piece indices, to know which piece we are promoting to:

            N_INDEX   to promote to a knight
            B_INDEX   to promote to a bishop
            R_INDEX   to promote to a rook
            Q_INDEX   to promote to a queen

     *** NOTE 2 ***
     Move::source will have its MSB set if the move has already
     been discovered to cause check to the opponent.
     If a Move is used after having called ChessBoard::MakeWhiteMove()
     or ChessBoard::MakeBlackMove() with it, you should always
     mask the source with BOARD_OFFSET_MASK to make sure the
     high order bit is cleared.  To test for check, test against
     the CAUSES_CHECK_BIT.

-------------------------------------------------------------------*/

#define   CAUSES_CHECK_BIT     0x80
#define   BOARD_OFFSET_MASK    0x7F

#define   SPECIAL_MOVE_PROMOTE_NORM          (0x80)
#define   SPECIAL_MOVE_PROMOTE_CAP_EAST      (0x90)
#define   SPECIAL_MOVE_PROMOTE_CAP_WEST      (0xA0)
#define   SPECIAL_MOVE_KCASTLE               (0xB0)
#define   SPECIAL_MOVE_QCASTLE               (0xC0)
#define   SPECIAL_MOVE_EP_EAST               (0xD0)
#define   SPECIAL_MOVE_EP_WEST               (0xE0)
#define   SPECIAL_MOVE_EDIT                  (0xF0)
#define   SPECIAL_MOVE_NULL                  (0x00)

#define   SPECIAL_MOVE_MASK       0xF0


struct Move   // A nice 32-bit structure!
{
    BYTE   source;   // NOTE: High bit of source used as a special flag!
    BYTE   dest;     // Sometimes this isn't a dest...see comments above.
    SCORE  score;    // Used for move-ordering

    cBOOLEAN operator== ( const Move &other ) const
    {
        return ((source ^ other.source) & BOARD_OFFSET_MASK) == 0 &&
                dest == other.dest;
    }

    void Fix ( unsigned long openBookData, ChessBoard & );
    cBOOLEAN Fix ( const ChessBoard &, int Source, int Dest, ChessUI & );

    cBOOLEAN ShouldIgnore() const
    {
        return dest == SPECIAL_MOVE_NULL ||
                (dest & SPECIAL_MOVE_MASK) == SPECIAL_MOVE_EDIT;
    }

    void actualOffsets ( const ChessBoard &, int &ofs1, int &ofs2 ) const;

    int blackHash() const;
    int whiteHash() const;

    void whiteOffsets ( int &ofs1, int &ofs2 ) const;
    void blackOffsets ( int &ofs1, int &ofs2 ) const;
};


/*-----------------------------------------------------------------
      The following flags are defined for
      ChessBoard::flags and related fields in UnmoveInfo.
-----------------------------------------------------------------*/
#define   SF_WKMOVED         0x0001      // has the white king moved?
#define   SF_WKRMOVED        0x0002      // has the white king rook moved?
#define   SF_WQRMOVED        0x0004      // has the white queen rook moved?
#define   SF_WKCASTLED       0x0008      // has white kingside castled?
#define   SF_WQCASTLED       0x0010      // has white queenside castled?
#define   SF_WCHECK          0x0020      // is white in check?

#define   SF_BKMOVED         0x0100      // has the black king moved?
#define   SF_BKRMOVED        0x0200      // has the black king rook moved?
#define   SF_BQRMOVED        0x0400      // has the black queen rook moved?
#define   SF_BKCASTLED       0x0800      // has black kingside castled?
#define   SF_BQCASTLED       0x1000      // has black queenside castled?
#define   SF_BCHECK          0x2000      // is black in check?


struct UnmoveInfo
{
    SQUARE    capture;          //whatever was captured, or EMPTY
    UINT16    flags;
    SCORE     wmaterial;
    SCORE     bmaterial;
    Move      prev_move;        //prev ChessBoard::prev_move (for e.p.)
    INT16     hashCode;
    INT16     lastCapOrPawn;
};


#define  MAX_MOVES  80        // maximum number of moves in a MoveList

struct MoveList
{
    UINT16  num;                   // number of moves in list
    Move    m [MAX_MOVES];         // moves themselves

    int IsLegal ( Move ) const;   // does a simple check for inclusion

    void WhiteSort ();
    void BlackSort ();

    void AddMove ( int source, int dest );
    void AddMove ( Move );

    void SendToFront ( Move );
};


#define MAX_BESTPATH_DEPTH  50

struct BestPath
{
    int    depth;
    Move   m [MAX_BESTPATH_DEPTH];

    BestPath():  depth(0)   {}
};



//------------------------------------------------------------------

enum ChessSide { SIDE_NEITHER, SIDE_WHITE, SIDE_BLACK };


class ChessUI
{
public:
    ChessUI();
    virtual ~ChessUI();

    virtual cBOOLEAN Activate()  {return cTRUE;}

    virtual ChessPlayer *CreatePlayer ( ChessSide ) = 0;

    // NOTE:  The following is called only when there is a
    //        checkmate, stalemate, or draw by attrition.
    //        The function is NOT called if a player resigns.
    virtual void ReportEndOfGame ( ChessSide winner ) = 0;

    virtual void Resign ( ChessSide iGiveUp ) = 0;

    //--------------------------------------------------------------//
    //  The following function should return cTRUE if the Move      //
    //  was read successfully, or cFALSE to abort the game.         //
    //  The move does not have to be legal; if an illegal move      //
    //  is returned, the caller will repeatedly call until a        //
    //  legal move is obtained.                                     //
    //--------------------------------------------------------------//
    virtual cBOOLEAN ReadMove ( ChessBoard &, int &source, int &dest ) = 0;

    virtual SQUARE PromotePawn ( int PawnDest, ChessSide ) = 0;

    //--------------------------------------------------------------
    //  The following function should display the given Move
    //  in the context of the given ChessBoard.  The Move
    //  passed to this function will always be legal for the
    //  given ChessBoard.
    //
    //  The purpose of this function is for a non-human ChessPlayer
    //  to have a chance to animate the chess board.
    //  The ChessGame does not call this function; it is up
    //  to a particular kind of ChessPlayer to do so if needed.
    //--------------------------------------------------------------
    virtual void DisplayMove ( ChessBoard &, Move ) = 0;

    //----------------------------------------------------------------
    //  The following function is called by the ChessGame after
    //  each player moves, so that the move can be displayed
    //  in standard chess notation, if the UI desires.
    //  It is helpful to use the ::FormatChessMove() function
    //  for this purpose.
    //  The parameter 'thinkTime' is how long the player thought
    //  about the move, expressed in hundredths of seconds.
    //
    //  NOTE:  This function is called BEFORE the move
    //  has been made on the given ChessBoard.
    //  This is necessary for ::FormatChessMove() to work.
    //----------------------------------------------------------------
    virtual void RecordMove ( ChessBoard &, Move, INT32 thinkTime ) = 0;

    //----------------------------------------------------------------
    //   The following function is called when a method other than
    //   searching has been used to choose a move, e.g.
    //   opening book or experience database.
    //----------------------------------------------------------------
    virtual void ReportSpecial ( const char *msg )  {}

    virtual void ReportComputerStats (
        INT32   thinkTime,
        UINT32  nodesVisited,
        UINT32  nodesEvaluated,
        UINT32  nodesGenerated,
        int     fwSearchDepth,
        UINT32  vis [NODES_ARRAY_SIZE],
        UINT32  gen [NODES_ARRAY_SIZE] );

    //--------------------------------------------------------------//
    //  The following function should display the given ChessBoard  //
    //  to the user.                                                //
    //  In a GUI, this means updating the board display to the      //
    //  given board's contents.                                     //
    //  In TTY-style UI, this means printing out a new board.       //
    //--------------------------------------------------------------//
    virtual void DrawBoard ( const ChessBoard & ) = 0;

    virtual void DisplayBestMoveSoFar (
        const ChessBoard &,
        Move,
        int level );

    virtual void DisplayCurrentMove (
        const ChessBoard &,
        Move,
        int level );

    virtual void DisplayBestPath (
        const ChessBoard &,
        const BestPath & )  {}

    // The ComputerChessPlayer calls this to announce that it
    // has found a forced mate.

    virtual void PredictMate ( int numMovesFromNow );

    // The following function is a hook for debugging every move in a
    // search.  The board is given before the move is made in it, so
    // that correct move notation can be calculated.

    virtual void DebugPly ( int depth, ChessBoard &, Move ) {}

    // The following function is called whenever a level is exited.
    virtual void DebugExit ( int depth, ChessBoard &, SCORE ) {}
};



//------------------------------------------------------------------

class ChessPlayer
{
public:
    ChessPlayer ( ChessUI & );
    virtual ~ChessPlayer();

    //-------------------------------------------------------------
    // The following function should store the player's move
    // and return cTRUE, or return cFALSE to resign the game.
    // This function will not be called if it is already the
    // end of the game.
    //-------------------------------------------------------------
    virtual cBOOLEAN GetMove ( ChessBoard &, Move &, INT32 &timeSpent ) = 0;

protected:
    ChessUI &userInterface;
};


class HumanChessPlayer: public ChessPlayer
{
public:
    HumanChessPlayer ( ChessUI & );  // needed for parent
    ~HumanChessPlayer();

    cBOOLEAN GetMove ( ChessBoard &, Move &, INT32 &timeSpent );
};


class ComputerChessPlayer: public ChessPlayer
{
public:
    ComputerChessPlayer ( ChessUI & );
    ~ComputerChessPlayer();

    cBOOLEAN GetMove ( ChessBoard &, Move &, INT32 &timeSpent );

    void SetSearchDepth ( int NewSearchDepth );       // cancels time limit
    void SetTimeLimit ( INT32 hundredthsOfSeconds );  // cancels search depth

	INT32 QueryTimeLimit() const { return timeLimit; }

    // The following method is used to abort a search in progress
    // in multi-threaded environments.  There is no proper use of this
    // function in a single-threaded program (at least none that I can
    // think of!)

    void AbortSearch ();

    void SetSearchBias  ( SCORE NewSearchBias );
    void SetOpeningBookEnable ( cBOOLEAN newEnable )
    {
        openingBookSearchEnabled = newEnable;
    }

    void SetTrainingEnable ( cBOOLEAN newTrainingEnable )
    {
        trainingEnabled = newTrainingEnable;
    }

    void ResetHistoryBuffers();

	cBOOLEAN queryResignFlag() const { return allowResignation; }
	void setResignFlag ( cBOOLEAN x ) { allowResignation = x; }

protected:
    typedef SCORE (ComputerChessPlayer::*EvalFunction)
                ( ChessBoard &,
                  int depth,
                  SCORE alphaOrBeta );

    void GetWhiteMove ( ChessBoard &, Move & );
    void GetBlackMove ( ChessBoard &, Move & );

    cBOOLEAN  WhiteOpening ( ChessBoard &, Move & );
    cBOOLEAN  BlackOpening ( ChessBoard &, Move & );

    int WhiteKingFreedom ( ChessBoard & );
    int BlackKingFreedom ( ChessBoard & );

    SCORE WhiteSearchRoot ( ChessBoard &, Move & );
    SCORE BlackSearchRoot ( ChessBoard &, Move & );

    SCORE WhiteSearch (
        ChessBoard &,
        int depth,
        SCORE alpha,
        SCORE beta,
        cBOOLEAN bestPathFlag );

    SCORE BlackSearch (
        ChessBoard &,
        int depth,
        SCORE alpha,
        SCORE beta,
        cBOOLEAN bestPathFlag );

    SCORE WhiteQSearch (
        ChessBoard &,
        int depth,
        SCORE alpha,
        SCORE beta,
        cBOOLEAN bestPathFlag );

    SCORE BlackQSearch (
        ChessBoard &,
        int depth,
        SCORE alpha,
        SCORE beta,
        cBOOLEAN bestPathFlag );

    void GenWhiteCapturesAndChecks ( ChessBoard &board, MoveList &ml );
    void GenBlackCapturesAndChecks ( ChessBoard &board, MoveList &ml );

    SCORE  CommonMidgameEval ( ChessBoard &board );
    SCORE  WhiteMidgameEval ( ChessBoard &board, int depth, SCORE beta );
    SCORE  BlackMidgameEval ( ChessBoard &board, int depth, SCORE alpha );

    // The following eval functions are for endgames in which
    // the ComputerChessPlayer has a mating net advantage against
    // a lone enemy king.

    SCORE  EndgameEval1 ( ChessBoard &board, int depth, SCORE );
    SCORE  ProximityBonus ( ChessBoard &, int ofs, int mask );

    SCORE  WhitePawnBonus ( ChessBoard &, int ofs, int x, int ybase );
    SCORE  BlackPawnBonus ( ChessBoard &, int ofs, int x, int ybase );

    void  WhiteMoveOrdering (
        const ChessBoard &,
        Move &,
        const UnmoveInfo &,
        int depth,
        cBOOLEAN bestPathFlag );

    void  BlackMoveOrdering (
        const ChessBoard &,
        Move &,
        const UnmoveInfo &,
        int depth,
        cBOOLEAN bestPathFlag );

    cBOOLEAN CheckTimeLimit();

    void ChooseEvalFunctions ( ChessBoard &board );

    void FindPrevBestPath ( Move );
    void FoundBestMove ( Move, int depth );
    void HitBottom ( int depth )   { nextBestPath[depth].depth = depth - 1; }
    BestPath *SaveTLMBestPath ( Move );

private:
   MoveList   rootml;              // MoveList at the root of the search
   int        maxlevel, level;
   int        maxCheckDepth;     // used in quiescence search to limit check search
   UINT32     visited, evaluated, generated;
   UINT32     visnodes [NODES_ARRAY_SIZE];
   UINT32     gennodes [NODES_ARRAY_SIZE];
   SCORE      searchBias;    // how much to skew alpha-beta pruning

   cBOOLEAN   timedSearch;    // whether or not to do a timed search
   cBOOLEAN   searchAborted;  // gets set when search is aborted
   INT32      stopTime;       // interval marker to stop thinking at (0.01 sec)
   INT32      timeLimit;      // holds time limit from search to search

   INT32      timeCheckCounter;
   INT32      timeCheckLimit;
   INT32      prevTime;

   EvalFunction   whiteEval;   // eval used for White's turn nodes
   EvalFunction   blackEval;   // eval used for Black's turn nodes

   // Best path stuff...

   int eachBestPathCount;      // number of paths in eachBestPath[]
   BestPath eachBestPath [MAX_MOVES];   // array of BestPath for each Top Level Move
   BestPath currentBestPath;  // BestPath from prev search of current TLM
   BestPath nextBestPath [MAX_BESTPATH_DEPTH];  // BestPath for next search
   UINT32   expectedNextBoardHash;  // hash value of board if opponent makes expected move
   int prevCompletedLevel;   // highest full-width level completed in previous search

   // The following two variables sneak information to move ordering

   int       moveOrder_depth;
   cBOOLEAN  moveOrder_bestPathFlag;

   SCORE *whiteHist;
   SCORE *blackHist;

   cBOOLEAN openingBookSearchEnabled;
   cBOOLEAN trainingEnabled;

   cBOOLEAN allowResignation;

   friend class ChessBoard;    // needed for move ordering
};


//----------------------------------------------------------------------
// The following symbol is the max number of plies that can be stored
// in ChessBoard::gameHistory.  This ought to be enough!
//----------------------------------------------------------------------

#define MAX_GAME_HISTORY   1000
#define MAX_MOVE_STRLEN    20


class ChessBoard
{
public:
    ChessBoard();
    ~ChessBoard();

    ChessBoard ( const ChessBoard & );
    ChessBoard & operator= ( const ChessBoard & );

    int operator== ( const ChessBoard &other ) const;

    void   Init();        // resets the chess board to beginning-of-game
    UINT32 Hash() const;  // calculates 32-bit hash code of board

    void SetSquareContents ( SQUARE, int x, int y );
    void SetSquareContents ( SQUARE, int offset );
    void ClearEverythingButKings();
    void EditCommand ( Move );   // pass a move with a SPECIAL_MOVE_EDIT 'dest'

    cBOOLEAN HasBeenEdited() const {return editedFlag;}

    SQUARE GetSquareContents ( int x, int y ) const;  // (x,y) 0-based on board
    SQUARE GetSquareContents ( int offset ) const;    // standard offset

    int  GenWhiteMoves ( MoveList &, ComputerChessPlayer *player = 0 );
    int  GenBlackMoves ( MoveList &, ComputerChessPlayer *player = 0 );
    int  GenMoves ( MoveList &ml )
    {
        int rc = 0;
        if ( WhiteToMove() )
            rc = GenWhiteMoves ( ml );
        else
            rc = GenBlackMoves ( ml );
        return rc;
    }

    int  GenWhiteCaptures ( MoveList &, ComputerChessPlayer *player = 0 );
    int  GenBlackCaptures ( MoveList &, ComputerChessPlayer *player = 0 );

    int  NumWhiteMoves() const;
    int  NumBlackMoves() const;

    cBOOLEAN  WhiteCanMove();
    cBOOLEAN  BlackCanMove();

    cBOOLEAN  WhiteToMove()  const;
    cBOOLEAN  BlackToMove()  const;

    cBOOLEAN  WhiteInCheck()  const;
    cBOOLEAN  BlackInCheck()  const;

    // Does NOT find stalemate!
    cBOOLEAN  IsDefiniteDraw ( int maxRepeatedPositions );

    void   SaveSpecialMove ( Move );

    void   MakeWhiteMove (
        Move       &move,     // can have side-effects!
        UnmoveInfo &unmove,
        cBOOLEAN    look_for_self_check,
        cBOOLEAN    look_for_enemy_check );

    void   MakeBlackMove (
        Move       &move,     // can have side-effects!
        UnmoveInfo &unmove,
        cBOOLEAN    look_for_self_check,
        cBOOLEAN    look_for_enemy_check );

    void MakeMove ( Move &move, UnmoveInfo &unmove )
    {
       if ( WhiteToMove() )
           MakeWhiteMove ( move, unmove, cTRUE, cTRUE );
       else
           MakeBlackMove ( move, unmove, cTRUE, cTRUE );
    }

    void   UnmakeWhiteMove ( Move move, UnmoveInfo &unmove );
    void   UnmakeBlackMove ( Move move, UnmoveInfo &unmove );

    void UnmakeMove ( Move move, UnmoveInfo &unmove )
    {
       if ( WhiteToMove() )
           UnmakeBlackMove ( move, unmove );
       else
           UnmakeWhiteMove ( move, unmove );
    }

    cBOOLEAN  IsAttackedByWhite ( int offset ) const;
    cBOOLEAN  IsAttackedByBlack ( int offset ) const;

    int    GetCurrentPlyNumber() const;
    Move   GetPastMove ( int plyNumber ) const;
    INT16  GetHashCode() const {return hashCode;}

    const SQUARE *queryBoardPointer() const { return board; }

protected:
   SQUARE      board [144];     // The board's squares

   UINT16      flags;			// See SF_xxx above

   SCORE       wmaterial;       // raw total of white's material
   SCORE       bmaterial;       // raw total of black's material

   UINT16      wk_offset;       // location of white king in the board
   UINT16      bk_offset;       // location of black king in the board

   cBOOLEAN    white_to_move;    // Is it white's turn to move?
   INT16       inventory [PIECE_ARRAY_SIZE];   // how many of each piece
   Move        prev_move;        // Need for en passant
   INT16       ply_number;       // What ply are we on?  0=beginning of game
   Move       *gameHistory;      // array using ply_number as index
   cBOOLEAN    editedFlag;       // has the board been edited?

#define  HASH_COUNT_TABLE_SIZE  128

   INT16  hashCode;
   INT16  hashCount [ HASH_COUNT_TABLE_SIZE ];

   UINT32 iddCallCount;           // number of times IsDefiniteDraw called
   UINT32 iddHashMatchCount;      // number of times board hash matched

   INT16  lastCapOrPawn;   // ply number of last capture or pawn advance

private:

   void  GenMoves_WP ( MoveList &, int source, int ybase );
   void  GenMoves_WN ( MoveList &, int source );
   void  GenMoves_WB ( MoveList &, int source );
   void  GenMoves_WR ( MoveList &, int source );
   void  GenMoves_WQ ( MoveList &, int source );
   void  GenMoves_WK ( MoveList &, int source );

   void  GenMoves_BP ( MoveList &, int source, int ybase );
   void  GenMoves_BN ( MoveList &, int source );
   void  GenMoves_BB ( MoveList &, int source );
   void  GenMoves_BR ( MoveList &, int source );
   void  GenMoves_BQ ( MoveList &, int source );
   void  GenMoves_BK ( MoveList &, int source );

   void  GenCaps_WP ( MoveList &, int source, int ybase );
   void  GenCaps_WN ( MoveList &, int source );
   void  GenCaps_WB ( MoveList &, int source );
   void  GenCaps_WR ( MoveList &, int source );
   void  GenCaps_WQ ( MoveList &, int source );
   void  GenCaps_WK ( MoveList &, int source );

   void  GenCaps_BP ( MoveList &, int source, int ybase );
   void  GenCaps_BN ( MoveList &, int source );
   void  GenCaps_BB ( MoveList &, int source );
   void  GenCaps_BR ( MoveList &, int source );
   void  GenCaps_BQ ( MoveList &, int source );
   void  GenCaps_BK ( MoveList &, int source );

   cBOOLEAN  WP_CanMove ( int ofs, int ybase );
   cBOOLEAN  WN_CanMove ( int ofs );
   cBOOLEAN  WB_CanMove ( int ofs );
   cBOOLEAN  WR_CanMove ( int ofs );
   cBOOLEAN  WQ_CanMove ( int ofs );
   cBOOLEAN  WK_CanMove ( int ofs );

   cBOOLEAN  BP_CanMove ( int ofs, int ybase );
   cBOOLEAN  BN_CanMove ( int ofs );
   cBOOLEAN  BB_CanMove ( int ofs );
   cBOOLEAN  BR_CanMove ( int ofs );
   cBOOLEAN  BQ_CanMove ( int ofs );
   cBOOLEAN  BK_CanMove ( int ofs );

   void  RemoveIllegalWhite (
        MoveList &ml,
        ComputerChessPlayer *myPlayer = 0 );

   void  RemoveIllegalBlack (
        MoveList &ml,
        ComputerChessPlayer *myPlayer = 0 );

   void Update();   // For refreshing internal stuff after editing the board

   friend class ComputerChessPlayer;    // for efficiency reasons!
   friend class ChessUI_dos_cga;
   friend class ChessUI_dos_vga;

   friend void FormatChessMove (
        const ChessBoard &,
        Move,
        char movestr [MAX_MOVE_STRLEN + 1] );
};


class ChessGame
{
public:
    // The following constructor automatically creates players.
    ChessGame ( ChessBoard &, ChessUI & );

    // The following constructor refrains from creating players,
    // but is told what kind of player objects to expect.
    // Each of the cBOOLEAN parameters should be set to cTRUE iff
    // its respective player is an instance of the C++ class HumanPlayer.
    // Otherwise, the value should be cFALSE.
    ChessGame ( 
        ChessBoard &, 
        ChessUI &, 
        cBOOLEAN whiteIsHuman,
        cBOOLEAN blackIsHuman,
        ChessPlayer *_whitePlayer,
        ChessPlayer *_blackPlayer );

    virtual ~ChessGame();

    void Play();

    // The following function tells the ChessGame to save the state
    // of the game in the given file after each move.
    // If filename == NULL, any existing auto-save is canceled.
    // The function returns cFALSE if any errors occur (e.g. out of memory).
    cBOOLEAN AutoSaveToFile ( const char *filename );

    void SetPlayers ( ChessPlayer * _white,  ChessPlayer * _black )
    {
        whitePlayer = _white;
        blackPlayer = _black;
    }

protected:
    ChessBoard   &board;
    ChessPlayer  *whitePlayer;
    ChessPlayer  *blackPlayer;
    ChessUI      &ui;

    char *autoSave_Filename;
};

cBOOLEAN SaveGame ( const ChessBoard &board, const char *filename );
cBOOLEAN LoadGame ( ChessBoard &board, const char *filename );


//-------------------------------------------------------
//   Global functions that must be defined on a
//   per-project basis...
//-------------------------------------------------------
void  ChessFatal ( const char *message );   // displays message and aborts
INT32 ChessTime();   // returns interval marker in hundredths of a second


//-------------------------------------------------------
//   Helpful utility functions that are available...
//-------------------------------------------------------
int ChessRandom ( int n );   // returns random value between 0 and n-1
char PieceRepresentation ( SQUARE square );

// The Distance function returns minimum number of squares that a king
// would have to travel to get from ofs1 to ofs2.
inline int Distance ( int ofs1, int ofs2 )
{
    int xdist = XPART(ofs1) - XPART(ofs2);
    if ( xdist < 0 ) xdist = -xdist;
    int ydist = YPART(ofs1) - YPART(ofs2);
    if ( ydist < 0 ) ydist = -ydist;
    return (xdist > ydist) ? xdist : ydist;
}


void FormatChessMove (
    const ChessBoard &,
    Move,
    char movestr [MAX_MOVE_STRLEN + 1] );


void GetGameListing (
    const ChessBoard &,
    char *listing,
    unsigned maxListingLength );


cBOOLEAN ParseFancyMove (
    const char *fancyMoveNotation,
    ChessBoard &,
    int &source,
    int &dest );

extern int Learn_Output;

#pragma pack()      // For Microsoft C++

#endif  // __DDC_CHESS_32_H


/*--- end of file chess.h ---*/
