/*==========================================================================

     eval.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Contains evaluation heuristics for chess.

     Revision history:

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1993 October 19 [Don Cross]
     Added bonuses for being castled and being able to castle.
     Added small bonus for putting opponent in check.
     Added knowledge of definite draws that aren't stalements.
     Added all kinds of cool positional knowledge about
     bishops, knights, and especially pawns.

1993 October 24 [Don Cross]
     Added MaterialEval() which encourages trades when ahead
     in material, and discourages trades when down in material.

1993 October 25 [Don Cross]
     Added castling security knowledge.

1993 October 31 [Don Cross]
     Added knowledge of knight forks.

1993 December 31 [Don Cross]
     Added knowledge of pawn forks.

1994 January 5 [Don Cross]
     Increased the value of pawn forks.
     Improved blocked-bishop heuristics.
     Improved efficiency of split-pawn heuristics.
     Increased value of split (isolated) pawn.

1994 February 3 [Don Cross]
     Tried to improve knight fork heuristics.

1994 February 6 [Don Cross]
     Changing knight and king position heuristics over to
     lookup tables.

1994 February 10 [Don Cross]
     Changing over to using indexed pieces.

1994 February 14 [Don Cross]
     Made castled position heuristics a lot better!

1994 February 17 [Don Cross]
     Moved all piece positional code into bonus functions.

1994 February 18 [Don Cross]
     Changed eval function to not do positional eval if material
     eval is far enough beyond pruning that position doesn't
     matter anyway.

1995 March 31 [Don Cross]
     Adding bonuses for having pawns attack key center squares.
     This will be most important for opening play.
     Also noticed & fixed a little problem with the asymmetry
     in the KnightPosition array.  (I wasn't correcting for
     White's point of view.)

1995 April 16 [Don Cross]
     Added pawn balance code.
     This code corrects for the strategic benefit
     of being the only player with pawns.
     See ComputerChessPlayer::CommonMidgameEval().

1996 July 21 [Don Cross]
     Refining castling heuristics.  Was ignoring the "KingPosition"
     tables when king had not yet moved.  This resulted in less than
     desired net bonus for castling behavior.

==========================================================================*/

#include "chess.h"

#define SAFE_EVAL_PRUNE_MARGIN  400

//----------------------------------------------------------------------
// All heuristic constants below are positive, whether they are
// bonuses or penalties.  This is done so that, when looking at
// code, it is clear whether they are bonuses or penalties by
// looking at how they are used, without referring to the definition
// of the constant to see if it has a '-' in front of it!
//----------------------------------------------------------------------

// Miscellaneous -------------------------------------------------------
#define  CHECK_BONUS           2
#define  TEMPO_BONUS           1

// Castling and king ---------------------------------------------------

// Opening and Midgame
static const SCORE WKingPosition1 [] =
{
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,

    0000,0000,    15,  18,  20, -25, -25, -27,  20,  15,  0000,0000,
    0000,0000,    -5,  -7, -10, -30, -30, -10,  -7,  -5,  0000,0000,
    0000,0000,   -30, -40, -40, -45, -45, -40, -40, -30,  0000,0000,
    0000,0000,   -50, -55, -60, -65, -65, -60, -55, -50,  0000,0000,
    0000,0000,   -55, -60, -65, -75, -75, -65, -60, -55,  0000,0000,
    0000,0000,   -70, -80, -90,-100,-100, -90, -80, -70,  0000,0000,
    0000,0000,  -250,-240,-230,-220,-220,-230,-240,-250,  0000,0000,
    0000,0000,  -350,-340,-330,-320,-320,-330,-340,-350
};


// Opening and midgame
static const SCORE BKingPosition1 [] =
{
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,

    0000,0000,  -350,-340,-330,-320,-320,-330,-340,-350,  0000,0000,
    0000,0000,  -250,-240,-230,-220,-220,-230,-240,-250,  0000,0000,
    0000,0000,   -70, -80, -90,-100,-100, -90, -80, -70,  0000,0000,
    0000,0000,   -55, -60, -65, -75, -75, -65, -60, -55,  0000,0000,
    0000,0000,   -50, -55, -60, -65, -65, -60, -55, -50,  0000,0000,
    0000,0000,   -30, -40, -40, -45, -45, -40, -40, -30,  0000,0000,
    0000,0000,    -5,  -7, -10, -30, -30, -10,  -7,  -5,  0000,0000,
    0000,0000,    15,  18,  20, -25, -25, -27,  20,  15
};


// Endgame
static const SCORE WKingPosition2 [] =
{
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,

    0000,0000,   -30, -20, -15, -10, -10, -15, -20, -30,  0000,0000,
    0000,0000,   -25, -10,  -5,   0,   0,  -5, -10, -25,  0000,0000,
    0000,0000,   -30,   5,  10,  15,  15,  10,   5, -30,  0000,0000,
    0000,0000,   -20,  10,  20,  20,  20,  20,  10, -20,  0000,0000,
    0000,0000,   -10,  10,  15,  20,  20,  15,  10, -10,  0000,0000,
    0000,0000,   -10,   0,   5,  15,  15,   5,   0, -10,  0000,0000,
    0000,0000,   -20, -12,   0,   0,   0,   0, -12, -20,  0000,0000,
    0000,0000,   -35, -25, -20, -15, -15, -20, -25, -35
};

// Endgame
static const SCORE BKingPosition2 [] =
{
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,

    0000,0000,   -35, -25, -20, -15, -15, -20, -25, -35,  0000,0000,
    0000,0000,   -20, -12,   0,   0,   0,   0, -12, -20,  0000,0000,
    0000,0000,   -10,  10,  15,  20,  20,  15,  10, -10,  0000,0000,
    0000,0000,   -10,   0,   5,  15,  15,   5,   0, -10,  0000,0000,
    0000,0000,   -20,  10,  20,  20,  20,  20,  10, -20,  0000,0000,
    0000,0000,   -30,   5,  10,  15,  15,  10,   5, -30,  0000,0000,
    0000,0000,   -25, -10,  -5,   0,   0,  -5, -10, -25,  0000,0000,
    0000,0000,   -30, -20, -15, -10, -10, -15, -20, -30
};

#define  ROOK_TRAPPED_BY_KING  40
#define  PAWN_PROTECTS_KING1   37
#define  PAWN_PROTECTS_KING2   28
#define  PAWN_PROTECTS_KING3    9
#define  CASTLE_KNIGHT_GUARD   10
#define  CASTLE_HOLE1          15
#define  CASTLE_HOLE2          35
#define  CASTLE_HOLE3          29
#define  CASTLE_HOLE_DANGER    23

#define  KING_OPPOSITION       20    // used only in endgame

#define  CAN_KCASTLE_BONUS    25
#define  CAN_QCASTLE_BONUS    22
#define  CAN_KQCASTLE_BONUS   33
#define  KCASTLE_PATH_EMPTY    4
#define  QCASTLE_PATH_EMPTY    3

// The following 'CTEK' values are bonuses for having pieces
// 'Close To Enemy King'.

#define CTEK_KNIGHT    8      // knight less than 4 away
#define CTEK_QUEEN    18      // queen less than 3 away
#define CTEK_BISHOP    5      // bishop less than 4 away
#define CTEK_ROOK     10      // rook less than 3 away

// Knight --------------------------------------------------------------
#define  NFORK_UNCERTAINTY    50

static const SCORE KnightPosition [] =
{
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,

    0000,0000,    -8,  -6,  -5,  -4,  -4,  -5,  -6,  -8,  0000,0000,
    0000,0000,    -6,   2,   1,   0,   0,   1,   2,  -6,  0000,0000,
    0000,0000,    -4,   3,   6,   8,   8,   6,   3,  -4,  0000,0000,
    0000,0000,    -4,   6,   8,  10,  10,   8,   6,  -4,  0000,0000,
    0000,0000,    -5,   2,   6,   7,   7,   6,   2,  -5,  0000,0000,
    0000,0000,    -7,   1,   5,   3,   3,   5,   1,  -7,  0000,0000,
    0000,0000,    -8,  -3,  -1,  -1,  -1,  -1,  -3,  -8,  0000,0000,
    0000,0000,   -10,  -9,  -8,  -7,  -7,  -8,  -9, -10
};

// Bishop --------------------------------------------------------------
#define  BISHOP_BACK_RANK         11
#define  BISHOP_IMMOBILE          20
#define  CENTER_BLOCK_BISHOP1     20
#define  CENTER_BLOCK_BISHOP2      7
#define  TWO_BISHOP_SYNERGY       10

// Pawn ----------------------------------------------------------------
#define  PAWN_FORK                40
#define  PAWN_SIDE_FILE           20
#define  PAWN_DOUBLED             28
#define  PAWN_SPLIT               32
#define  PAWN_PROTECT1             3
#define  PAWN_PROTECT2             5
#define  BISHOP_PROTECT_PAWN       2
#define  PASSED_PAWN_PROTECT1     40
#define  PASSED_PAWN_PROTECT2     45
#define  PASSED_PAWN_ALONE        30
#define  PASSED_3_FROM_PROM       50
#define  PASSED_2_FROM_PROM       75
#define  PASSED_1_FROM_PROM      150
#define  BLOCKED_2_FROM_PROM      10

// Pawn center attacks.
// NOTE: This table exists in addition to other
// pawn position heuristics.  Therefore, it may
// appear that certain key ideas (e.g. pawns
// close to promotion) have been ignored, but not really.

static const SCORE PawnCenter [] =
{
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,

    0000,0000,     0,   0,   0,   0,   0,   0,   0,   0,  0000,0000,
    0000,0000,     0,   0,   0,   0,   0,   0,   0,   0,  0000,0000,
    0000,0000,     0,   0,   0,   0,   0,   0,   0,   0,  0000,0000,
    0000,0000,     0,   0,   1,   3,   3,   1,   0,   0,  0000,0000,
    0000,0000,     0,   0,   4,   7,   7,   4,   0,   0,  0000,0000,
    0000,0000,     1,   0,  -1,   1,   1,  -1,   0,   1,  0000,0000,
    0000,0000,     0,   0,   1,  -1,  -1,   1,   0,   0,  0000,0000,
    0000,0000,     0,   0,   0,   0,   0,   0,   0,   0
};



#define PAWN_BALANCE 1
#if PAWN_BALANCE

    // The following table gives pawn imbalance scores.
    // This must be scaled relative to the enemy pieces which
    // confront the pawns.

    static const SCORE PawnBalance [9][9] =
    {
        //   0     1     2     3     4     5     6     7     8

        {    0,  120,  140,  150,  157,  162,  167,  170,  172},    // 0
        {    0,    0,   55,   70,   80,   90,   97,  105,  110},    // 1
        {    0,    0,    0,   10,   20,   30,   40,   50,   60},    // 2
        {    0,    0,    0,    0,    6,   15,   20,   25,   30},    // 3
        {    0,    0,    0,    0,    0,    4,   10,   12,   15},    // 4
        {    0,    0,    0,    0,    0,    0,    3,    5,   10},    // 5
        {    0,    0,    0,    0,    0,    0,    0,    2,    3},    // 6
        {    0,    0,    0,    0,    0,    0,    0,    0,    1},    // 7
        {    0,    0,    0,    0,    0,    0,    0,    0,    0},    // 8
    };

#endif // PAWN_BALANCE

// Rook ----------------------------------------------------------------
#define  ROOK_OPEN_FILE             6
#define  ROOK_CAN_REACH_7TH_RANK    7
#define  ROOK_ON_7TH_RANK          12
#define  ROOK_CONNECT_VERT          4
#define  ROOK_CONNECT_HOR           2
#define  ROOK_IMMOBILE_HORIZ        5
#define  ROOK_IMMOBILE             15


//----------------------------------------------------------------------


static SCORE WhiteRookBonus ( const SQUARE *b, int ofs, int wk_offset )
{
   SCORE bonus = 0;

   // First, see if we are on the seventh rank or the eighth rank...

   if ( ofs >= OFFSET(2,8) )
   {
      if ( ofs <= OFFSET(9,8) )
         bonus += ROOK_ON_7TH_RANK;
      else
         bonus += (ROOK_CAN_REACH_7TH_RANK + ROOK_OPEN_FILE);
   }
   else
   {
      // A file is open if none of our pawns is blocking it before 7th.
      for ( int z=ofs + NORTH; b[z] == EMPTY && z < OFFSET(2,8); z += NORTH );

      bonus += (YPART(z) - 2) / 2;

      if ( z < OFFSET(2,8) )
      {
         if ( !(b[z] & WP_MASK) )
            bonus += ROOK_OPEN_FILE;
      }
      else if ( z > OFFSET(9,8) )
      {
         bonus += ROOK_CAN_REACH_7TH_RANK;
      }

      for ( z=ofs + SOUTH; b[z] == EMPTY; z += SOUTH );
      if ( b[z] & (WR_MASK | WQ_MASK) )
         bonus += ROOK_CONNECT_VERT;

      for ( z=ofs + WEST; b[z] == EMPTY; z += WEST );
      if ( b[z] & WR_MASK )
         bonus += ROOK_CONNECT_HOR;
   }

   if ( (b[ofs+EAST] & (WHITE_MASK | OFFBOARD)) &&
        (b[ofs+WEST] & (WHITE_MASK | OFFBOARD)) )
   {
      if ( (b[ofs+NORTH] & (WHITE_MASK | OFFBOARD)) &&
           (b[ofs+SOUTH] & (WHITE_MASK | OFFBOARD)) )
         bonus -= ROOK_IMMOBILE;
      else
         bonus -= ROOK_IMMOBILE_HORIZ;
   }

   if ( Distance ( ofs, wk_offset ) < 3 )
      bonus += CTEK_ROOK;

   return bonus;
}


static SCORE BlackRookBonus ( const SQUARE *b, int ofs, int wk_offset )
{
   SCORE bonus = 0;

   // First, see if we are on the seventh rank or the eighth rank...

   if ( ofs <= OFFSET(9,3) )
   {
      if ( ofs >= OFFSET(2,3) )
         bonus += ROOK_ON_7TH_RANK;
      else
         bonus += (ROOK_CAN_REACH_7TH_RANK + ROOK_OPEN_FILE);
   }
   else
   {
      // A file is open if none of our pawns is blocking it before 7th.
      for ( int z=ofs + SOUTH; b[z] == EMPTY && z > OFFSET(9,3); z += SOUTH );

      bonus += (9 - YPART(z)) / 2;

      if ( z > OFFSET(9,3) )
      {
         if ( !(b[z] & BP_MASK) )
            bonus += ROOK_OPEN_FILE;
      }
      else if ( z < OFFSET(3,3) )
      {
         bonus += ROOK_CAN_REACH_7TH_RANK;
      }

      for ( z=ofs + NORTH; b[z] == EMPTY; z += NORTH );
      if ( b[z] & (BR_MASK | BQ_MASK) )
         bonus += ROOK_CONNECT_VERT;

      for ( z=ofs + WEST; b[z] == EMPTY; z += WEST );
      if ( b[z] & BR_MASK )
         bonus += ROOK_CONNECT_HOR;
   }

   if ( (b[ofs+EAST] & (BLACK_MASK | OFFBOARD)) &&
        (b[ofs+WEST] & (BLACK_MASK | OFFBOARD)) )
   {
      if ( (b[ofs+NORTH] & (BLACK_MASK | OFFBOARD)) &&
           (b[ofs+SOUTH] & (BLACK_MASK | OFFBOARD)) )
         bonus -= ROOK_IMMOBILE;
      else
         bonus -= ROOK_IMMOBILE_HORIZ;
   }

   if ( Distance ( ofs, wk_offset ) < 3 )
      bonus += CTEK_ROOK;

   return bonus;
}


static SCORE WhiteBishopBonus ( const SQUARE *b,
                                int ofs,
                                int ybase,
                                int bk_offset )
{
   const SQUARE *insideBoard = b + ofs;

   SCORE score;

   if ( (insideBoard[NORTHEAST] & (WHITE_MASK|OFFBOARD)) &&
        (insideBoard[NORTHWEST] & (WHITE_MASK|OFFBOARD)) &&
        (insideBoard[SOUTHEAST] & (WHITE_MASK|OFFBOARD)) &&
        (insideBoard[SOUTHWEST] & (WHITE_MASK|OFFBOARD)) )
   {
      score = -BISHOP_IMMOBILE;
   }
   else
   {
      int count = 0;
      const SQUARE *p;

      for ( p = insideBoard + NORTHEAST;
            *p == EMPTY;
            p += NORTHEAST )
      {
         if ( !(p[NORTHEAST] & BP_MASK) && !(p[NORTHWEST] & BP_MASK) )
            ++count;
      }

      for ( p = insideBoard + NORTHWEST;
            *p == EMPTY;
            p += NORTHWEST )
      {
         if ( !(p[NORTHEAST] & BP_MASK) && !(p[NORTHWEST] & BP_MASK) )
            ++count;
      }

      for ( p = insideBoard + SOUTHEAST;
            *p == EMPTY;
            p += SOUTHEAST )
      {
         if ( !(p[NORTHEAST] & BP_MASK) && !(p[NORTHWEST] & BP_MASK) )
            ++count;
      }

      for ( p = insideBoard + SOUTHWEST;
            *p == EMPTY;
            p += SOUTHWEST )
      {
         if ( !(p[NORTHEAST] & BP_MASK) && !(p[NORTHWEST] & BP_MASK) )
            ++count;
      }

      score = count;
   }

   if ( ybase == OFFSET(2,2) )
      score -= BISHOP_BACK_RANK;

   if ( Distance ( ofs, bk_offset ) < 4 )
      score += CTEK_BISHOP;

   // This is a kludge to try to prevent computer from putting
   // pieces in front of QP or KP and blocking the other bishop
   // from development through the center.

   if ( ofs == OFFSET(7,2) )
   {
      if ( b [ OFFSET(6,4) ] != EMPTY && (b [ OFFSET(6,3) ] & WP_MASK) )
         score -= CENTER_BLOCK_BISHOP1;

      if ( b [ OFFSET(5,4) ] != EMPTY )
         score -= CENTER_BLOCK_BISHOP2;
   }
   else if ( ofs == OFFSET(4,2) )
   {
      if ( b [ OFFSET(5,4) ] != EMPTY && (b [ OFFSET(5,3) ] & WP_MASK) )
         score -= CENTER_BLOCK_BISHOP1;

      if ( b [ OFFSET(6,4) ] != EMPTY )
         score -= CENTER_BLOCK_BISHOP2;
   }

   return score;
}


static SCORE BlackBishopBonus ( const SQUARE *b,
                                int ofs,
                                int ybase,
                                int bk_offset )
{
   const SQUARE *insideBoard = b + ofs;

   SCORE score;

   if ( (insideBoard[NORTHEAST] & (BLACK_MASK|OFFBOARD)) &&
        (insideBoard[NORTHWEST] & (BLACK_MASK|OFFBOARD)) &&
        (insideBoard[SOUTHEAST] & (BLACK_MASK|OFFBOARD)) &&
        (insideBoard[SOUTHWEST] & (BLACK_MASK|OFFBOARD)) )
   {
      score = -BISHOP_IMMOBILE;
   }
   else
   {
      int count = 0;
      const SQUARE *p;

      for ( p = insideBoard + NORTHEAST;
            *p == EMPTY;
            p += NORTHEAST )
      {
         if ( !(p[SOUTHEAST] & WP_MASK) && !(p[SOUTHWEST] & WP_MASK) )
            ++count;
      }

      for ( p = insideBoard + NORTHWEST;
            *p == EMPTY;
            p += NORTHWEST )
      {
         if ( !(p[SOUTHEAST] & WP_MASK) && !(p[SOUTHWEST] & WP_MASK) )
            ++count;
      }

      for ( p = insideBoard + SOUTHEAST;
            *p == EMPTY;
            p += SOUTHEAST )
      {
         if ( !(p[SOUTHEAST] & WP_MASK) && !(p[SOUTHWEST] & WP_MASK) )
            ++count;
      }

      for ( p = insideBoard + SOUTHWEST;
            *p == EMPTY;
            p += SOUTHWEST )
      {
         if ( !(p[SOUTHEAST] & WP_MASK) && !(p[SOUTHWEST] & WP_MASK) )
            ++count;
      }

      score = count;
   }

   if ( ybase == OFFSET(2,2) )
      score -= BISHOP_BACK_RANK;

   if ( Distance ( ofs, bk_offset ) < 4 )
      score += CTEK_BISHOP;

   if ( ofs == OFFSET(7,9) )
   {
      if ( b [ OFFSET(6,7) ] != EMPTY && (b [ OFFSET(6,8) ] & BP_MASK) )
         score -= CENTER_BLOCK_BISHOP1;

      if ( b [ OFFSET(5,7) ] != EMPTY )
         score -= CENTER_BLOCK_BISHOP2;
   }
   else if ( ofs == OFFSET(4,9) )
   {
      if ( b [ OFFSET(5,7) ] != EMPTY && (b [ OFFSET(5,8) ] & BP_MASK) )
         score -= CENTER_BLOCK_BISHOP1;

      if ( b [ OFFSET(6,7) ] != EMPTY )
         score -= CENTER_BLOCK_BISHOP2;
   }

   return score;
}


static SCORE WhiteKnightFork ( const SQUARE *p )
{
   int count = 0;
   int rcount = 0;
   int qcount = 0;
   int kcount = 0;
   SQUARE s;

   if ( (s = p[OFFSET(1,2)]) & (BR_MASK | BQ_MASK | BK_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-1,2)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK)        ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(1,-2)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-1,-2)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(2,1)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(2,-1)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-2,1)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-2,-1)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( count > 1 )
   {
      // Something looks forked

      if ( kcount > 0 )
      {
         if ( qcount > 0 )
            return QUEEN_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
         else
            return ROOK_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
      }
      else if ( qcount > 1 )
      {
         return QUEEN_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
      }
      else
      {
         return ROOK_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
      }
   }

   return 0;
}


static SCORE BlackKnightFork ( const SQUARE *p )
{
   int count = 0;
   int rcount = 0;
   int qcount = 0;
   int kcount = 0;
   SQUARE s;

   if ( (s = p[OFFSET(1,2)]) & (WR_MASK | WQ_MASK | WK_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-1,2)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(1,-2)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-1,-2)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(2,1)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(2,-1)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-2,1)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-2,-1)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( count > 1 )
   {
      // Something looks forked

      if ( kcount > 0 )
      {
         if ( qcount > 0 )
            return QUEEN_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
         else
            return ROOK_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
      }
      else if ( qcount > 1 )
      {
         return QUEEN_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
      }
      else
      {
         return ROOK_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
      }
   }

   return 0;
}


static SCORE WhiteKnightBonus ( const SQUARE *inBoard,
                                int ofs,
                                int bk_offset )
{
   // The offset is subtracted from OFFSET(11,11) as
   // a trick for "rotating" the array, since the contents
   // are asymmetric.

   SCORE score = KnightPosition [ OFFSET(11,11) - ofs ];

   if ( Distance ( ofs, bk_offset ) < 4 )
      score += CTEK_KNIGHT;

   score += WhiteKnightFork ( inBoard );

   return score;
}


static SCORE BlackKnightBonus ( const SQUARE *inBoard,
                                int ofs,
                                int wk_offset )
{
   SCORE score = KnightPosition [ ofs ];

   if ( Distance ( ofs, wk_offset ) < 4 )
      score += CTEK_KNIGHT;

   score += BlackKnightFork ( inBoard );

   return score;
}


static SCORE WhiteQueenBonus ( const SQUARE *,
                               int ofs,
                               int bk_offset )
{
   SCORE score = 0;

   if ( Distance ( ofs, bk_offset ) < 3 )
      score += CTEK_QUEEN;

   return score;
}


static SCORE BlackQueenBonus ( const SQUARE *,
                               int ofs,
                               int wk_offset )
{
   SCORE score = 0;

   if ( Distance ( ofs, wk_offset ) < 3 )
      score += CTEK_QUEEN;

   return score;
}



static SCORE CastleHoleDanger ( const SQUARE *p,
                                int dir,
                                SQUARE mask )
{
   SCORE s = 0;

   for(;;)
   {
      if ( *p & mask )
         s += CASTLE_HOLE_DANGER;
      else if ( *p != EMPTY )
         break;

      p += dir;
   }

   return s;
}


// Put stuff in CommonMidgameEval() which does not depend on whose turn it is.

SCORE ComputerChessPlayer::CommonMidgameEval ( ChessBoard &board )
{
   SCORE score = 0;
   const SQUARE *b = board.board;
   const int wk = board.wk_offset;
   const int bk = board.bk_offset;

   //---------------------------- WHITE KING ----------------------------

   // White king is timid if there is a black queen or a pair of black rooks with
   // assistance from either black knight(s) or black bishop(s).
   cBOOLEAN timidWKing = 
	   ( (board.inventory[BQ_INDEX] > 0 || board.inventory[BR_INDEX] > 1) &&
           (board.inventory[BB_INDEX] > 0 || board.inventory[BN_INDEX] > 0) );

   if ( timidWKing )
       score += WKingPosition1 [wk];    // cautious position table
   else
       score += WKingPosition2 [wk];    // aggressive position table
    
   // Check white castling...

   if ( board.white_flags & SF_KMOVED )
   {
      if ( timidWKing )
      {
         // Play it safe with king...

         if ( wk == OFFSET(8,2) )
         {
            if ( b[OFFSET(9,2)] & WR_MASK )
               score -= ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(9,3)] & WR_MASK )
	       score -= ROOK_TRAPPED_BY_KING;

            if ( b[OFFSET(7,4)] & WN_MASK )
               score += CASTLE_KNIGHT_GUARD;
         }
         else if ( wk == OFFSET(7,2) )
         {
            if ( b[OFFSET(8,2)] & WR_MASK )
               score -= ROOK_TRAPPED_BY_KING;

            if ( b[OFFSET(9,2)] & WR_MASK )
               score -= ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(8,3)] & WR_MASK )
	       score -= ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(9,3)] & WR_MASK )
	       score -= ROOK_TRAPPED_BY_KING;

            if ( b[OFFSET(7,4)] & WN_MASK )
               score += CASTLE_KNIGHT_GUARD;
         }
         else if ( wk == OFFSET(4,2) )
         {
            if ( b[OFFSET(3,2)] & WR_MASK )
               score -= ROOK_TRAPPED_BY_KING;

            if ( b[OFFSET(2,2)] & WR_MASK )
               score -= ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(2,3)] & WR_MASK )
	       score -= ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(3,3)] & WR_MASK )
	       score -= ROOK_TRAPPED_BY_KING;

            if ( b[OFFSET(4,4)] & WN_MASK )
               score += CASTLE_KNIGHT_GUARD;
         }
         else if ( wk == OFFSET(3,2) )
         {
            if ( b[OFFSET(2,2)] & WR_MASK )
               score -= ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(3,3)] & WR_MASK )
	       score -= ROOK_TRAPPED_BY_KING;

            if ( b[OFFSET(4,4)] & WN_MASK )
               score += CASTLE_KNIGHT_GUARD;
         }
         else if ( wk == OFFSET(2,2) )
         {
            if ( b[OFFSET(4,4)] & WN_MASK )
               score += CASTLE_KNIGHT_GUARD;
         }
         else if ( wk == OFFSET(9,2) )
         {
            if ( b[OFFSET(7,4)] & WN_MASK )
               score += CASTLE_KNIGHT_GUARD;
         }

         if ( wk != OFFSET(6,2) && wk != OFFSET(5,2) )
         {
            // Try to keep good pawn formation in front of king

            if ( b[wk + OFFSET(-1,1)] & WP_MASK )
               score += PAWN_PROTECTS_KING2;
            else if ( wk < OFFSET(2,6) &&
                      !(b[wk + OFFSET(-1,2)] & (WP_MASK | OFFBOARD)) &&
                      !(b[wk + OFFSET(-1,3)] & (WP_MASK | OFFBOARD)) )
            {
               score -= CASTLE_HOLE1;
               score -= CastleHoleDanger ( &b[wk+OFFSET(-1,0)],
                                           NORTH,
                                           (BR_MASK | BQ_MASK) );
            }

            if ( b[wk + OFFSET(0,1)] & WP_MASK )
               score += PAWN_PROTECTS_KING1;
            else if ( wk < OFFSET(2,6) &&
                      !(b[wk + OFFSET(0,2)] & WP_MASK) &&
                      !(b[wk + OFFSET(0,3)] & WP_MASK) )
            {
               score -= CASTLE_HOLE2;
               score -= CastleHoleDanger ( &b[wk],
                                           NORTH,
                                           (BR_MASK | BQ_MASK) );
            }

            if ( b[wk + OFFSET(1,1)] & WP_MASK )
               score += PAWN_PROTECTS_KING2;
            else if ( wk < OFFSET(2,6) &&
                      !(b[wk + OFFSET(0,2)] & WP_MASK) &&
                      !(b[wk + OFFSET(0,3)] & WP_MASK) )
            {
               score -= CASTLE_HOLE3;
               score -= CastleHoleDanger ( &b[wk+OFFSET(1,0)],
                                           NORTH,
                                           (BR_MASK | BQ_MASK) );
            }

            if ( b[wk + OFFSET(-1,2)] & WP_MASK )
               score += PAWN_PROTECTS_KING3;

            if ( b[wk + OFFSET(0,2)] & WP_MASK )
               score += PAWN_PROTECTS_KING2;

            if ( b[wk + OFFSET(1,2)] & WP_MASK )
               score += PAWN_PROTECTS_KING3;
         }
      }
      else   // Be aggressive with white king...
      {
         if ( board.white_to_move )
         {
            // Look for opposition

            int kk = wk - bk;

            if ( kk==OFFSET(2,0)  || kk==OFFSET(0,2) ||
                 kk==OFFSET(-2,0) || kk==OFFSET(0,-2) )
               score += KING_OPPOSITION;
         }
      }
   }
   else
   {
      // The white king has not yet moved
      if ( (board.white_flags & SF_KRMOVED) == 0 )
      {
         if ( b[OFFSET(7,2)] == EMPTY )
            score += KCASTLE_PATH_EMPTY;

         if ( b[OFFSET(8,2)] == EMPTY )
            score += KCASTLE_PATH_EMPTY;

         if ( (board.white_flags & SF_QRMOVED) == 0 )
         {
            // Might castle either way
            score += CAN_KQCASTLE_BONUS;

            if ( b[OFFSET(3,2)] == EMPTY )
               score += QCASTLE_PATH_EMPTY;

            if ( b[OFFSET(4,2)] == EMPTY )
               score += QCASTLE_PATH_EMPTY;

            if ( b[OFFSET(5,2)] == EMPTY )
               score += QCASTLE_PATH_EMPTY;
         }
         else
         {
            // Might castle kingside only
            score += CAN_KCASTLE_BONUS;
         }
      }
      else if ( (board.white_flags & SF_QRMOVED) == 0 )
      {
         // Might castle queenside only
         score += CAN_QCASTLE_BONUS;

         if ( b[OFFSET(3,2)] == EMPTY )
            score += QCASTLE_PATH_EMPTY;

         if ( b[OFFSET(4,2)] == EMPTY )
            score += QCASTLE_PATH_EMPTY;

         if ( b[OFFSET(5,2)] == EMPTY )
            score += QCASTLE_PATH_EMPTY;
      }
   }

   //---------------------------- BLACK KING ----------------------------

   // Black king is timid if there is a white queen or a pair of white rooks with
   // assistance from either white knight(s) or white bishop(s).
   cBOOLEAN timidBKing = 
	   ( (board.inventory[WQ_INDEX] > 0 || board.inventory[WR_INDEX] > 1) &&
           (board.inventory[WB_INDEX] > 0 || board.inventory[WN_INDEX] > 0) );

   if ( timidBKing )
       score -= BKingPosition1 [bk];  // cautious position table
   else
       score -= BKingPosition2 [bk];  // aggressive position table

   // Check black castling...

   if ( board.black_flags & SF_KMOVED )
   {
      if ( timidBKing )      
      {
         // Play it safe with king...

         if ( bk == OFFSET(8,9) )
         {
            if ( b[OFFSET(9,9)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(9,8)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;
         }
         else if ( bk == OFFSET(7,9) )
         {
            if ( b[OFFSET(8,9)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(8,8)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(9,8)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;

            if ( b[OFFSET(9,9)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;
         }
         else if ( bk == OFFSET(9,9) )
         {
            if ( b[OFFSET(7,7)] & BN_MASK )
               score -= CASTLE_KNIGHT_GUARD;
         }
         else if ( bk == OFFSET(4,9) )
         {
            if ( b[OFFSET(3,9)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(3,8)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;

            if ( b[OFFSET(2,9)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(2,8)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;

            if ( b[OFFSET(4,7)] & BN_MASK )
               score -= CASTLE_KNIGHT_GUARD;
         }
         else if ( bk == OFFSET(3,9) )
         {
            if ( b[OFFSET(2,9)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;

	    if ( b[OFFSET(2,8)] & BR_MASK )
               score += ROOK_TRAPPED_BY_KING;

            if ( b[OFFSET(4,7)] & BN_MASK )
               score -= CASTLE_KNIGHT_GUARD;
         }
         else if ( bk == OFFSET(2,9) )
         {
            if ( b[OFFSET(4,7)] & BN_MASK )
               score -= CASTLE_KNIGHT_GUARD;
         }

         if ( bk != OFFSET(6,9) && bk != OFFSET(5,9) )
         {
            // Try to keep good pawn formation in front of king

            if ( b[bk + OFFSET(-1,-1)] & BP_MASK )
               score -= PAWN_PROTECTS_KING2;
            else if ( bk > OFFSET(9,5) &&
                      !(b[bk + OFFSET(-1,-2)] & (BP_MASK | OFFBOARD) ) &&
                      !(b[bk + OFFSET(-1,-3)] & (BP_MASK | OFFBOARD) ) )
            {
               score += CASTLE_HOLE1;
               score += CastleHoleDanger ( &b[bk+OFFSET(-1,0)],
                                           SOUTH,
                                           (WR_MASK | WQ_MASK) );
            }

            if ( b[bk + OFFSET(0,-1)] & BP_MASK )
               score -= PAWN_PROTECTS_KING1;
            else if ( bk > OFFSET(9,5) &&
                      !(b[bk + OFFSET(0,-2)] & BP_MASK) &&
                      !(b[bk + OFFSET(0,-3)] & BP_MASK) )
            {
               score += CASTLE_HOLE2;
               score += CastleHoleDanger ( &b[bk],
                                           SOUTH,
                                           (WR_MASK | WQ_MASK) );
            }

            if ( b[bk + OFFSET(1,-1)] & BP_MASK )
               score -= PAWN_PROTECTS_KING2;
            else if ( bk > OFFSET(9,5) &&
                      !(b[bk + OFFSET(1,-2)] & BP_MASK) &&
                      !(b[bk + OFFSET(1,-3)] & BP_MASK) )
            {
               score += CASTLE_HOLE3;
               score += CastleHoleDanger ( &b[bk + OFFSET(1,0)],
                                           SOUTH,
                                           (WR_MASK | WQ_MASK) );
            }

            if ( b[bk + OFFSET(-1,-2)] & BP_MASK )
               score -= PAWN_PROTECTS_KING3;

            if ( b[bk + OFFSET(0,-2)] & BP_MASK )
               score -= PAWN_PROTECTS_KING2;

            if ( b[bk + OFFSET(1,-2)] & BP_MASK )
               score -= PAWN_PROTECTS_KING3;
         }
      }
      else
      {
         // Be aggressive with king...

         if ( !board.white_to_move )
         {
            // Look for opposition

            int kk = wk - bk;

            if ( kk==OFFSET(2,0)  || kk==OFFSET(0,2) ||
                 kk==OFFSET(-2,0) || kk==OFFSET(0,-2) )
               score -= KING_OPPOSITION;
         }
      }
   }
   else
   {
      // The black king has not yet moved

      if ( (board.black_flags & SF_KRMOVED) == 0 )
      {
         if ( b[OFFSET(7,9)] == EMPTY )
            score -= KCASTLE_PATH_EMPTY;

         if ( b[OFFSET(8,9)] == EMPTY )
            score -= KCASTLE_PATH_EMPTY;

         if ( (board.black_flags & SF_QRMOVED) == 0 )
         {
            // Might castle either way
            score -= CAN_KQCASTLE_BONUS;

            if ( b[OFFSET(3,9)] == EMPTY )
               score -= QCASTLE_PATH_EMPTY;

            if ( b[OFFSET(4,9)] == EMPTY )
               score -= QCASTLE_PATH_EMPTY;

            if ( b[OFFSET(5,9)] == EMPTY )
               score -= QCASTLE_PATH_EMPTY;
         }
         else
         {
            // Might castle kingside only
            score -= CAN_KCASTLE_BONUS;
         }
      }
      else if ( (board.black_flags & SF_QRMOVED) == 0 )
      {
         // Might castle queenside only
         score -= CAN_QCASTLE_BONUS;

         if ( b[OFFSET(3,9)] == EMPTY )
            score -= QCASTLE_PATH_EMPTY;

         if ( b[OFFSET(4,9)] == EMPTY )
            score -= QCASTLE_PATH_EMPTY;

         if ( b[OFFSET(5,9)] == EMPTY )
            score -= QCASTLE_PATH_EMPTY;
      }
   }

   // *** POSITIONAL STUFF ***

   for ( int ybase = OFFSET(2,2); ybase <= OFFSET(2,9); ybase += NORTH )
   {
      int x, ofs;

      for ( x=0, ofs=ybase; x < 8; x++, ofs++ )
      {
         SQUARE piece = b[ofs];
         if ( piece == EMPTY ) continue;

         if ( piece & WHITE_MASK )
         {
            // It's a White piece
            if ( piece & WP_MASK )
               score += WhitePawnBonus ( board, ofs, x, ybase );
            else if ( piece & (WN_MASK|WB_MASK) )
            {
               if ( piece & WN_MASK )
                  score += WhiteKnightBonus ( b + ofs, ofs, bk );
               else
                  score += WhiteBishopBonus ( b, ofs, ybase, bk );
            }
            else if ( piece & WR_MASK )
               score += WhiteRookBonus ( b, ofs, bk );
            else if ( piece & WQ_MASK )
               score += WhiteQueenBonus ( b, ofs, bk );
         }
         else  // It's a Black piece
         {
            if ( piece & BP_MASK )
               score -= BlackPawnBonus ( board, ofs, x, ybase );
            else if ( piece & (BN_MASK|BB_MASK) )
            {
               if ( piece & BN_MASK )
               {
                  score -= BlackKnightBonus ( b + ofs, ofs, wk );
               }
               else
               {
                  score -= BlackBishopBonus ( b, ofs, ybase, wk );
               }
            }
            else if ( piece & BR_MASK )
               score -= BlackRookBonus ( b, ofs, wk );
            else if ( piece & BQ_MASK )
               score -= BlackQueenBonus ( b, ofs, wk );
         }
      }
   }

   #if PAWN_BALANCE

       // Correct for the strategic importance of possible pawn promotion.
       // (This is not reflected in the simplistic material evaluation.)
       const int nwp = board.inventory [WP_INDEX];   // number of white pawns
       const int nbp = board.inventory [BP_INDEX];   // number of black pawns

       if ( nwp > nbp )
       {
           // 'denom' is value of White's pieces (excludes pawns)
           int denom = board.bmaterial - PAWN_VAL*nbp;
           int balance = (KING_VAL * PawnBalance[nbp][nwp]) / denom;
           score += balance;
       }
       else if ( nbp > nwp )
       {
           // 'denom' is value of Black's pieces (excludes pawns)
           int denom = board.wmaterial - PAWN_VAL*nwp;
           int balance = (KING_VAL * PawnBalance[nwp][nbp]) / denom;
           score -= balance;
       }

   #endif // PAWN_BALANCE

   // Check for two-bishop synergy

   if ( board.inventory[WB_INDEX] == 2 )
      score += TWO_BISHOP_SYNERGY;

   if ( board.inventory[BB_INDEX] == 2 )
      score -= TWO_BISHOP_SYNERGY;

   return score;
}


SCORE ComputerChessPlayer::WhiteMidgameEval ( ChessBoard &board,
                                              int depth,
                                              SCORE beta )
{
   ++evaluated;

   if ( board.IsDefiniteDraw(3) )
      return DRAW;   // We have found a non-stalemate draw

   SCORE score;

   if ( board.WhiteCanMove() )
   {
      score = MaterialEval ( board.wmaterial, board.bmaterial );

      if ( score > beta + SAFE_EVAL_PRUNE_MARGIN )
         return score;

      score += CommonMidgameEval ( board );

      if ( board.white_in_check )
         score -= CHECK_BONUS;
   }
   else
   {
      if ( board.white_in_check )
         return BLACK_WINS + WIN_POSTPONEMENT(depth);
      else
         return DRAW;      // This is a stalemate
   }

   // This is a quiescence nudge for ending up with a tempo.
   score += TEMPO_BONUS;

   if ( score < 0 )
      score += depth;
   else if ( score > 0 )
      score -= depth;

   return score;
}


SCORE ComputerChessPlayer::BlackMidgameEval ( ChessBoard &board,
                                              int depth,
                                              SCORE alpha )
{
   ++evaluated;

   if ( board.IsDefiniteDraw(3) )
      return DRAW;   // We have found a non-stalemate draw

   SCORE score;

   if ( board.BlackCanMove() )
   {
      score = MaterialEval ( board.wmaterial, board.bmaterial );

      if ( score < alpha - SAFE_EVAL_PRUNE_MARGIN )
         return score;

      score += CommonMidgameEval ( board );

      if ( board.black_in_check )
         score += CHECK_BONUS;
   }
   else
   {
      if ( board.black_in_check )
         return WHITE_WINS - WIN_POSTPONEMENT(depth);
      else
         return DRAW;      // This is a stalemate
   }

   // This is a quiescence nudge for ending up with a tempo.
   score -= TEMPO_BONUS;

   if ( score < 0 )
      score += depth;
   else if ( score > 0 )
      score -= depth;

   return score;
}


SCORE ComputerChessPlayer::WhitePawnBonus ( ChessBoard &b,
                                            int         ofs,
                                            int         x,
                                            int         ybase )
{
   // We subtract 'ofs' from OFFSET(11,11) to "rotate" the
   // asymmetric values in the array to be appropriate for White.

   SCORE score = PawnCenter [ OFFSET(11,11) - ofs ];

   if ( x == 0 || x == 7 )
      score -= PAWN_SIDE_FILE;

   // Look for pawns in front of us only.
   // We don't look for pawns behind us, because they would have
   // found us already.
   // Also, we stop as soon as we find a pawn in front of us,
   // because we will let that pawn find any ones ahead of it.
   // This is not just for efficiency; it changes the way things
   // are scored.

   int z;
   cBOOLEAN isPassedPawn = cTRUE;
   const SQUARE *board = b.board;

   for ( z = ofs + NORTH; (board[z] & OFFBOARD) == 0; z += NORTH )
   {
      if ( board[z] & WP_MASK )
      {
         isPassedPawn = cFALSE;
         score -= PAWN_DOUBLED;
         break;
      }

      if ( board[z] & BP_MASK )
      {
         isPassedPawn = cFALSE;
         break;
      }

      if ( (board[z+EAST] & BP_MASK) || (board[z+WEST] & BP_MASK) )
      {
         isPassedPawn = cFALSE;
      }
   }

   #if PAWN_SPLIT
      cBOOLEAN isSplitPawn = cTRUE;
      for ( z = OFFSET(XPART(ofs),3); z < OFFSET(2,9); z += NORTH )
      {
         if ( board[z+EAST] & WP_MASK || board[z+WEST] & WP_MASK )
         {
            isSplitPawn = cFALSE;
            break;
         }
      }

      if ( isSplitPawn )
      {
         score -= PAWN_SPLIT;
      }
   #endif  // PAWN_SPLIT

   if ( board [ofs + SOUTHWEST] & WP_MASK )
   {
      if ( board [ofs + SOUTHEAST] & WP_MASK )
      {
         score += isPassedPawn ? PASSED_PAWN_PROTECT2 : PAWN_PROTECT2;
      }
      else
      {
         score += isPassedPawn ? PASSED_PAWN_PROTECT1 : PAWN_PROTECT1;
      }
   }
   else if ( board [ofs + SOUTHEAST] & WP_MASK )
   {
      score += isPassedPawn ? PASSED_PAWN_PROTECT1 : PAWN_PROTECT1;
   }
   else if ( isPassedPawn )
   {
      score += PASSED_PAWN_ALONE;
   }

   if ( (board [ofs + NORTHEAST] & WB_MASK) ||
        (board [ofs + NORTHWEST] & WB_MASK) )
   {
      score += BISHOP_PROTECT_PAWN;
   }

   if ( (board[ofs+NORTHEAST] & (BN_MASK|BB_MASK|BR_MASK|BQ_MASK|BK_MASK)) &&
        (board[ofs+NORTHWEST] & (BN_MASK|BB_MASK|BR_MASK|BQ_MASK|BK_MASK)) )
   {
      score += PAWN_FORK;
   }

   if ( isPassedPawn )
   {
      switch ( ybase )
      {
         case OFFSET(2,6):  score += PASSED_3_FROM_PROM;   break;
         case OFFSET(2,7):  score += PASSED_2_FROM_PROM;   break;
         case OFFSET(2,8):  score += PASSED_1_FROM_PROM;   break;
      }
   }
   else
   {
      if ( ybase == OFFSET(2,7) )
      {
         score += BLOCKED_2_FROM_PROM;
      }
   }

   return score;
}


SCORE ComputerChessPlayer::BlackPawnBonus ( ChessBoard &b,
                                            int         ofs,
                                            int         x,
                                            int         ybase )
{
   SCORE score = PawnCenter [ofs];

   if ( x == 0 || x == 7 )
   {
      score -= PAWN_SIDE_FILE;
   }

   // Look for pawns in front of us only.
   // We don't look for pawns behind us, because they would have
   // found us already.
   // Also, we stop as soon as we find a pawn in front of us,
   // because we will let that pawn find any ones ahead of it.
   // This is not just for efficiency; it changes the way things
   // are scored.

   int z;
   cBOOLEAN isPassedPawn = cTRUE;
   const SQUARE *board = b.board;

   for ( z = ofs + SOUTH; (board[z] & OFFBOARD) == 0; z += SOUTH )
   {
      if ( board[z] & BP_MASK )
      {
         isPassedPawn = cFALSE;
         score -= PAWN_DOUBLED;
         break;
      }

      if ( board[z] & WP_MASK )
      {
         isPassedPawn = cFALSE;
         break;
      }

      if ( (board[z+EAST] & WP_MASK) || (board[z+WEST] & WP_MASK) )
      {
         isPassedPawn = cFALSE;
      }
   }

   #if PAWN_SPLIT
      cBOOLEAN isSplitPawn = cTRUE;
      for ( z = OFFSET(XPART(ofs),8); z > OFFSET(9,2); z += SOUTH )
      {
         if ( (board[z+EAST] & BP_MASK) || (board[z+WEST] & BP_MASK) )
         {
            isSplitPawn = cFALSE;
            break;
         }
      }

      if ( isSplitPawn )
      {
         score -= PAWN_SPLIT;
      }
   #endif // PAWN_SPLIT

   if ( board [ofs + NORTHWEST] & BP_MASK )
   {
      if ( board [ofs + NORTHEAST] & BP_MASK )
      {
         score += isPassedPawn ? PASSED_PAWN_PROTECT2 : PAWN_PROTECT2;
      }
      else
      {
         score += isPassedPawn ? PASSED_PAWN_PROTECT1 : PAWN_PROTECT1;
      }
   }
   else if ( board [ofs + NORTHEAST] & BP_MASK )
   {
      score += isPassedPawn ? PASSED_PAWN_PROTECT1 : PAWN_PROTECT1;
   }
   else if ( isPassedPawn )
   {
      score += PASSED_PAWN_ALONE;
   }

   if ( (board [ofs + SOUTHEAST] & BB_MASK) ||
        (board [ofs + SOUTHWEST] & BB_MASK) )
   {
      score += BISHOP_PROTECT_PAWN;
   }

   if ( (board[ofs+SOUTHEAST] & (WN_MASK|WB_MASK|WR_MASK|WQ_MASK|WK_MASK)) &&
        (board[ofs+SOUTHWEST] & (WN_MASK|WB_MASK|WR_MASK|WQ_MASK|WK_MASK)) )
   {
      score += PAWN_FORK;
   }

   if ( isPassedPawn )
   {
      switch ( ybase )
      {
         case OFFSET(2,5):  score += PASSED_3_FROM_PROM;   break;
         case OFFSET(2,4):  score += PASSED_2_FROM_PROM;   break;
         case OFFSET(2,3):  score += PASSED_1_FROM_PROM;   break;
      }
   }
   else
   {
      if ( ybase == OFFSET(2,3) )
      {
         score += BLOCKED_2_FROM_PROM;
      }
   }

   return score;
}


/*--- end of file eval.cpp ---*/
