/*=======================================================================

     endgame.cpp   -   Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Knows how to play certain endgame positions better
     than the min-max search does.
     Its opinion (when it offers one) will supersede
     doing a search.

     Revision History:

1994 January 19 [Don Cross]
     Started adding real endgame algorithms!

1994 January 22 [Don Cross]
     I have decided to try something different, and abandon
     the Chester-esque use of special endgame heuristics.
     Instead, I am going to adapt these heuristics into
     a special eval function, and use the normal search code.
     Before starting a search, the ComputerChessPlayer object
     will determine which evaluation function is appropriate
     and use it throughout the search.

1994 January 25 [Don Cross]
     Adding small bonus for getting pieces other than strong king
     close to weak king in EndgameEval1.

=======================================================================*/

#include "chess.h"
#include "profiler.h"

int Distance2 ( int ofs1, int ofs2 )
{
   int dx = XPART(ofs1) - XPART(ofs2);
   int dy = YPART(ofs1) - YPART(ofs2);
   return dx*dx + dy*dy;
}


//--------------------------------------------------------------------------

static const int KingPosTable[] =
{
     0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
     0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,

     0,   0,   100,  90,  70,  50,  50,  70,  90, 100,     0,   0,
     0,   0,    90,  70,  35,  20,  20,  35,  70,  90,     0,   0,
     0,   0,    70,  35,  15,  10,  10,  15,  35,  70,     0,   0,
     0,   0,    50,  20,  10,   0,   0,  10,  20,  50,     0,   0,
     0,   0,    50,  20,  10,   0,   0,  10,  20,  50,     0,   0,
     0,   0,    70,  35,  15,  10,  10,  15,  35,  70,     0,   0,
     0,   0,    90,  70,  35,  20,  20,  35,  70,  90,     0,   0,
     0,   0,   100,  90,  70,  50,  50,  70,  90, 100
};



SCORE ComputerChessPlayer::EndgameEval1 ( ChessBoard &board,
                                          int depth,
                                          SCORE )
{
	PROFILER_ENTER(PX_EVAL);
   ++evaluated;

   // Look for definite draw...
   if ( board.IsDefiniteDraw(3) )
   {
      return DRAW;
   }

   // Look for stalemates and checkmates...
   if ( board.white_to_move )
   {
      if ( !board.WhiteCanMove() )
      {
         if ( board.flags & SF_WCHECK )
         {
            // White has been checkmated...
            return BLACK_WINS + WIN_POSTPONEMENT(depth);
         }
         else
         {
            // White has been stalemated...
            return DRAW;
         }
      }
   }
   else  // It's Black's turn to move
   {
      if ( !board.BlackCanMove() )
      {
         if ( board.flags & SF_BCHECK )
         {
            // Black has been checkmated...
            return WHITE_WINS - WIN_POSTPONEMENT(depth);
         }
         else
         {
            // Black has been stalemated...
            return DRAW;
         }
      }
   }

   SCORE score = MaterialEval ( board.wmaterial, board.bmaterial );

   // If we are using this eval, exactly one side had a lone king
   // at the root of the search tree.  Figure out which side that
   // is by looking at material balance.

   int kdist = Distance2 ( board.wk_offset, board.bk_offset );
   if ( score < 0 )
   {
      // Black is winning...see how bad off White's lone king is now...
      score += (kdist - 10000 - KingPosTable[board.wk_offset]);

      score -= ProximityBonus ( board, board.wk_offset, BR_MASK|BQ_MASK );
   }
   else
   {
      // White is winning...see how bad off Black's lone king is now...
      score -= (kdist - 10000 - KingPosTable[board.bk_offset]);

      score += ProximityBonus ( board, board.bk_offset, WR_MASK|WQ_MASK );
   }

	PROFILER_EXIT()

   return score;
}


SCORE ComputerChessPlayer::ProximityBonus ( ChessBoard &board,
                                            int         target_ofs,
                                            int         mask )
{
   int x, y;
   SCORE bonus = 0;

   for ( y=OFFSET(2,2); y <= OFFSET(2,9); y += NORTH )
   {
      for ( x=0; x < 8; x++ )
      {
         int ofs;
         if ( board.board [ofs = x + y] & mask )
         {
            bonus -= Distance2 ( ofs, target_ofs );
         }
      }
   }

   return bonus / 4;
}




/*--- end of file endgame.cpp ---*/