/*=============================================================

    fancy.cpp  -  Copyright (C) 1993-1996 by Don Cross
    email: dcross@intersrv.com
    WWW:   http://www.intersrv.com/~dcross/

    Contains ParseFancyMove, a function for reading
    more natural move notation from the user.

    Move syntax:

        <piece> - <square>
        <square>              {implicit pawn move}
        <piece> x <piece>
        OO
        O-O
        OOO
        O-O-O

    Revision history:

1996 January 1 [Don Cross]
     Started writing.

=============================================================*/

#include <string.h>

#include "chess.h"


void ComputeMoveSet (
    ChessBoard &,
    const char *fromString,
    const char *toString,
    MoveList & );

void ComputeCaptureSet (
    ChessBoard &,
    const char *fromString,
    const char *toString,
    MoveList & );


cBOOLEAN FindEP (
    ChessBoard &,
    int &source,
    int &dest );


int my_stricmp ( const char *a, const char *b )
{
    while ( *a && *b )
    {
        char ca = *a++;
        if ( ca>='a' && ca<='z' )
            ca -= 'a' - 'A';
        char cb = *b++;
        if ( cb>='a' && cb<='z' )
            cb -= 'a' - 'A';
        if ( ca < cb )
            return -1;
        else if ( ca > cb )
            return 1;
    }

    if ( *a )
       return 1;
    else if ( *b )
       return -1;

    return 0;
}


cBOOLEAN ParseFancyMove (
    const char *string,
    ChessBoard &board,
    int &source,
    int &dest )
{
    if ( my_stricmp(string,"oo")==0 || my_stricmp(string,"o-o")==0 )
    {
        // king-side castle
        if ( board.WhiteToMove() )
        {
            source = OFFSET(6,2);
            dest   = OFFSET(8,2);
        }
        else
        {
            source = OFFSET(6,9);
            dest   = OFFSET(8,9);
        }
        return cTRUE;
    }
    else if ( my_stricmp(string,"ooo")==0 || my_stricmp(string,"o-o-o")==0 )
    {
        // queen-side castle
        if ( board.WhiteToMove() )
        {
            source = OFFSET(6,2);
            dest   = OFFSET(4,2);
        }
        else
        {
            source = OFFSET(6,9);
            dest   = OFFSET(4,9);
        }
        return cTRUE;
    }
    else if ( my_stricmp(string,"ep")==0 || my_stricmp(string,"e.p.")==0 )
    {
        // look for an e.p. capture

        return FindEP ( board, source, dest );
    }
    else
    {
        char buffer [128];
        strncpy ( buffer, string, sizeof(buffer) );

        MoveList moveSet;
        moveSet.num = 0;
        
        // See if this is a non-capture move

        char *p = strchr ( buffer, '-' );
        if ( p )
        {
            *p = '\0';
            const char *fromString = buffer;
            const char *toString = p + 1;
            ComputeMoveSet ( board, fromString, toString, moveSet );
        }
        else
        {
            // See if this is a capture move
            p = strchr ( buffer, 'x' );
            if ( p )
            {
                *p = '\0';
                const char *fromString = buffer;
                const char *toString = p + 1;
                ComputeCaptureSet ( board, fromString, toString, moveSet );
            }
            else
            {
                // See if this is an implicit pawn move (e.g. "e4")
                ComputeMoveSet ( board, "P", buffer, moveSet );
            }
        }

        if ( moveSet.num == 1 )
        {
            // We have found a definite and unambiguous move!

            moveSet.m[0].actualOffsets ( board, source, dest );
            return cTRUE;
        }
    }

    return cFALSE;  // nothing recognizable here
}


cBOOLEAN FindEP (
    ChessBoard &board,
    int &source,
    int &dest )
{
    MoveList legalMoves;
    board.GenMoves ( legalMoves );

    for ( UINT16 i=0; i < legalMoves.num; i++ )
    {
        BYTE tDest = legalMoves.m[i].dest;

        if ( tDest == SPECIAL_MOVE_EP_EAST || tDest == SPECIAL_MOVE_EP_WEST )
        {
            legalMoves.m[i].actualOffsets ( board, source, dest );
            return cTRUE;
        }
    }

    return cFALSE;  // no e.p. capture was found
}


void ComputeMoveSet ( 
    ChessBoard &board, 
    const char *fromString, 
    const char *toString, 
    MoveList &moveSet )
{
    moveSet.num = 0;

    MoveList legalMoves;
    board.GenMoves ( legalMoves );
    const cBOOLEAN white = board.WhiteToMove();

    SQUARE piece = EMPTY;  // piece being moved
    int source = -1;

    // Try to translate 'fromString' into a piece being moved.
    if ( my_stricmp(fromString,"P") == 0 )
        piece = white ? WPAWN : BPAWN;
    else if ( my_stricmp(fromString,"N") == 0 )
        piece = white ? WKNIGHT : BKNIGHT;
    else if ( my_stricmp(fromString,"B") == 0 )
        piece = white ? WBISHOP : BBISHOP;
    else if ( my_stricmp(fromString,"R") == 0 )
        piece = white ? WROOK : BROOK;
    else if ( my_stricmp(fromString,"Q") == 0 )
        piece = white ? WQUEEN : BQUEEN;
    else if ( my_stricmp(fromString,"K") == 0 )
        piece = white ? WKING : BKING;
    else if (
        fromString[0] >= 'a' &&
        fromString[0] <= 'h' &&
        fromString[1] >= '1' &&
        fromString[1] <= '8' &&
        fromString[2] == '\0' )
    {
        piece = white ? WPAWN : BPAWN;
        source = OFFSET ( fromString[0] - 'a' + 2, fromString[1] - '1' + 2 );
    }
    else
        return;  // no good!

    // Try to translate 'toString' into a destination square
    int dest = 0;
    if ( toString[0] >= 'a' &&
         toString[0] <= 'h' &&
         toString[1] >= '1' &&
         toString[1] <= '8' &&
         toString[2] == '\0' )
    {
        dest = OFFSET ( toString[0] - 'a' + 2, toString[1] - '1' + 2 );
    }
    else
        return;  // no good!

    // Search through the list of all legal moves.
    // For each legal move which matches the description,
    // add that move to the moveSet.

    for ( UINT16 i=0; i < legalMoves.num; i++ )
    {
        Move tMove = legalMoves.m[i];
        int tSource, tDest;
        tMove.actualOffsets ( board, tSource, tDest );
        if ( board.GetSquareContents(tSource) == piece &&
             tDest == dest &&
             (source==-1 || source==tSource) )
        {
            // Special case for pawn promotion:
            // Only include promotion to queen.
            // This way, the move does not look "non-unique"
            // just because the pawn can be promoted four 
            // different ways.  The user will be asked how to 
            // promote the pawn anyway.

            if ( (piece & P_MASK) && 
                 (YPART(dest)==2 || YPART(dest)==9) &&
                 (tMove.dest & PIECE_MASK) != Q_INDEX )
            {
                // Ingore the non-queen pawn promotion
            }
            else
            {
                moveSet.AddMove ( tMove );
            }
        }
    }
}


void ComputeCaptureSet ( 
    ChessBoard &board, 
    const char *fromString, 
    const char *toString, 
    MoveList &moveSet )
{
    moveSet.num = 0;

    MoveList legalMoves;
    board.GenMoves ( legalMoves );
    const cBOOLEAN white = board.WhiteToMove();

    SQUARE piece = EMPTY;  // piece being moved
    int source = -1;

    // Try to translate 'fromString' into a piece being moved.
    if ( my_stricmp(fromString,"P") == 0 )
        piece = white ? WPAWN : BPAWN;
    else if ( my_stricmp(fromString,"N") == 0 )
        piece = white ? WKNIGHT : BKNIGHT;
    else if ( my_stricmp(fromString,"B") == 0 )
        piece = white ? WBISHOP : BBISHOP;
    else if ( my_stricmp(fromString,"R") == 0 )
        piece = white ? WROOK : BROOK;
    else if ( my_stricmp(fromString,"Q") == 0 )
        piece = white ? WQUEEN : BQUEEN;
    else if ( my_stricmp(fromString,"K") == 0 )
        piece = white ? WKING : BKING;
    else if (
        fromString[0] >= 'a' &&
        fromString[0] <= 'h' &&
        fromString[1] >= '1' &&
        fromString[1] <= '8' &&
        fromString[2] == '\0' )
    {
        piece = white ? WPAWN : BPAWN;
        source = OFFSET ( fromString[0] - 'a' + 2, fromString[1] - '1' + 2 );
    }
    else
        return;  // no good!

    // Try to translate 'toString' into a piece being captured
    SQUARE capture = EMPTY;
    int dest = -1;

    if ( my_stricmp(toString,"P") == 0 )
        capture = white ? BPAWN : WPAWN;
    else if ( my_stricmp(toString,"N") == 0 )
        capture = white ? BKNIGHT : WKNIGHT;
    else if ( my_stricmp(toString,"B") == 0 )
        capture = white ? BBISHOP : WBISHOP;
    else if ( my_stricmp(toString,"R") == 0 )
        capture = white ? BROOK : WROOK;
    else if ( my_stricmp(toString,"Q") == 0 )
        capture = white ? BQUEEN : WQUEEN;
    else if ( 
        toString[0] >= 'a' &&
        toString[0] <= 'h' &&
        toString[1] >= '1' &&
        toString[1] <= '8' &&
        toString[2] == '\0' )
    {
        dest = OFFSET ( toString[0] - 'a' + 2, toString[1] - '1' + 2 );
        capture = white ? BPAWN : WPAWN;
    }
    else
        return;  // no good!

    // Search through the list of all legal moves.
    // For each legal move which matches the description,
    // add that move to the moveSet.

    for ( UINT16 i=0; i < legalMoves.num; i++ )
    {
        Move tMove = legalMoves.m[i];
        int tSource, tDest;
        tMove.actualOffsets ( board, tSource, tDest );
        if ( board.GetSquareContents(tSource) == piece && 
             board.GetSquareContents(tDest) == capture &&
             (dest==-1 || dest==tDest) &&
             (source==-1 || source==tSource) )
        {
            // Special case for pawn promotion:
            // Only include promotion to queen.
            // This way, the move does not look "non-unique"
            // just because the pawn can be promoted four 
            // different ways.  The user will be asked how to 
            // promote the pawn anyway.

            if ( (piece & P_MASK) && 
                 (YPART(dest)==2 || YPART(dest)==9) &&
                 (tMove.dest & PIECE_MASK) != Q_INDEX )
            {
                // Ingore the non-queen pawn promotion
            }
            else
            {
                moveSet.AddMove ( tMove );
            }
        }
    }
}



/*--- end of file fancy.cpp ---*/