/*==========================================================================

     freelist.h  -  Don Cross, January 1993.

     Freelist class templates.
     These templates are for specialized use:  they are designed to make
     lists of objects which are of uniform and constant memory size.
     Also, the user of these templates must specify in advance a maximum
     number of objects which will ever be needed.

     A FreeList object is not really a container...it is more like a
     a resource from which you may allocate and deallocate chunks of memory
     of uniform size.

Revision history:

1996 July 25 [Don Cross]
     Started adapting this template to Chenard.

==========================================================================*/

#ifndef __DDC_FREELIST_H
#define __DDC_FREELIST_H

#ifndef __DDC_DDC_H
#define __DDC_DDC_H
enum DDCRET
{
   DDC_SUCCESS,           // The operation succeded
   DDC_FAILURE,           // The operation failed for unspecified reasons
   DDC_OUT_OF_MEMORY,     // Operation failed due to running out of memory
   DDC_FILE_ERROR,        // Operation encountered file I/O error
   DDC_INVALID_CALL,      // Operation was called with invalid parameters
   DDC_USER_ABORT,        // Operation was aborted by the user
   DDC_INVALID_FILE       // File format does not match
};
#endif

template <class ListItem>
struct FreeListNode
{
   FreeListNode<ListItem> *next;       // used for linking up the freelist
   ListItem item;                      // the substance of a single item
};


template <class ListItem>
struct FreeListChunk
{
   unsigned numNodes;               // number of nodes in array below
   FreeListNode<ListItem>  *nodes;  // dynamically allocated array of nodes
};


template <class ListItem>
class    FreeList
{
public:
   FreeList()
   {
      numChunks = 0;
      numItems = 0L;
      numCheckedOut = 0;
      chunk = 0;
      front = 0;
   }

   ~FreeList()
   {
      Reset();
   }

   DDCRET SetNumItems ( unsigned long NumItems, unsigned ItemsPerChunk = 0 );
   ListItem *CheckOut();
   void CheckIn ( ListItem *item );

   FreeListNode<ListItem> *CheckOutNode();
   void CheckIn ( FreeListNode<ListItem> *node );

   unsigned long NumItemsTotal() const  { return numItems; }
   unsigned long NumCheckedOut() const  { return numCheckedOut; }
   unsigned long NumAvailable() const { return numItems - numCheckedOut; }

protected:
   void Reset();

private:
   unsigned numChunks;          // number of chunks of items
   unsigned long numItems;      // total number of items in the whole freelist
   unsigned long numCheckedOut;   // number of items checked out of freelist

   FreeListChunk<ListItem> *chunk;   // array of chunks
   FreeListNode<ListItem>  *front;   // push and pop from this node
};


template <class ListItem>
DDCRET FreeList<ListItem>::SetNumItems (
   unsigned long NumItems,
   unsigned ItemsPerChunk )
{
   if ( NumItems == 0 )
   {
      Reset();
      return DDC_SUCCESS;
   }

   const unsigned long MAX_DATA_SIZE = 65400u;

   if ( ItemsPerChunk == 0 )
   {
      ItemsPerChunk = MAX_DATA_SIZE / sizeof(FreeListNode<ListItem>);

      if ( ItemsPerChunk == 0 )
      {
         return DDC_INVALID_CALL;
      }
   }

   unsigned long bytes_per_chunk =
          (unsigned long)ItemsPerChunk *
          (unsigned long)sizeof(FreeListNode<ListItem>);

   if ( bytes_per_chunk > MAX_DATA_SIZE )
   {
      return DDC_INVALID_CALL;   // The chunks would be too big
   }

   unsigned long number_of_chunks =
         (NumItems + (unsigned long)ItemsPerChunk - 1) /
              (unsigned long)ItemsPerChunk;

   if ( number_of_chunks * sizeof(FreeListChunk<ListItem>) > MAX_DATA_SIZE )
   {
      return DDC_INVALID_CALL;   // There would be too many chunks
   }

   chunk = new FreeListChunk<ListItem> [ unsigned(number_of_chunks) ];

   if ( chunk == 0 )
   {
      return DDC_OUT_OF_MEMORY;
   }

   unsigned nodes_in_this_chunk = ItemsPerChunk;
   for ( unsigned i=0; i < unsigned(number_of_chunks); i++ )
   {
      if ( i == unsigned(number_of_chunks) - 1 )
      {
         nodes_in_this_chunk = unsigned ( NumItems % ItemsPerChunk );
         if ( nodes_in_this_chunk == 0 )
         {
            nodes_in_this_chunk = ItemsPerChunk;
         }
      }
      chunk[i].nodes = new FreeListNode<ListItem> [ nodes_in_this_chunk ];
      if ( chunk[i].nodes == 0 )
      {
         while ( --i != unsigned(-1) )
         {
            delete[] chunk[i].nodes;
         }
         delete[] chunk;
         chunk = 0;
         return DDC_OUT_OF_MEMORY;
      }

      chunk[i].numNodes = nodes_in_this_chunk;

      if ( i > 0 )
      {
         chunk[i-1].nodes[ItemsPerChunk - 1].next =
             &(chunk[i].nodes[0]);
      }

      for ( unsigned j=1; j < nodes_in_this_chunk; j++ )
      {
         chunk[i].nodes[j-1].next = &(chunk[i].nodes[j]);
      }
   }

   front = &(chunk[0].nodes[0]);
   numChunks = unsigned(number_of_chunks);
   numItems = NumItems;
   numCheckedOut = 0;

   chunk[numChunks-1].nodes[nodes_in_this_chunk-1].next = 0;

   return DDC_SUCCESS;
}


template <class ListItem>
FreeListNode<ListItem> *FreeList<ListItem>::CheckOutNode()
{
   if ( front == 0 )
   {
      return 0;    // The freelist has been depleted
   }
   else
   {
      FreeListNode<ListItem> *newNode = front;
      front = front->next;
      ++numCheckedOut;
      return newNode;
   }
}


template <class ListItem>
ListItem *FreeList<ListItem>::CheckOut()
{
   FreeListNode<ListItem> *node = CheckOutNode();

   return node ? &(node->item) : 0;
}


template <class ListItem>
void FreeList<ListItem>::CheckIn ( FreeListNode<ListItem> *OldNode )
{
   OldNode->next = front;
   front = OldNode;
   --numCheckedOut;
}



template <class ListItem>
void FreeList<ListItem>::CheckIn ( ListItem *OldItem )
{
   FreeListNode<ListItem> *oldNode =
         (FreeListNode<ListItem> *) ((void **)(OldItem) - 1);

   CheckIn ( oldNode );
}


template <class ListItem>
void FreeList<ListItem>::Reset()
{
   if ( chunk )
   {
      for ( unsigned i=0; i < numChunks; i++ )
      {
         delete[] chunk[i].nodes;
      }

      delete[] chunk;
      chunk = 0;
      numChunks = 0;
      numCheckedOut = 0;
      front = 0;
      numItems = 0;
   }
}


#endif /* __DDC_FREELIST_H */

/*--- end of file freelist.h ---*/
