/*=========================================================================

     game.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Contains class ChessGame.

     Revision history:

1993 July 14 [Don Cross]
     Changed constructor to use initializer list.

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1993 October 19 [Don Cross]
     Added calls to ChessBoard::IsDefiniteDraw() to end the
     game automatically when an obvious draw has occurred.

1993 October 21 [Don Cross]
     Added the ChessUI::RecordMove() function and
     narrowed the meaning of ChessUI::DisplayMove().

1993 October 22 [Don Cross]
     Adding runtime check for sizes of standard integer data types.

1994 January 15 [Don Cross]
     Added check for SPECIAL_MOVE_NULL, which causes the call
     move returned by GetMove to be ignored.  This is useful
     for resetting the game, and might be useful for other things.

1994 January 16 [Don Cross]
     Added logic to fix problem with causing an instant draw
     and ending the game when the board is edited into a drawn
     position.

1994 January 17 [Don Cross]
     Changed check for SPECIAL_MOVE_NULL to a call to Move::ShouldIgnore()

1994 February 9 [Don Cross]
     Changed ChessPlayer::GetMove() to return time spent thinking.
     This way, the ComputerChessPlayer can internally exclude time
     spent animating the move.

1995 March 26 [Don Cross]
     Added new constructor ChessGame::ChessGame ( ChessBoard &, ChessUI &, cBOOLEAN, cBOOLEAN ).
     This constructor refrains from creating the players, but is told what
     kind of players to expect.

1995 July 13 [Don Cross]
     Added ChessGame::~ChessGame.
     Added ChessGame::AutoSaveToFile.

1995 December 30 [Don Cross]
     Made ChessGame::~ChessGame delete blackPlayer and whitePlayer.

1996 August 23 [Don Cross]
     Replacing old memory-based learning tree with disk-based 
     class LearnTree.

1997 March 8 [Don Cross]
     Causing a resignation to issue a negative feedback for continuation
	 in the learning tree.

=========================================================================*/

#include <string.h>
#include <stdio.h>

#include "chess.h"
#include "lrntree.h"

ChessGame::ChessGame ( ChessBoard &Board, ChessUI &Ui ):
    board ( Board ),
    ui ( Ui ),
    autoSave_Filename ( 0 )
{
    whitePlayer = ui.CreatePlayer ( SIDE_WHITE );
    blackPlayer = ui.CreatePlayer ( SIDE_BLACK );
}

ChessGame::ChessGame ( 
    ChessBoard &_board, 
    ChessUI &_ui, 
    cBOOLEAN _whiteIsHuman, 
    cBOOLEAN _blackIsHuman,
    ChessPlayer *_whitePlayer,
    ChessPlayer *_blackPlayer ):
        whitePlayer ( _whitePlayer ),
        blackPlayer ( _blackPlayer ),
        board ( _board ),
        ui ( _ui ),
        autoSave_Filename ( 0 )
{
}


ChessGame::~ChessGame()
{
    if ( autoSave_Filename )
    {
        delete[] autoSave_Filename;
        autoSave_Filename = 0;
    }

    if ( whitePlayer )
    {
        delete whitePlayer;
        whitePlayer = 0;
    }

    if ( blackPlayer )
    {
        delete blackPlayer;
        blackPlayer = 0;
    }
}


cBOOLEAN ChessGame::AutoSaveToFile ( const char *filename )
{
    if ( autoSave_Filename )
    {
        delete[] autoSave_Filename;
    }

    if ( !filename )
    {
        // Interpret this as 'stop saving to file'
        autoSave_Filename = 0;
    }
    else
    {
        autoSave_Filename = new char [ strlen(filename) + 1 ];
        if ( autoSave_Filename )
        {
            strcpy ( autoSave_Filename, filename );
        }
        else
        {
            return cFALSE;   // out of memory!
        }
    }

    return cTRUE;
}


void ChessGame::Play()
{
    if ( sizeof(INT16) != 2 || sizeof(INT32) != 4 )
    {
        // It's OK if this gives us an "unreachable code" warning...
        ChessFatal ( "Integer types are wrong size!" );
    }

    MoveList     ml;
    Move         move;
    UnmoveInfo   unmove;

    if ( !whitePlayer || !blackPlayer )
    {
        ChessFatal ( "Chess player(s) not initialized!" );
    }

    move.dest = SPECIAL_MOVE_NULL;   // to assist in checking for board edit

    for(;;)
    {
        ui.DrawBoard ( board );
        if ( autoSave_Filename )
        {
            SaveGame ( board, autoSave_Filename );
        }

        if ( board.WhiteToMove() )
        {
            board.GenWhiteMoves ( ml );
            if ( ml.num == 0 )
            {
                // The game is over...

                ChessSide winner = board.WhiteInCheck() ? SIDE_BLACK : SIDE_NEITHER;
                ui.ReportEndOfGame ( winner );
                LearnTree tree;
                tree.learnFromGame ( board, winner );
                return;
            }
            else if ( !move.ShouldIgnore() && board.IsDefiniteDraw(3) )
            {
                ui.ReportEndOfGame ( SIDE_NEITHER );
                LearnTree tree;
                tree.learnFromGame ( board, SIDE_NEITHER );
                break;
            }

            INT32 timeSpent = 0;
            if ( !whitePlayer->GetMove ( board, move, timeSpent ) )
            {
                ui.Resign ( SIDE_WHITE );
				LearnTree tree;
				tree.learnFromGame ( board, SIDE_BLACK );
                return;
            }

            if ( !move.ShouldIgnore() )
            {
                ui.RecordMove ( board, move, timeSpent );
                board.MakeWhiteMove ( move, unmove, cTRUE, cTRUE );
            }
        }
        else
        {
            board.GenBlackMoves ( ml );
            if ( ml.num == 0 )
            {
                // The game is over...

                ChessSide winner = board.BlackInCheck() ? SIDE_WHITE : SIDE_NEITHER;
                ui.ReportEndOfGame ( winner );
                LearnTree tree;
                tree.learnFromGame ( board, winner );
                return;
            }
            else if ( !move.ShouldIgnore() && board.IsDefiniteDraw(3) )
            {
                ui.ReportEndOfGame ( SIDE_NEITHER );
                LearnTree tree;
                tree.learnFromGame ( board, SIDE_NEITHER );
                return;
            }

            INT32 timeSpent = 0;
            if ( !blackPlayer->GetMove ( board, move, timeSpent ) )
            {
                ui.Resign ( SIDE_BLACK );
				LearnTree tree;
				tree.learnFromGame ( board, SIDE_WHITE );
                return;
            }

            if ( !move.ShouldIgnore() )
            {
                ui.RecordMove ( board, move, timeSpent );
                board.MakeBlackMove ( move, unmove, cTRUE, cTRUE );
            }
        }
    }
}


cBOOLEAN SaveGame ( const ChessBoard &board, const char *filename )
{
    FILE *gameFile = fopen ( filename, "wb" );
    if ( gameFile )
    {
        UINT32 n = board.GetCurrentPlyNumber();
        for ( UINT32 i=0; i < n; i++ )
        {
            Move m = board.GetPastMove(i);
            fwrite ( &m, sizeof(m), 1, gameFile );
        }
        fclose ( gameFile );
    }
    else
    {
        return cFALSE;
    }

    return cTRUE;
}


cBOOLEAN LoadGame ( ChessBoard &board, const char *filename )
{
    FILE *gameFile = fopen ( filename, "rb" );
    if ( gameFile )
    {
        board.Init();
        Move move;
        for ( UINT32 i=0; fread(&move,sizeof(move),1,gameFile) == 1; i++ )
        {
            if ( (move.dest & SPECIAL_MOVE_MASK) == SPECIAL_MOVE_EDIT )
            {
                board.EditCommand ( move );
                board.SaveSpecialMove ( move );
            }
            else
            {
                UnmoveInfo unmove;
                board.MakeMove ( move, unmove );
            }
        }
        fclose ( gameFile );
    }
    else
    {
        return cFALSE;
    }

    return cTRUE;
}


/*--- end of file game.cpp ---*/
