/*===============================================================================

    lrntree.cpp  -  Copyright (C) 1996 by Don Cross
    email: dcross@intersrv.com
    WWW:   http://www.intersrv.com/~dcross/

    This is a new module for learning new openings.
    This file replaces the old file 'learn.cpp'.
    Now, instead of reading the entire tree into memory, we encapsulate
    direct I/O to a binary file which contains the tree.  This saves
    a lot of dynamic memory and prevents us from having to read and write
    the entire tree all at once.

    Revision history:

1996 August 22 [Don Cross]
     Started writing this code.

1996 September 5 [Don Cross]
     Fixed a bug in LearnTree::familiarPosition().  It was using the
     variable 'n' instead of 'numChildren' in the loop where the random
     number 'r' is used to pick the move.

1996 September 8 [Don Cross]
     Fixed a bug in LearnTree::rememberPosition().  It was not writing
     to the tree file when an existing branch was confirmed at a higher
     search time limit.

1996 September 9 [Don Cross]
     Making LearnTree::trainDepth() call LearnTree::rememberPosition(), so
     as to fix a bug in the former.
     Making rememberPosition() return an integer code indicating whether
     it updated an existing branch or added a new one.

1996 September 29 [Don Cross]
     Fixed bug in LearnTree::familiarPosition() which allowed ChessRandom
     to be called with an argument of 0.  Now, if total==0, we know that
     there is no move which is good enough to be trusted from experience,
     so we return cFALSE, causing the move to be thought about.

=============================================================================*/

// standard includes...
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

// project includes...
#include "chess.h"
#include "lrntree.h"


int Learn_Output = 0;


LearnTree::~LearnTree()
{
    close();
}


int LearnTree::create ( const char *filename )
{
    close();
    f = fopen ( filename, "w+b" );
    if ( !f )
        return 0;

    return 1;
}


int LearnTree::open ( const char *filename )
{
    close();
    f = fopen ( filename, "r+b" );
    if ( !f )
        return 0;   // failure

    return 1;  // success
}


int LearnTree::flush()
{
    if ( !f )
        return 0;    // failure

    fflush(f);
    return 1;   // success
}


int LearnTree::close()
{
    if ( f )
    {
        if ( fclose(f) == EOF )
        {
            f = 0;
            return 0;
        }
        f = 0;
    }

    return 1;   // success
}


INT32 LearnTree::numNodes()
{
    fseek ( f, 0, SEEK_END );
    long pos = ftell(f);
    return pos / LearnBranchFileSize;
}


int LearnTree::read ( INT32 offset, LearnBranch &branch )
{
    if ( !f )
        return 0;   // failure

    if ( fseek ( f, offset * LearnBranchFileSize, SEEK_SET ) != 0 )
        return 0;   // failure

    if ( !read (branch.move) )
        return 0;

    if ( !read (branch.timeAnalyzed) )
        return 0;

    if ( !read (branch.winsAndLosses) )
        return 0;

    if ( !read (branch.child) )
        return 0;

    if ( !read (branch.sibling) )
        return 0;

    for ( int i=0; i < sizeof(branch.reserved)/sizeof(INT32); i++ )
    {
        if ( !read (branch.reserved[i]) )
            return 0;
    }

    return 1;   // success
}


int LearnTree::write ( INT32 offset, const LearnBranch &branch )
{
    if ( !f )
        return 0;   // failure

    if ( fseek ( f, offset * LearnBranchFileSize, SEEK_SET ) != 0 )
        return 0;   // failure

    if ( !write (branch.move) )
        return 0;

    if ( !write (branch.timeAnalyzed) )
        return 0;

    if ( !write (branch.winsAndLosses) )
        return 0;

    if ( !write (branch.child) )
        return 0;

    if ( !write (branch.sibling) )
        return 0;

    for ( int i=0; i < sizeof(branch.reserved)/sizeof(INT32); i++ )
    {
        if ( !write (branch.reserved[i]) )
            return 0;
    }

    return 1;   // success
}


int LearnTree::append ( INT32 &offset, const LearnBranch &branch )
{
    if ( !f )
        return 0;

    if ( fseek ( f, 0, SEEK_END ) != 0 )    // seek to end of file
        return 0;

    long pos = ftell(f);         // get current file location (file size)
    if ( pos == -1 )
        return 0;

    assert ( pos % LearnBranchFileSize == 0 );
    offset = pos / LearnBranchFileSize;
    return write ( offset, branch );
}


int LearnTree::write ( INT32 x )
{
    if ( fputc ( x & 0xff, f ) == EOF )          return 0;
    if ( fputc ( (x >> 8) & 0xff, f ) == EOF )   return 0;
    if ( fputc ( (x >> 16) & 0xff, f ) == EOF )  return 0;
    if ( fputc ( (x >> 24) & 0xff, f ) == EOF )  return 0;
    return 1;
}


int LearnTree::read ( INT32 &x )
{
    int a = fgetc(f);
    int b = fgetc(f);
    int c = fgetc(f);
    int d = fgetc(f);
    if ( a==EOF || b==EOF || c==EOF || d==EOF )
    {
        x = -1;
        return 0;
    }

    unsigned long aa = unsigned(a) & 0xff;
    unsigned long bb = unsigned(b) & 0xff;
    unsigned long cc = unsigned(c) & 0xff;
    unsigned long dd = unsigned(d) & 0xff;

    x = aa | (bb<<8) | (cc<<16) | (dd<<24);
    return 1;
}


int LearnTree::write ( Move m )
{
    if ( fputc ( m.source, f ) == EOF )               return 0;
    if ( fputc ( m.dest, f ) == EOF )                 return 0;
    if ( fputc ( m.score & 0xff, f ) == EOF )         return 0;
    if ( fputc ( (m.score >> 8) & 0xff, f ) == EOF )  return 0;
    return 1;
}


int LearnTree::read ( Move &m )
{
    int a = fgetc(f);
    int b = fgetc(f);
    int c = fgetc(f);
    int d = fgetc(f);

    if ( a==EOF || b==EOF || c==EOF || d==EOF )
    {
        m.source = 0;
        m.dest = 0;
        m.score = 0;
        return 0;
    }
    m.source = a;
    m.dest   = b;
    m.score = c | (d<<8);
    return 1;
}


int LearnTree::insert (
    LearnBranch &branch,
    INT32 parent,
    INT32 &offset )
{
    branch.sibling   =  -1;
    branch.child     =  -1;
    if ( !append ( offset, branch ) ) return 0;

    LearnBranch tbranch;
    if ( parent == -1 )
    {
        if ( offset != 0 )
        {
            // Special case: link new node as sibling to node at offset 0
            if ( !read ( 0, tbranch ) ) return 0;
            branch.sibling = tbranch.sibling;
            tbranch.sibling = offset;
            if ( !write ( 0, tbranch ) ) return 0;
        }
        else
        {
            // This is the first node to be written to the tree.
            // No need to do anything else.
        }
    }
    else
    {
        // Need to backpatch parent to point at this...
        if ( !read ( parent, tbranch ) ) return 0;
        branch.sibling = tbranch.child;
        tbranch.child = offset;
        if ( !write ( parent, tbranch ) ) return 0;
    }

    return write ( offset, branch );
}


//--------------------- high level functions ----------------------


#ifndef LEARNTREE_NO_HIGH_LEVEL


const unsigned MaxLearnDepth = 30;


const char *DefaultFilename = "chenard.tre";


static INT32 ScoreBranch ( const LearnBranch &b, cBOOLEAN whiteToMove )
{
    INT32 score = b.timeAnalyzed / 100;
    const INT32 gameValue = 5;

    if ( whiteToMove )
        score += gameValue * b.winsAndLosses;
    else
        score -= gameValue * b.winsAndLosses;

    if ( score < 0 )
        score = 0;

    return score;
}


cBOOLEAN LearnTree::familiarPosition (
    ChessBoard &board,
    Move &bestmove,
    long timeLimit,
    const MoveList &legalMoves )
{
    if ( board.HasBeenEdited() )
        return cFALSE;

    if ( !open(DefaultFilename) )
        return cFALSE;

    int n = board.GetCurrentPlyNumber();
    INT32 offset = 0;
    LearnBranch branch;

    for ( int i=0; i<n; i++ )
    {
        Move move = board.GetPastMove(i);

        cBOOLEAN foundContinuation = cFALSE;
        while ( offset != -1 && !foundContinuation )
        {
            if ( !read ( offset, branch ) )
                return cFALSE;

            if ( branch.move == move )
                foundContinuation = cTRUE;
            else
                offset = branch.sibling;
        }

        if ( !foundContinuation )
            return cFALSE;

        offset = branch.child;
    }

    // Find the maximum time we have thought about this position.

    INT32 maxTime = 0;
    int numChildren = 0;
    INT32 scores [128];
    Move  moves [128];
    INT32 total = 0;

    while ( offset != -1 )
    {
        if ( !read ( offset, branch ) )
            offset = -1;
        else
        {
            total += scores[numChildren] = ScoreBranch(branch,board.WhiteToMove());
            moves[numChildren] = branch.move;
            ++numChildren;

            if ( branch.timeAnalyzed > maxTime )
                maxTime = branch.timeAnalyzed;

            offset = branch.sibling;
        }
    }

    if ( maxTime >= timeLimit && total > 0 )
    {
        // Choose randomly between the moves with probabilities
        // proportional to the max amount of time Chenard has ever
        // spent deciding that move was best.  This is a kind of credibility
        // factor which should allow play that improves as max training time
        // is progressively increased.

        INT32 r = ChessRandom(total);
        for ( i=0; i<numChildren; i++ )
        {
            r -= scores[i];
            if ( r < 0 )
            {
                bestmove = moves[i];

                // Must do sanity check to make sure file has not been corrupted.
                if ( legalMoves.IsLegal(bestmove) )
                    return cTRUE;
                else
                    return cFALSE;   // If we get here, it means corrupted experience!!!
            }
        }
    }

    return cFALSE;
}


int LearnTree::rememberPosition (
    ChessBoard &board,
    Move bestmove,
    long timeLimit )
{
    if ( board.HasBeenEdited() )
        return 0;

    int n = board.GetCurrentPlyNumber();
    if ( n > MaxLearnDepth )
        return 0;

    if ( !open(DefaultFilename) )
    {
        // Assume this is Chenard's maiden voyage on this computer.
        // Create a brand new, empty tree...

        if ( !create(DefaultFilename) )
            return 0;   // Ain't nothin' we can do!
    }

    // Traverse path to current position, creating missing branches as we go.

    LearnBranch branch;
    INT32 offset = 0;
    INT32 parent = -1;
    for ( int i=0; i<n; i++ )
    {
        INT32 child = -1;
        cBOOLEAN foundReply = cFALSE;
        Move move = board.GetPastMove(i);
        while ( offset != -1 && !foundReply )
        {
            if ( read ( offset, branch ) )
            {
                if ( branch.move == move )
                {
                    foundReply = cTRUE;
                    child = branch.child;
                }
                else
                    offset = branch.sibling;
            }
            else
            {
                assert ( offset == 0 );   // otherwise the file is corrupt
                child = offset = -1;
            }
        }

        if ( !foundReply )
        {
            // Need to add new node now because it doesn't exist already.
            branch.move = move;
            branch.timeAnalyzed = 0;     // computer did NOT think about this
            insert ( branch, parent, offset );   // sets 'offset' to new node
            child = -1;
        }

        parent = offset;
        offset = child;
    }

    // Search for this move in this level of the tree.
    // If we find it, update it if necessary.
    // Otherwise, add a new branch.

    while ( offset != -1 )
    {
        if ( !read ( offset, branch ) )
            return 0;

        if ( branch.move == bestmove )
        {
            // We found it!  Update only if new think time is greater.

            if ( timeLimit > branch.timeAnalyzed )
            {
                branch.timeAnalyzed = timeLimit;
                branch.move.score = bestmove.score;
                write ( offset, branch );
                flush();
            }

            return 1;
        }

        offset = branch.sibling;
    }

    branch.move = bestmove;
    branch.timeAnalyzed = timeLimit;
    insert ( branch, parent, offset );
    flush();

    return 2;
}


void LearnTree::learnFromGame (
    ChessBoard &board,
    ChessSide winner )
{
    if ( winner == SIDE_NEITHER )
    {
        // Currently, we ignore draw games.
        // This might change in the future, so callers should
        // not assume this behavior.

        return;
    }

    if ( !open(DefaultFilename) )  return;

    LearnBranch branch;
    INT32 score = (winner == SIDE_WHITE) ? 1 : -1;
    INT32 offset = 0;
    const int numPlies = board.GetCurrentPlyNumber();
    for ( int ply=0; ply < numPlies; ply++ )
    {
        Move move = board.GetPastMove(ply);

        // search for this move in the current location of the tree.

        cBOOLEAN foundMove = cFALSE;
        while ( offset != -1 )
        {
            if ( !read(offset,branch) )
                return;

            if ( branch.move == move )
            {
                branch.winsAndLosses += score;
                write ( offset, branch );
                foundMove = cTRUE;
                offset = branch.child;
                break;
            }
            else
                offset = branch.sibling;
        }

        if ( !foundMove )
            break;   // We didn't find the continuation, so give up
    }

    flush();
}


//---------------------------- tree trainer -----------------------------

INT32  NodesAtDepthTable [MaxLearnDepth+1];



INT32 LearnTree::numNodesAtDepth (
    INT32 root,
    int targetDepth,
    int currentDepth,
    INT32 searchTime )
{
    LearnBranch branch;
    INT32 total = 0;

    if ( currentDepth < targetDepth )
    {
        while ( root != -1 )
        {
            if ( !read(root,branch) )
                return 0;

            total += numNodesAtDepth (
                branch.child,
                targetDepth,
                currentDepth + 1,
                searchTime );

            root = branch.sibling;
        }
    }
    else if ( targetDepth == currentDepth )
    {
        while ( root != -1 )
        {
            if ( !read(root,branch) )
                return 0;

            if ( branch.timeAnalyzed >= searchTime )
                return 0;

            root = branch.sibling;
        }

        total = 1;
    }

    return total;
}


static void EstimateRemainingTime ( int currentDepth, double thinkTime )
{
    long seconds = long(thinkTime * NodesAtDepthTable[currentDepth]);
    long minutes = seconds/60;
    seconds %= 60;
    long hours = minutes/60;
    minutes %= 60;
    long days = hours/24;
    hours %= 24;

    printf ( "(" );
    if ( days > 0 )
    printf ( "%ld day%s + ", days, (days==1 ? "" : "s") );
    printf ( "%02ld:%02ld:%02ld)", hours, minutes, seconds );

}


void LearnTree::trainDepth (
    INT32 offset,
    INT32 timeLimit,
    ChessUI &ui,
    ComputerChessPlayer *whitePlayer,
    ComputerChessPlayer *blackPlayer,
    ChessBoard &board,
    int targetDepth,
    int currentDepth )
{
    LearnBranch branch;

    if ( currentDepth < targetDepth )
    {
        while ( offset != -1 )
        {
            if ( !read(offset,branch) )
                return;

            UnmoveInfo unmove;
            Move move = branch.move;
            board.MakeMove ( move, unmove );

            trainDepth (
                branch.child,
                timeLimit,
                ui,
                whitePlayer,
                blackPlayer,
                board,
                targetDepth,
                1 + currentDepth );

            board.UnmakeMove ( move, unmove );
            offset = branch.sibling;
        }
    }
    else if ( currentDepth == targetDepth )
    {
        if ( Learn_Output && NodesAtDepthTable[targetDepth]>0 )
        {
            printf ( "\r###  depth=%d --- Remaining nodes at this depth: %-10lu",
                targetDepth,
                NodesAtDepthTable[targetDepth] );

            EstimateRemainingTime ( targetDepth, 0.01*timeLimit );
        }

        // Before we spend time analyzing this position,
        // make sure it hasn't been done so well (or better)
        // before now.

        while ( offset != -1 )
        {
            if ( !read(offset,branch) )
                return;

            if ( branch.timeAnalyzed >= timeLimit )
            {
                // Don't bother...it's already been done as well or better.
                return;
            }

            offset = branch.sibling;
        }

        if ( Learn_Output )
        {
            printf ( "\n" );
        }

        --NodesAtDepthTable [targetDepth];

        ui.DrawBoard ( board );
        INT32 timeSpent = 0;
        Move bestmove;
        MoveList ml;

        if ( board.WhiteToMove() )
        {
            board.GenWhiteMoves ( ml );
            if ( ml.num == 0 )
                return;

            whitePlayer->GetMove ( board, bestmove, timeSpent );

        }
        else
        {
            board.GenBlackMoves ( ml );
            if ( ml.num == 0 )
                return;

            blackPlayer->GetMove ( board, bestmove, timeSpent );
        }

        char moveString [MAX_MOVE_STRLEN + 1];
        FormatChessMove ( board, bestmove, moveString );
        if ( Learn_Output )
        {
            printf ( ">>> Best move = [%s]    score = %d\n", moveString, int(bestmove.score) );
        }

        // See if this move matches any existing replies...
        static unsigned long NumPositionsFinished = 0;
        static unsigned long NumBranches = 0;
        static unsigned long NumUpdates = 0;

        int rc = rememberPosition ( board, bestmove, timeLimit );

        ++NumPositionsFinished;
        if ( rc == 1 )
        {
            ++NumUpdates;
            if ( Learn_Output )
                printf ( ">>> Just modified an existing branch.\n" );
        }
        else if ( rc == 2 )
        {
            ++NumBranches;
            ++NodesAtDepthTable [targetDepth + 1];
            if ( Learn_Output )
                printf ( ">>> Just added a new branch!\n" );
        }

        if ( Learn_Output )
        {
            printf ( ">>> finished=%lu  branches=%lu  updates=%lu\n",
                NumPositionsFinished,
                NumBranches,
                NumUpdates );
        }
    }
}


void LearnTree::trainer (
    ComputerChessPlayer *whitePlayer,
    ComputerChessPlayer *blackPlayer,
    ChessBoard &theBoard,
    ChessUI &theUserInterface,
    INT32 timeLimit )
{
    printf ( "Entering tree trainer!\n\n" );
    if ( !open(DefaultFilename) )
    {
        printf ( "Cannot open training file '%s'\n", DefaultFilename );
        return;
    }

    UINT32 totalNodes = numNodes();
    printf ( "Number of nodes in tree = %lu\n", (unsigned long)totalNodes );

    printf ( "[ " );
    for ( unsigned d=0; d <= MaxLearnDepth+1; d++ )
    {
        NodesAtDepthTable[d] = numNodesAtDepth ( 0, d, 0, timeLimit );
        printf ( "%lu ", (unsigned long)(NodesAtDepthTable[d]) );
    }
    printf ( "]\n" );

    printf ( "\nStarting training...\n\n" );

    for ( d=0; d <= MaxLearnDepth; d++ )
    {
        printf ( "###  Examining depth %u\n", d );
        trainDepth (
            0,
            timeLimit,
            theUserInterface,
            whitePlayer,
            blackPlayer,
            theBoard,
            d,
            0 );
    }
}


#endif   // LEARNTREE_NO_HIGH_LEVEL


/*--- end of file lrntree.cpp ---*/
