/*=========================================================================

     morder.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Contains move-ordering heuristics for the min-max search.

     Revision history:

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1994 February 3 [Don Cross]
     Adding BestPath support.

1994 February 5 [Don Cross]
     Added "killer move" heuristics.

1994 February 10 [Don Cross]
     Adding piece indexing.

1994 February 20 [Don Cross]
     Adding most-probable capture combination min-maxing.

1995 April 5 [Don Cross]
     Added experiment with shuffling move list before sorting it.

1997 June 18 [Don Cross]
     Added PAWN_CAPTURE_PENALTY and FORWARD_BONUS move ordering
	 heuristics.  These improved benchmark score by 13%.

=========================================================================*/

#include "chess.h"
#include "profiler.h"

#define  PREV_SQUARE_BONUS      180
#define  CHECK_BONUS             32
#define  KILLER_MOVE_BONUS       30
#define  HASH_HIST_SHIFT          4
#define  PAWN_CAPTURE_PENALTY     3
#define  PAWN_DANGER_PENALTY      1
#define  FORWARD_BONUS            5

#define  WHITE_BEST_PATH     (20000)
#define  BLACK_BEST_PATH    (-20000)


#ifdef __BORLANDC__
    #pragma argsused
#endif

void ComputerChessPlayer::WhiteMoveOrdering ( const ChessBoard &board,
                                              Move &move,
                                              const UnmoveInfo &unmove,
                                              int depth,
                                              cBOOLEAN bestPathFlag )
{
#ifndef NO_SEARCH
	PROFILER_ENTER(PX_MOVEORDER)

	// First, check for best path.  It overrides everything!

	if ( bestPathFlag &&
		depth <= currentBestPath.depth &&
		depth < MAX_BESTPATH_DEPTH &&
		currentBestPath.m[depth] == move )
	{
		move.score = WHITE_BEST_PATH;
		return;
	}

	move.score = MaterialEval ( board.wmaterial, board.bmaterial );

	const int source = move.source & BOARD_OFFSET_MASK;

	// "killer move" heuristic...

	if ( depth > 0 && depth < MAX_BESTPATH_DEPTH &&
		depth <= nextBestPath[depth-1].depth &&
		nextBestPath[depth-1].m[depth] == move )
	{
		move.score += KILLER_MOVE_BONUS;
	}

	if ( board.flags & SF_BCHECK )
	{
		move.score += CHECK_BONUS;
	}

	if ( move.dest == unmove.prev_move.dest )
	{
		move.score += PREV_SQUARE_BONUS;
	}

	if ( move.dest <= OFFSET(9,9) )
	{
		SQUARE piece = board.board [move.dest];
		move.score -= UPIECE_INDEX(piece);

		if ( piece & WP_MASK )
		{
			const int delta = move.dest - source;
			if ( delta==NORTHEAST || delta==NORTHWEST )
				move.score -= PAWN_CAPTURE_PENALTY;
		}
		else
		{
			const SQUARE *p = & board.board [move.dest];
			if ( (p[NORTHEAST] | p[NORTHWEST]) & BP_MASK )
				move.score -= PAWN_DANGER_PENALTY;
		}

		if ( source <= OFFSET(9,4) && move.dest >= source+10 )
			move.score += FORWARD_BONUS;
	}

	move.score += whiteHist [ move.whiteHash() ] >> HASH_HIST_SHIFT;

	PROFILER_EXIT();
#endif
}


#ifdef __BORLANDC__
    #pragma argsused
#endif

void ComputerChessPlayer::BlackMoveOrdering ( const ChessBoard &board,
                                              Move &move,
                                              const UnmoveInfo &unmove,
                                              int depth,
                                              cBOOLEAN bestPathFlag )
{
#ifndef NO_SEARCH
	PROFILER_ENTER(PX_MOVEORDER);

	// First, check for best path.  It overrides everything!

	if ( bestPathFlag &&
		depth <= currentBestPath.depth &&
		depth < MAX_BESTPATH_DEPTH &&
		currentBestPath.m[depth] == move )
	{
		move.score = BLACK_BEST_PATH;
		return;
	}

	move.score = MaterialEval ( board.wmaterial, board.bmaterial );

	const int source = move.source & BOARD_OFFSET_MASK;

	// "killer move" heuristic...

	if ( depth > 0 && depth < MAX_BESTPATH_DEPTH &&
		depth <= nextBestPath[depth-1].depth &&
		nextBestPath[depth-1].m[depth] == move )
	{
		move.score -= KILLER_MOVE_BONUS;
	}

	if ( board.flags & SF_WCHECK )
	{
		move.score -= CHECK_BONUS;
	}

	if ( move.dest == unmove.prev_move.dest )
	{
		move.score -= PREV_SQUARE_BONUS;
	}

	if ( move.dest <= OFFSET(9,9) )
	{
		const SQUARE piece = board.board [move.dest];
		move.score += UPIECE_INDEX(piece);
		if ( piece & BP_MASK )
		{
			const int delta = move.dest - source;
			if ( delta==SOUTHEAST || delta==SOUTHWEST )
				move.score += PAWN_CAPTURE_PENALTY;
		}
		else
		{
			const SQUARE *p = & board.board [move.dest];
			if ( (p[SOUTHEAST] | p[SOUTHWEST]) & WP_MASK )
				move.score += PAWN_DANGER_PENALTY;
		}

		if ( source >= OFFSET(2,7) && move.dest <= source-10 )
			move.score -= FORWARD_BONUS;
	}

	move.score -= blackHist [ move.blackHash() ] >> HASH_HIST_SHIFT;

	PROFILER_EXIT();
#endif
}


//----------------------------------------------------------------------

void MoveList::WhiteSort()
{
#ifndef NO_SEARCH
    PROFILER_ENTER(PX_MOVEORDER);

    if ( num > 1 )
    {
		SCORE bestscore;
		int   besti, i;
		int   limit = num - 1;

		for ( int dest=0; dest < limit; dest++ )
		{
			bestscore = m [besti = dest].score;
			for ( i=dest + 1; i < num; i++ )
			{
				if ( m[i].score > bestscore )
				{
					bestscore = m[i].score;
					besti = i;
				}
			}

			if ( besti != dest )
			{
				Move t = m[dest];
				m[dest] = m[besti];
				m[besti] = t;
			}
		}
    }

	PROFILER_EXIT()
#endif
}


void MoveList::BlackSort()
{
#ifndef NO_SEARCH
   PROFILER_ENTER(PX_MOVEORDER)

	if ( num > 1 )
	{
		SCORE bestscore;
		int   besti, i;
		int   limit = num - 1;

		for ( int dest=0; dest < limit; dest++ )
		{
			bestscore = m [besti = dest].score;
			for ( i=dest + 1; i < num; i++ )
			{
				if ( m[i].score < bestscore )
				{
					bestscore = m[i].score;
					besti = i;
				}
			}

			if ( besti != dest )
			{
				Move t = m[dest];
				m[dest] = m[besti];
				m[besti] = t;
			}
		}
	}

   PROFILER_EXIT();
#endif
}


/*--- end of file morder.cpp ---*/
