/*==========================================================================

      move.cpp  -  Copyright (C) 1993-1996 by Don Cross
      email: dcross@intersrv.com
      WWW:   http://www.intersrv.com/~dcross/

      Contains code for making moves on a ChessBoard.

      Revision history:

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1993 October 17 [Don Cross]
     Splitting single 'look_for_check' parameter into
     'look_for_self_check' and 'look_for_enemy_check'.

1993 October 18 [Don Cross]
     Fixed bug that happened in simultateous pawn promotion
     and capture:  I was destroying the fake 'dest' value
     which contained the promoted piece before I saved it away.

1994 February 10 [Don Cross]
     Started implementing indexed pieces.

==========================================================================*/

#include "chess.h"
#include "profiler.h"


#define  CHESS_MOVE_DEBUG  0

void ChessBoard::MakeWhiteMove ( Move         &move,
                                 UnmoveInfo   &unmove,
                                 cBOOLEAN      look_for_self_check,
                                 cBOOLEAN      look_for_enemy_check )
{
	PROFILER_ENTER(PX_MAKEMOVE);

   int dest   = move.dest;
   int source = (move.source & BOARD_OFFSET_MASK);

#if CHESS_MOVE_DEBUG
   if ( source < OFFSET(2,2) || source > OFFSET(9,9) )
   {
      ChessFatal ( "Invalid source in ChessBoard::MakeWhiteMove" );
   }
#endif

   SQUARE capture;
   SQUARE piece = board[source];

   unmove.flags		      =   flags;
   unmove.bmaterial       =   bmaterial;
   unmove.wmaterial       =   wmaterial;
   unmove.prev_move       =   prev_move;
   unmove.lastCapOrPawn   =   lastCapOrPawn;
   unmove.hashCode        =   hashCode;

   if ( dest > OFFSET(9,9) )
   {
      // If we get here, it means that this is a "special" move.
      // Special moves store a move code in 'dest', instead of the
      // actual board offset of the move's destination.
      // The move's actual destination is calculated based on
      // what kind of special move this is.

      switch ( dest & SPECIAL_MOVE_MASK )      // Get kind of special move from 'dest'
      {
         case SPECIAL_MOVE_PROMOTE_NORM:
              capture = EMPTY;
              piece = PROM_PIECE ( dest, WHITE_IND, piece );
              --inventory [WP_INDEX];
              ++inventory [SPIECE_INDEX(piece)];
              wmaterial += (RAW_PIECE_VALUE(piece) - PAWN_VAL);
              board [source] = EMPTY;
              board [dest = source + NORTH] = piece;
              break;

         case SPECIAL_MOVE_PROMOTE_CAP_EAST:
              capture = board [source + NORTHEAST];
              piece = PROM_PIECE ( dest, WHITE_IND, piece );
              --inventory [WP_INDEX];    // promoted pawn "disappears"
              ++inventory [SPIECE_INDEX(piece)];   // prom piece "created"
              wmaterial += (RAW_PIECE_VALUE(piece) - PAWN_VAL);
              board [source] = EMPTY;
              board [dest = source + NORTHEAST] = piece;
              break;

         case SPECIAL_MOVE_PROMOTE_CAP_WEST:
              capture = board [source + NORTHWEST];
              piece = PROM_PIECE ( dest, WHITE_IND, piece );
              --inventory [WP_INDEX];    // promoted pawn "disappears"
              ++inventory [SPIECE_INDEX(piece)];   // prom piece "created"
              wmaterial += (RAW_PIECE_VALUE(piece) - PAWN_VAL);
              board [source] = EMPTY;
              board [dest = source + NORTHWEST] = piece;
              break;

         case SPECIAL_MOVE_KCASTLE:
              capture = EMPTY;
              dest = wk_offset = OFFSET(8,2);
              board [OFFSET(6,2)] = EMPTY;
              board [OFFSET(8,2)] = WKING;
              board [OFFSET(9,2)] = EMPTY;
              board [OFFSET(7,2)] = WROOK;
              flags |= (SF_WKMOVED | SF_WKRMOVED | SF_WKCASTLED);
              break;

         case SPECIAL_MOVE_QCASTLE:
              capture = EMPTY;
              dest = wk_offset = OFFSET(4,2);
              board [OFFSET(6,2)] = EMPTY;
              board [OFFSET(4,2)] = WKING;
              board [OFFSET(2,2)] = EMPTY;
              board [OFFSET(5,2)] = WROOK;
              flags |= (SF_WKMOVED | SF_WQRMOVED | SF_WQCASTLED);
              break;

         case SPECIAL_MOVE_EP_EAST:
              board [source] = EMPTY;            // pick up w-pawn
              board [dest = source + NORTHEAST] = piece;   // put w-pawn down
              capture = board [source + EAST];   // remove captured b-pawn
              board [source + EAST] = EMPTY;
              break;

         case SPECIAL_MOVE_EP_WEST:
              board [source] = EMPTY;              // pick up w-pawn
              board [dest = source + NORTHWEST] = piece;  // put w-pawn down
              capture = board [source + WEST];     // remove captured b-pawn
              board [source + WEST] = EMPTY;
              break;
      }
   }
   else   // This is a "normal" move...
   {
      capture        =  board[dest];      // Remember what was captured
      board[dest]    =  board[source];    // Move the piece
      board[source]  =  EMPTY;            // Erase piece from old square

      if ( piece & WK_MASK )
      {
         flags |= SF_WKMOVED;
         wk_offset = dest;
      }
      else
      {
         if ( source == OFFSET(9,2) )
         {
            if ( piece & WR_MASK )
            {
               flags |= SF_WKRMOVED;
            }
         }
         else if ( source == OFFSET(2,2) )
         {
            if ( piece & WR_MASK )
            {
               flags |= SF_WQRMOVED;
            }
         }
      }
   }

   if ( (unmove.capture = capture) != EMPTY )
   {
      // Update material, inventory, etc.
      --inventory [SPIECE_INDEX(capture)];    // one less of the captured piece
      bmaterial -= RAW_PIECE_VALUE(capture);  // deduct material from black

      lastCapOrPawn = ply_number;

      // See if we captured a black rook which had not yet moved...
      // If so, we set its "moved" flag, so that the special case
      // of black moving the other rook onto the square later does not
      // confuse the legal move generator into thinking it can castle!
      // This is because the legal move generator simply checks for
      // the flag NOT being set and a black rook in the square.
      // This is safe even if the rook being captured isn't the original,
      // unmoved rook.  In this case, we are setting the flag redundantly.

      if ( dest == OFFSET(9,9) )
      {
         if ( capture & BR_MASK )
         {
            flags |= SF_BKRMOVED;
         }
      }
      else if ( dest == OFFSET(2,9) )
      {
         if ( capture & BR_MASK )
         {
            flags |= SF_BQRMOVED;
         }
      }
   }
   else if ( piece & WP_MASK )  // not a capture, but might be pawn advance
   {
      lastCapOrPawn = ply_number;
   }

   if ( look_for_self_check )
   {
      if ( IsAttackedByBlack(wk_offset) )
	      flags |= SF_WCHECK;
	  else
	      flags &= ~SF_WCHECK;


      if ( look_for_enemy_check )
      {
         if ( IsAttackedByWhite(bk_offset) )
         {
		    flags |= SF_BCHECK;
            move.source |= CAUSES_CHECK_BIT;
         }
		 else
		    flags &= ~SF_BCHECK;
      }
   }
   else
   {
      // If we get here, it means that we are re-making the move
      // on the board with the knowledge that it is a legal move.
      // It also means that the high-order bit of move.source tells
      // whether this move causes check to black.

      flags &= ~(SF_WCHECK | SF_BCHECK);
	  if ( move.source & CAUSES_CHECK_BIT )
	      flags |= SF_BCHECK;
   }

   hashCode ^= (source ^ dest);
   ++hashCount [hashCode];

   if ( ply_number < MAX_GAME_HISTORY )
   {
      gameHistory [ply_number] = move;
   }

   ++ply_number;
   prev_move = move;
   white_to_move = cFALSE;

	PROFILER_EXIT()
}


//---------------------------------------------------------------------------


void ChessBoard::MakeBlackMove ( Move         &move,
                                 UnmoveInfo   &unmove,
                                 cBOOLEAN      look_for_self_check,
                                 cBOOLEAN      look_for_enemy_check )
{
	PROFILER_ENTER(PX_MAKEMOVE);

   int dest   = move.dest;
   int source = (move.source & BOARD_OFFSET_MASK);

#if CHESS_MOVE_DEBUG
   if ( source < OFFSET(2,2) || source > OFFSET(9,9) )
   {
      ChessFatal ( "Invalid source in ChessBoard::MakeBlackMove" );
   }
#endif

   SQUARE capture;
   SQUARE piece   = board[source];

   unmove.flags           =   flags;
   unmove.bmaterial       =   bmaterial;
   unmove.wmaterial       =   wmaterial;
   unmove.prev_move       =   prev_move;
   unmove.lastCapOrPawn   =   lastCapOrPawn;
   unmove.hashCode        =   hashCode;

   if ( dest > OFFSET(9,9) )
   {
      // If we get here, it means that this is a "special" move.
      // Special moves store a move code in 'dest', instead of the
      // actual board offset of the move's destination.
      // The move's actual destination is calculated based on
      // what kind of special move this is.

      switch ( dest & SPECIAL_MOVE_MASK )      // Get kind of special move from 'dest'
      {
         case SPECIAL_MOVE_PROMOTE_NORM:
              capture = EMPTY;
              piece = PROM_PIECE ( dest, BLACK_IND, piece );
              --inventory [BP_INDEX];
              ++inventory [SPIECE_INDEX(piece)];
              bmaterial += (RAW_PIECE_VALUE(piece) - PAWN_VAL);
              board [source] = EMPTY;
              board [dest = source + SOUTH] = piece;
              break;

         case SPECIAL_MOVE_PROMOTE_CAP_EAST:
              capture = board [source + SOUTHEAST];
              piece = PROM_PIECE ( dest, BLACK_IND, piece );
              --inventory [BP_INDEX];    // promoted pawn "disappears"
              ++inventory [SPIECE_INDEX(piece)];   // prom piece "created"
              bmaterial += (RAW_PIECE_VALUE(piece) - PAWN_VAL);
              board [source] = EMPTY;
              board [dest = source + SOUTHEAST] = piece;
              break;

         case SPECIAL_MOVE_PROMOTE_CAP_WEST:
              capture = board [source + SOUTHWEST];
              piece = PROM_PIECE ( dest, BLACK_IND, piece );
              --inventory [BP_INDEX];    // promoted pawn "disappears"
              ++inventory [SPIECE_INDEX(piece)];   // prom piece "created"
              bmaterial += (RAW_PIECE_VALUE(piece) - PAWN_VAL);
              board [source] = EMPTY;
              board [dest = source + SOUTHWEST] = piece;
              break;

         case SPECIAL_MOVE_KCASTLE:
              capture = EMPTY;
              dest = bk_offset = OFFSET(8,9);
              board [OFFSET(6,9)] = EMPTY;
              board [OFFSET(8,9)] = BKING;
              board [OFFSET(9,9)] = EMPTY;
              board [OFFSET(7,9)] = BROOK;
              flags |= (SF_BKMOVED | SF_BKRMOVED | SF_BKCASTLED);
              break;

         case SPECIAL_MOVE_QCASTLE:
              capture = EMPTY;
              dest = bk_offset = OFFSET(4,9);
              board [OFFSET(6,9)] = EMPTY;
              board [OFFSET(4,9)] = BKING;
              board [OFFSET(2,9)] = EMPTY;
              board [OFFSET(5,9)] = BROOK;
              flags |= (SF_BKMOVED | SF_BQRMOVED | SF_BQCASTLED);
              break;

         case SPECIAL_MOVE_EP_EAST:
              board [source] = EMPTY;
              board [dest = source + SOUTHEAST] = piece;
              capture = board [source + EAST];
              board [source + EAST] = EMPTY;
              break;

         case SPECIAL_MOVE_EP_WEST:
              board [source] = EMPTY;
              board [dest = source + SOUTHWEST] = piece;
              capture = board [source + WEST];
              board [source + WEST] = EMPTY;
              break;
      }
   }
   else
   {
      capture        =  board[dest];
      board[dest]    =  board[source];
      board[source]  =  EMPTY;

      if ( piece & BK_MASK )
      {
         flags |= SF_BKMOVED;
         bk_offset = dest;
      }
      else
      {
         if ( source == OFFSET(9,9) )
         {
            if ( piece & BR_MASK )
            {
               flags |= SF_BKRMOVED;
            }
         }
         else if ( source == OFFSET(2,9) )
         {
            if ( piece & BR_MASK )
            {
               flags |= SF_BQRMOVED;
            }
         }
      }
   }

   if ( (unmove.capture = capture) != EMPTY )
   {
      // Update material, inventory, etc.
      --inventory [SPIECE_INDEX(capture)];    // one less of the captured piece
      wmaterial -= RAW_PIECE_VALUE(capture);  // deduct material from white

      lastCapOrPawn = ply_number;

      // See if we captured a white rook which had not yet moved...
      // If so, we set its "moved" flag, so that the special case
      // of white moving the other rook onto the square later does not
      // confuse the legal move generator into thinking it can castle!
      // This is because the legal move generator simply checks for
      // the flag NOT being set and a white rook in the square.
      // This is safe even if the rook being captured isn't the original,
      // unmoved rook.  In this case, we are setting the flag redundantly.

      if ( dest == OFFSET(9,2) )
      {
         if ( capture & WR_MASK )
         {
            flags |= SF_WKRMOVED;
         }
      }
      else if ( dest == OFFSET(2,2) )
      {
         if ( capture & WR_MASK )
         {
            flags |= SF_WQRMOVED;
         }
      }
   }
   else if ( piece & BP_MASK )  // not a capture, but might be pawn advance
   {
      lastCapOrPawn = ply_number;
   }

   if ( look_for_self_check )
   {
      if ( IsAttackedByWhite(bk_offset) )
	      flags |= SF_BCHECK;
	  else
	      flags &= ~SF_BCHECK;

      if ( look_for_enemy_check )
      {
         if ( IsAttackedByBlack(wk_offset) )
		 {
		    flags |= SF_WCHECK;
            move.source |= CAUSES_CHECK_BIT;
         }
		 else
		    flags &= ~SF_WCHECK;
      }
   }
   else
   {
      // If we get here, it means that we are re-making the move
      // on the board with the knowledge that it is a legal move.
      // It also means that the high-order bit of move.source tells
      // whether this move causes check to white.

      flags &= ~(SF_WCHECK | SF_BCHECK);
	  if ( move.source & CAUSES_CHECK_BIT )
	      flags |= SF_WCHECK;
   }

   hashCode ^= (source ^ dest);
   ++hashCount [hashCode];

   if ( ply_number < MAX_GAME_HISTORY )
   {
      gameHistory [ply_number] = move;
   }

   ++ply_number;
   prev_move = move;
   white_to_move = cTRUE;

	PROFILER_EXIT()
}


/*--- end of file move.cpp ---*/