/*============================================================================

     nummoves.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Counts number of legal moves for a side

     Revision history:

1994 February 22
     Copied and modified genmove.cpp to this file.

============================================================================*/

#include "chess.h"


int ChessBoard::NumWhiteMoves() const
{
   int x, ybase, ofs;
   SQUARE  piece;

   int count = 0;

   for ( ybase = OFFSET(2,2); ybase < OFFSET(2,10); ybase += NORTH )
   {
      for ( x=0; x < 8; x++ )
      {
         piece = board [ofs = ybase + x];

         if ( piece & WHITE_MASK )
         {
            switch ( UPIECE_INDEX(piece) )
            {
               case P_INDEX:
                    count += NumMoves_WP ( ofs, ybase );
                    break;

               case N_INDEX:
                    count += NumMoves_WN ( ofs );
                    break;

               case B_INDEX:
                    count += NumMoves_WB ( ofs );
                    break;

               case R_INDEX:
                    count += NumMoves_WR ( ofs );
                    break;

               case Q_INDEX:
                    count += NumMoves_WQ ( ofs );
                    break;

               case K_INDEX:
                    count += NumMoves_WK ( ofs );
                    break;
            }
         }
      }
   }

   return count;
}


int ChessBoard::NumMoves_WP ( int ofs, int ybase ) const
{
   int count = 0;

   if ( ybase == OFFSET(2,3) )
   {
      // Pawn is on the home row.

      // Check for non-capture moves...
      if ( board [ofs + NORTH] == EMPTY )
      {
         // It can move one square forward...
         ++count;

         // See if we can go two squares!
         if ( board [ofs + 2*NORTH] == EMPTY )
         {
            ++count;
         }
      }

      // Check for capture moves...
      if ( board [ofs + NORTHEAST] & BLACK_MASK )
      {
         ++count;
      }

      if ( board [ofs + NORTHWEST] & BLACK_MASK )
      {
         ++count;
      }
   }
   else if ( ybase == OFFSET(2,6) )
   {
      // Need to look for en passant captures on this rank.

      if ( board [ofs + NORTH] == EMPTY )
      {
         ++count;
      }

      if ( board [ofs + NORTHEAST] & BLACK_MASK )
      {
         ++count;
      }
      else if ( (prev_move.source & BOARD_OFFSET_MASK) == ofs + OFFSET(1,2) &&
                prev_move.dest == ofs + OFFSET(1,0) &&
                (board [prev_move.dest] & BP_MASK) )
      {
         ++count;
      }

      if ( board [ofs + NORTHWEST] & BLACK_MASK )
      {
         ++count;
      }
      else if ( (prev_move.source & BOARD_OFFSET_MASK) == ofs + OFFSET(-1,2) &&
                prev_move.dest == ofs + OFFSET(-1,0) &&
                (board [prev_move.dest] & BP_MASK) )
      {
         ++count;
      }
   }
   else if ( ybase == OFFSET(2,8) )
   {
      // Pawn is one square away from promoting.  See if it can promote...
      if ( board [ofs + NORTH] == EMPTY )
      {
         count += 4;   // Can promote to 4 different pieces
      }

      if ( board [ofs + NORTHEAST] & BLACK_MASK )
      {
         count += 4;
      }

      if ( board [ofs + NORTHWEST] & BLACK_MASK )
      {
         count += 4;
      }
   }
   else
   {
      // Normal pawn move away from home square...

      if ( board [ofs + NORTH] == EMPTY )
      {
         ++count;
      }

      // Check for capture moves...
      if ( board [ofs + NORTHEAST] & BLACK_MASK )
      {
         ++count;
      }

      if ( board [ofs + NORTHWEST] & BLACK_MASK )
      {
         ++count;
      }
   }

   return count;
}


int ChessBoard::NumMoves_WN ( int source ) const
{
   int count = 0;

   if ( (board [source + OFFSET(1,2)] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(1,-2)] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(-1,2)] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(-1,-2)] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(2,1)] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(2,-1)] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(-2,1)] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(-2,-1)] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   return count;
}


int ChessBoard::NumMoves_WB ( int source ) const
{
   int count = 0;
   int dest;

   for ( dest = source + NORTHEAST; board[dest] == EMPTY; dest += NORTHEAST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + NORTHWEST; board[dest] == EMPTY; dest += NORTHWEST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTHEAST; board[dest] == EMPTY; dest += SOUTHEAST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTHWEST; board[dest] == EMPTY; dest += SOUTHWEST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   return count;
}


int ChessBoard::NumMoves_WR ( int source ) const
{
   int count = 0;
   int dest;

   for ( dest = source + NORTH; board[dest] == EMPTY; dest += NORTH )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + WEST; board[dest] == EMPTY; dest += WEST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + EAST; board[dest] == EMPTY; dest += EAST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTH; board[dest] == EMPTY; dest += SOUTH )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   return count;
}


int ChessBoard::NumMoves_WQ ( int source ) const
{
   int count = 0;
   int dest;

   for ( dest = source + NORTHEAST; board[dest] == EMPTY; dest += NORTHEAST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + NORTHWEST; board[dest] == EMPTY; dest += NORTHWEST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTHEAST; board[dest] == EMPTY; dest += SOUTHEAST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTHWEST; board[dest] == EMPTY; dest += SOUTHWEST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + NORTH; board[dest] == EMPTY; dest += NORTH )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + WEST; board[dest] == EMPTY; dest += WEST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + EAST; board[dest] == EMPTY; dest += EAST )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTH; board[dest] == EMPTY; dest += SOUTH )
   {
      ++count;
   }
   if ( board[dest] & BLACK_MASK )
   {
      ++count;
   }

   return count;
}


int ChessBoard::NumMoves_WK ( int source ) const
{
   int count = 0;
   if ( (board [source + NORTH] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + NORTHEAST] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + NORTHWEST] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + EAST] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + WEST] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + SOUTHEAST] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + SOUTHWEST] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + SOUTH] & (WHITE_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   // Check for castling...
   if ( (white_flags & SF_KMOVED) == 0 && !white_in_check )
   {
      // check for O-O
      if ( (white_flags & SF_KRMOVED) == 0  &&
           (board [OFFSET(9,2)] & WR_MASK)  &&
           board [OFFSET(8,2)] == EMPTY     &&
           board [OFFSET(7,2)] == EMPTY     &&
           !IsAttackedByBlack ( OFFSET(7,2) ) )
      {
         ++count;
      }

      // check for O-O-O
      if ( (white_flags & SF_QRMOVED) == 0  &&
           (board [OFFSET(2,2)] & WR_MASK)  &&
           board [OFFSET(3,2)] == EMPTY     &&
           board [OFFSET(4,2)] == EMPTY     &&
           board [OFFSET(5,2)] == EMPTY     &&
           !IsAttackedByBlack ( OFFSET(5,2) ) )
      {
         ++count;
      }
   }
   return count;
}



//--------------------------------------------------------------------------

int ChessBoard::NumBlackMoves() const
{
   int count = 0;
   int x, ybase, ofs;
   SQUARE  piece;

   for ( ybase = OFFSET(2,9); ybase > OFFSET(2,1); ybase += SOUTH )
   {
      for ( x=0; x < 8; x++ )
      {
         piece = board [ofs = ybase + x];

         if ( piece & BLACK_MASK )
         {
            switch ( UPIECE_INDEX(piece) )
            {
               case P_INDEX:
                    count += NumMoves_BP ( ofs, ybase );
                    break;

               case N_INDEX:
                    count += NumMoves_BN ( ofs );
                    break;

               case B_INDEX:
                    count += NumMoves_BB ( ofs );
                    break;

               case R_INDEX:
                    count += NumMoves_BR ( ofs );
                    break;

               case Q_INDEX:
                    count += NumMoves_BQ ( ofs );
                    break;

               case K_INDEX:
                    count += NumMoves_BK ( ofs );
                    break;
            }
         }
      }
   }

   return count;
}


int ChessBoard::NumMoves_BP ( int ofs, int ybase ) const
{
   int count = 0;
   if ( ybase == OFFSET(2,8) )
   {
      // Pawn is on the home row.

      // Check for non-capture moves...
      if ( board [ofs + SOUTH] == EMPTY )
      {
         // It can move one square forward...
         ++count;

         // See if we can go two squares!
         if ( board [ofs + 2*SOUTH] == EMPTY )
         {
            ++count;
         }
      }

      // Check for capture moves...
      if ( board [ofs + SOUTHEAST] & WHITE_MASK )
      {
         ++count;
      }

      if ( board [ofs + SOUTHWEST] & WHITE_MASK )
      {
         ++count;
      }
   }
   else if ( ybase == OFFSET(2,5) )
   {
      // Need to look for en passant captures on this rank.

      if ( board [ofs + SOUTH] == EMPTY )
      {
         ++count;
      }

      if ( board [ofs + SOUTHEAST] & WHITE_MASK )
      {
         ++count;
      }
      else if ( (prev_move.source & BOARD_OFFSET_MASK) == ofs + OFFSET(1,-2) &&
                prev_move.dest == ofs + OFFSET(1,0) &&
                (board [prev_move.dest] & WP_MASK) )
      {
         ++count;
      }

      if ( board [ofs + SOUTHWEST] & WHITE_MASK )
      {
         ++count;
      }
      else if ( (prev_move.source & BOARD_OFFSET_MASK) == ofs + OFFSET(-1,-2) &&
                prev_move.dest == ofs + OFFSET(-1,0) &&
                (board [prev_move.dest] & WP_MASK) )
      {
         ++count;
      }
   }
   else if ( ybase == OFFSET(2,3) )
   {
      // Pawn is one square away from promoting.  See if it can promote...
      if ( board [ofs + SOUTH] == EMPTY )
      {
         count += 4;
      }

      if ( board [ofs + SOUTHEAST] & WHITE_MASK )
      {
         count += 4;
      }

      if ( board [ofs + SOUTHWEST] & WHITE_MASK )
      {
         count += 4;
      }
   }
   else
   {
      // Normal pawn move away from home square...

      if ( board [ofs + SOUTH] == EMPTY )
      {
         ++count;
      }

      // Check for capture moves...
      if ( board [ofs + SOUTHEAST] & WHITE_MASK )
      {
         ++count;
      }

      if ( board [ofs + SOUTHWEST] & WHITE_MASK )
      {
         ++count;
      }
   }
   return count;
}


int ChessBoard::NumMoves_BN ( int source ) const
{
   int count = 0;
   if ( (board [source + OFFSET(1,2)] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(1,-2)] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(-1,2)] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(-1,-2)] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(2,1)] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(2,-1)] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(-2,1)] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + OFFSET(-2,-1)] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }
   return count;
}


int ChessBoard::NumMoves_BB ( int source ) const
{
   int count = 0;
   int dest;

   for ( dest = source + NORTHEAST; board[dest] == EMPTY; dest += NORTHEAST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + NORTHWEST; board[dest] == EMPTY; dest += NORTHWEST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTHEAST; board[dest] == EMPTY; dest += SOUTHEAST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTHWEST; board[dest] == EMPTY; dest += SOUTHWEST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }
   return count;
}


int ChessBoard::NumMoves_BR ( int source ) const
{
   int count = 0;
   int dest;

   for ( dest = source + NORTH; board[dest] == EMPTY; dest += NORTH )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + WEST; board[dest] == EMPTY; dest += WEST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + EAST; board[dest] == EMPTY; dest += EAST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTH; board[dest] == EMPTY; dest += SOUTH )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }
   return count;
}


int ChessBoard::NumMoves_BQ ( int source ) const
{
   int count = 0;
   int dest;

   for ( dest = source + NORTHEAST; board[dest] == EMPTY; dest += NORTHEAST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + NORTHWEST; board[dest] == EMPTY; dest += NORTHWEST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTHEAST; board[dest] == EMPTY; dest += SOUTHEAST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTHWEST; board[dest] == EMPTY; dest += SOUTHWEST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + NORTH; board[dest] == EMPTY; dest += NORTH )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + WEST; board[dest] == EMPTY; dest += WEST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + EAST; board[dest] == EMPTY; dest += EAST )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }

   for ( dest = source + SOUTH; board[dest] == EMPTY; dest += SOUTH )
   {
      ++count;
   }
   if ( board[dest] & WHITE_MASK )
   {
      ++count;
   }
   return count;
}


int ChessBoard::NumMoves_BK ( int source ) const
{
   int count = 0;
   if ( (board [source + NORTH] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + NORTHEAST] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + NORTHWEST] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + EAST] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + WEST] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + SOUTHEAST] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + SOUTHWEST] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   if ( (board [source + SOUTH] & (BLACK_MASK | OFFBOARD)) == 0 )
   {
      ++count;
   }

   // Check for castling...
   if ( (black_flags & SF_KMOVED) == 0 && !black_in_check )
   {
      // check for O-O
      if ( (black_flags & SF_KRMOVED) == 0  &&
           (board [OFFSET(9,9)] & BR_MASK)  &&
           board [OFFSET(8,9)] == EMPTY     &&
           board [OFFSET(7,9)] == EMPTY     &&
           !IsAttackedByWhite ( OFFSET(7,9) ) )
      {
         ++count;
      }

      // check for O-O-O
      if ( (black_flags & SF_QRMOVED) == 0  &&
           (board [OFFSET(2,9)] & BR_MASK)  &&
           board [OFFSET(3,9)] == EMPTY     &&
           board [OFFSET(4,9)] == EMPTY     &&
           board [OFFSET(5,9)] == EMPTY     &&
           !IsAttackedByWhite ( OFFSET(5,9) ) )
      {
         ++count;
      }
   }
   return count;
}


/*--- end of file nummoves.cpp ---*/