/*==========================================================================

     eval.cpp  -  Don Cross, May 1993.

     Contains evaluation heuristics for chess.

     Revision history:

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1993 October 19 [Don Cross]
     Added bonuses for being castled and being able to castle.
     Added small bonus for putting opponent in check.
     Added knowledge of definite draws that aren't stalements.
     Added all kinds of cool positional knowledge about
     bishops, knights, and especially pawns.

1993 October 24 [Don Cross]
     Added MaterialEval() which encourages trades when ahead
     in material, and discourages trades when down in material.

1993 October 25 [Don Cross]
     Added castling security knowledge.

1993 October 31 [Don Cross]
     Added knowledge of knight forks.

1993 December 31 [Don Cross]
     Added knowledge of pawn forks.

1994 January 5 [Don Cross]
     Increased the value of pawn forks.
     Improved blocked-bishop heuristics.
     Improved efficiency of split-pawn heuristics.
     Increased value of split (isolated) pawn.

1994 February 3 [Don Cross]
     Tried to improve knight fork heuristics.

1995 February 6 [Don Cross]
     Changing knight and king position heuristics over to
     lookup tables.

==========================================================================*/

#include "chess.h"

//----------------------------------------------------------------------
// All heuristic constants below are positive, whether they are
// bonuses or penalties.  This is done so that, when looking at
// code, it is clear whether they are bonuses or penalties by
// looking at how they are used, without referring to the definition
// of the constant to see if it has a '-' in front of it!
//----------------------------------------------------------------------

// Miscellaneous -------------------------------------------------------
#define  CHECK_BONUS           2
#define  TEMPO_BONUS           1

// Castling and king ---------------------------------------------------
#define  CAN_KCASTLE_BONUS    20
#define  CAN_QCASTLE_BONUS    18
#define  CAN_KQCASTLE_BONUS   30
#define  HAS_KCASTLED_BONUS   50
#define  HAS_QCASTLED_BONUS   45

#define  CASTLE_SECURITY1     40
#define  CASTLE_SECURITY2     15
#define  CASTLE_KNIGHT_GUARD   5

// The following 'CTEK' values are bonuses for having pieces
// 'Close To Enemy King'.

#define CTEK_KNIGHT   15      // knight less than 4 away
#define CTEK_QUEEN    20      // queen less than 3 away
#define CTEK_BISHOP    5      // bishop less than 4 away
#define CTEK_ROOK     12      // rook less than 3 away

// Knight --------------------------------------------------------------
#define  NFORK_UNCERTAINTY    50

static const SCORE KnightPosition [144] =
{
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,
    0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,0000,

    0000,0000,    -8,  -6,  -5,  -4,  -4,  -5,  -6,  -8,  0000,0000,
    0000,0000,    -6,   2,   1,   0,   0,   1,   2,  -6,  0000,0000,
    0000,0000,    -4,   3,   6,   8,   8,   6,   3,  -4,  0000,0000,
    0000,0000,    -4,   6,   8,  10,  10,   8,   6,  -4,  0000,0000,
    0000,0000,    -5,   2,   6,   7,   7,   6,   2,  -5,  0000,0000,
    0000,0000,    -7,   1,   5,   3,   3,   5,   1,  -7,  0000,0000,
    0000,0000,    -8,  -3,  -1,  -1,  -1,  -1,  -3,  -8,  0000,0000,
    0000,0000,   -10,  -9,  -8,  -7,  -7,  -8,  -9, -10
};

// Bishop --------------------------------------------------------------
#define  BISHOP_BACK_RANK         11
#define  BISHOP_MOBILITY_SCALER    2
#define  CENTER_BLOCK_BISHOP1     20
#define  CENTER_BLOCK_BISHOP2      7
#define  TWO_BISHOP_SYNERGY       10

// Pawn ----------------------------------------------------------------
#define  PAWN_FORK                40
#define  PAWN_SIDE_FILE           20
#define  PAWN_DOUBLED             28
#define  PAWN_SPLIT               32
#define  PAWN_PROTECT1             3
#define  PAWN_PROTECT2             5
#define  BISHOP_PROTECT_PAWN       2
#define  PASSED_PAWN_PROTECT1     40
#define  PASSED_PAWN_PROTECT2     45
#define  PASSED_PAWN_ALONE        30
#define  PASSED_3_FROM_PROM       50
#define  PASSED_2_FROM_PROM       75
#define  PASSED_1_FROM_PROM      150
#define  BLOCKED_2_FROM_PROM      10

// Rook ----------------------------------------------------------------
#define  ROOK_OPEN_FILE             6
#define  ROOK_CAN_REACH_7TH_RANK    7
#define  ROOK_ON_7TH_RANK          12
#define  ROOK_CONNECT_VERT          4
#define  ROOK_CONNECT_HOR           2
#define  ROOK_IMMOBILE_HORIZ        5
#define  ROOK_IMMOBILE             15



static SCORE WhiteOOSecurity ( const SQUARE *b,
                               cBOOLEAN      alreadyCastled,
                               const INT16   inventory[] )
{
   SCORE bonus = 0;

   if ( !alreadyCastled ||
        (alreadyCastled &&
            ((b[OFFSET(7,2)] & WK_MASK) ||
             (b[OFFSET(8,2)] & WK_MASK) ||
             (b[OFFSET(9,2)] & WK_MASK))) )
   {
      if ( inventory [BQ_INDEX] > 0 ||
           inventory [BR_INDEX] > 1 )
      {
         if ( (b[OFFSET(7,3)] & WP_MASK) &&
              ((b[OFFSET(8,3)] & WP_MASK) || (b[OFFSET(8,4)] & WP_MASK)) &&
              ((b[OFFSET(9,3)] & WP_MASK) || (b[OFFSET(9,4)] & WP_MASK)) )
         {
            bonus += alreadyCastled ? CASTLE_SECURITY1 : CASTLE_SECURITY2;

            if ( b[OFFSET(7,4)] & WN_MASK )
            {
               bonus += CASTLE_KNIGHT_GUARD;
            }
         }
      }
   }

   return bonus;
}


static SCORE WhiteOOOSecurity ( const SQUARE  *b,
                                cBOOLEAN       alreadyCastled,
                                const INT16    inventory[] )
{
   SCORE bonus = 0;

   if ( !alreadyCastled ||
        (alreadyCastled &&
            ((b[OFFSET(2,2)] & WK_MASK) ||
             (b[OFFSET(3,2)] & WK_MASK) ||
             (b[OFFSET(4,2)] & WK_MASK))) )
   {
      if ( inventory [BQ_INDEX] > 0 ||
           inventory [BR_INDEX] > 1 )
      {
         if ( (b[OFFSET(4,3)] & WP_MASK) &&
              ((b[OFFSET(3,3)] & WP_MASK) || (b[OFFSET(3,4)] & WP_MASK)) &&
              ((b[OFFSET(2,3)] & WP_MASK) || (b[OFFSET(2,4)] & WP_MASK)) )
         {
            bonus += alreadyCastled ? CASTLE_SECURITY1 : CASTLE_SECURITY2;

            if ( b[OFFSET(4,4)] & WN_MASK )
            {
               bonus += CASTLE_KNIGHT_GUARD;
            }
         }
      }
   }

   return bonus;
}


static SCORE BlackOOSecurity ( const SQUARE  *b,
                               cBOOLEAN       alreadyCastled,
                               const INT16    inventory[] )
{
   SCORE bonus = 0;

   if ( !alreadyCastled ||
        (alreadyCastled &&
        ((b[OFFSET(7,9)] & BK_MASK) ||
             (b[OFFSET(8,9)] & BK_MASK) ||
             (b[OFFSET(9,9)] & BK_MASK))) )
   {
      if ( inventory [WQ_INDEX] > 0 ||
           inventory [WR_INDEX] > 1 )
      {
         if ( (b[OFFSET(7,8)] & BP_MASK) &&
              ((b[OFFSET(8,8)] & BP_MASK) || (b[OFFSET(8,7)] & BP_MASK)) &&
              ((b[OFFSET(9,8)] & BP_MASK) || (b[OFFSET(9,7)] & BP_MASK)) )
         {
            bonus += alreadyCastled ? CASTLE_SECURITY1 : CASTLE_SECURITY2;

            if ( b[OFFSET(7,7)] & BN_MASK )
            {
               bonus += CASTLE_KNIGHT_GUARD;
            }
         }
      }
   }

   return bonus;
}


static SCORE BlackOOOSecurity ( const SQUARE  *b,
                                cBOOLEAN       alreadyCastled,
                                const INT16    inventory[] )
{
   SCORE bonus = 0;

   if ( !alreadyCastled ||
        (alreadyCastled &&
            ((b[OFFSET(2,9)] & BK_MASK) ||
             (b[OFFSET(3,9)] & BK_MASK) ||
             (b[OFFSET(4,9)] & BK_MASK))) )
   {
      if ( inventory [WQ_INDEX] > 0 ||
           inventory [WR_INDEX] > 1 )
      {
         if ( (b[OFFSET(4,8)] & BP_MASK) &&
              ((b[OFFSET(3,8)] & BP_MASK) || (b[OFFSET(3,7)] & BP_MASK)) &&
              ((b[OFFSET(2,8)] & BP_MASK) || (b[OFFSET(2,7)] & BP_MASK)) )
         {
            bonus += alreadyCastled ? CASTLE_SECURITY1 : CASTLE_SECURITY2;

            if ( b[OFFSET(4,7)] & BN_MASK )
            {
               bonus += CASTLE_KNIGHT_GUARD;
            }
         }
      }
   }

   return bonus;
}


static SCORE WhiteRookBonus ( const SQUARE *b, int ofs )
{
   SCORE bonus = 0;

   // First, see if we are on the seventh rank or the eighth rank...

   if ( ofs >= OFFSET(2,8) )
   {
      if ( ofs <= OFFSET(9,8) )
      {
         bonus += ROOK_ON_7TH_RANK;
      }
      else
      {
         bonus += (ROOK_CAN_REACH_7TH_RANK + ROOK_OPEN_FILE);
      }
   }
   else
   {
      // A file is open if none of our pawns is blocking it before 7th.
      for ( int z=ofs + NORTH; b[z] == EMPTY && z < OFFSET(2,8); z += NORTH );

      bonus += (YPART(z) - 2) / 2;

      if ( z < OFFSET(2,8) )
      {
         if ( !(b[z] & WP_MASK) )
         {
            bonus += ROOK_OPEN_FILE;
         }
      }
      else if ( z > OFFSET(9,8) )
      {
         bonus += ROOK_CAN_REACH_7TH_RANK;
      }

      for ( z=ofs + SOUTH; b[z] == EMPTY; z += SOUTH );
      if ( b[z] & (WR_MASK | WQ_MASK) )
      {
         bonus += ROOK_CONNECT_VERT;
      }

      for ( z=ofs + WEST; b[z] == EMPTY; z += WEST );
      if ( b[z] & WR_MASK )
      {
         bonus += ROOK_CONNECT_HOR;
      }
   }

   if ( (b[ofs+EAST] & (WHITE_MASK | OFFBOARD)) &&
        (b[ofs+WEST] & (WHITE_MASK | OFFBOARD)) )
   {
      if ( (b[ofs+NORTH] & (WHITE_MASK | OFFBOARD)) &&
           (b[ofs+SOUTH] & (WHITE_MASK | OFFBOARD)) )
      {
         bonus -= ROOK_IMMOBILE;
      }
      else
      {
         bonus -= ROOK_IMMOBILE_HORIZ;
      }
   }

   return bonus;
}


static SCORE BlackRookBonus ( const SQUARE *b, int ofs )
{
   SCORE bonus = 0;

   // First, see if we are on the seventh rank or the eighth rank...

   if ( ofs <= OFFSET(9,3) )
   {
      if ( ofs >= OFFSET(2,3) )
      {
         bonus += ROOK_ON_7TH_RANK;
      }
      else
      {
         bonus += (ROOK_CAN_REACH_7TH_RANK + ROOK_OPEN_FILE);
      }
   }
   else
   {
      // A file is open if none of our pawns is blocking it before 7th.
      for ( int z=ofs + SOUTH; b[z] == EMPTY && z > OFFSET(9,3); z += SOUTH );

      bonus += (9 - YPART(z)) / 2;

      if ( z > OFFSET(9,3) )
      {
         if ( !(b[z] & BP_MASK) )
         {
            bonus += ROOK_OPEN_FILE;
         }
      }
      else if ( z < OFFSET(3,3) )
      {
         bonus += ROOK_CAN_REACH_7TH_RANK;
      }

      for ( z=ofs + NORTH; b[z] == EMPTY; z += NORTH );
      if ( b[z] & (BR_MASK | BQ_MASK) )
      {
         bonus += ROOK_CONNECT_VERT;
      }

      for ( z=ofs + WEST; b[z] == EMPTY; z += WEST );
      if ( b[z] & BR_MASK )
      {
         bonus += ROOK_CONNECT_HOR;
      }
   }

   if ( (b[ofs+EAST] & (BLACK_MASK | OFFBOARD)) &&
        (b[ofs+WEST] & (BLACK_MASK | OFFBOARD)) )
   {
      if ( (b[ofs+NORTH] & (BLACK_MASK | OFFBOARD)) &&
           (b[ofs+SOUTH] & (BLACK_MASK | OFFBOARD)) )
      {
         bonus -= ROOK_IMMOBILE;
      }
      else
      {
         bonus -= ROOK_IMMOBILE_HORIZ;
      }
   }

   return bonus;
}


static int WhiteBishopMobility ( const SQUARE *insideBoard )
{
   int count = 0;
   const SQUARE *p;

   for ( p = insideBoard + NORTHEAST;
         *p == EMPTY;
         p += NORTHEAST )
   {
      if ( !(p[NORTHEAST] & BP_MASK) && !(p[NORTHWEST] & BP_MASK) )
      {
         ++count;
      }
   }

   for ( p = insideBoard + NORTHWEST;
         *p == EMPTY;
         p += NORTHWEST )
   {
      if ( !(p[NORTHEAST] & BP_MASK) && !(p[NORTHWEST] & BP_MASK) )
      {
         ++count;
      }
   }

   for ( p = insideBoard + SOUTHEAST;
         *p == EMPTY;
         p += SOUTHEAST )
   {
      if ( !(p[NORTHEAST] & BP_MASK) && !(p[NORTHWEST] & BP_MASK) )
      {
         ++count;
      }
   }

   for ( p = insideBoard + SOUTHWEST;
         *p == EMPTY;
         p += SOUTHWEST )
   {
      if ( !(p[NORTHEAST] & BP_MASK) && !(p[NORTHWEST] & BP_MASK) )
      {
         ++count;
      }
   }

   return count / BISHOP_MOBILITY_SCALER;
}


static int BlackBishopMobility ( const SQUARE *insideBoard )
{
   int count = 0;
   const SQUARE *p;

   for ( p = insideBoard + NORTHEAST;
         *p == EMPTY;
         p += NORTHEAST )
   {
      if ( !(p[SOUTHEAST] & WP_MASK) && !(p[SOUTHWEST] & WP_MASK) )
      {
         ++count;
      }
   }

   for ( p = insideBoard + NORTHWEST;
         *p == EMPTY;
         p += NORTHWEST )
   {
      if ( !(p[SOUTHEAST] & WP_MASK) && !(p[SOUTHWEST] & WP_MASK) )
      {
         ++count;
      }
   }

   for ( p = insideBoard + SOUTHEAST;
         *p == EMPTY;
         p += SOUTHEAST )
   {
      if ( !(p[SOUTHEAST] & WP_MASK) && !(p[SOUTHWEST] & WP_MASK) )
      {
         ++count;
      }
   }

   for ( p = insideBoard + SOUTHWEST;
         *p == EMPTY;
         p += SOUTHWEST )
   {
      if ( !(p[SOUTHEAST] & WP_MASK) && !(p[SOUTHWEST] & WP_MASK) )
      {
         ++count;
      }
   }

   return count / BISHOP_MOBILITY_SCALER;
}


static SCORE WhiteKnightFork ( const SQUARE *p )
{
   int count = 0;
   int rcount = 0;
   int qcount = 0;
   int kcount = 0;
   SQUARE s;

   if ( (s = p[OFFSET(1,2)]) & (BR_MASK | BQ_MASK | BK_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-1,2)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK)        ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(1,-2)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-1,-2)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(2,1)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(2,-1)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-2,1)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-2,-1)]) & (BR_MASK | BQ_MASK) )
   {
      ++count;
      if ( s & BK_MASK )       ++kcount;
      else if ( s & BQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( count > 1 )
   {
      // Something looks forked

      if ( kcount > 0 )
      {
         if ( qcount > 0 )
         {
            return QUEEN_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
         }
         else
         {
            return ROOK_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
         }
      }
      else if ( qcount > 1 )
      {
         return QUEEN_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
      }
      else
      {
         return ROOK_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
      }
   }

   return 0;
}


static SCORE BlackKnightFork ( const SQUARE *p )
{
   int count = 0;
   int rcount = 0;
   int qcount = 0;
   int kcount = 0;
   SQUARE s;

   if ( (s = p[OFFSET(1,2)]) & (WR_MASK | WQ_MASK | WK_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-1,2)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(1,-2)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-1,-2)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(2,1)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(2,-1)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-2,1)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( (s = p[OFFSET(-2,-1)]) & (WR_MASK | WQ_MASK) )
   {
      ++count;
      if ( s & WK_MASK )       ++kcount;
      else if ( s & WQ_MASK )  ++qcount;
      else                     ++rcount;
   }

   if ( count > 1 )
   {
      // Something looks forked

      if ( kcount > 0 )
      {
         if ( qcount > 0 )
     {
            return QUEEN_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
         }
         else
         {
            return ROOK_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
         }
      }
      else if ( qcount > 1 )
      {
         return QUEEN_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
      }
      else
      {
         return ROOK_VAL - KNIGHT_VAL - NFORK_UNCERTAINTY;
      }
   }

   return 0;
}


// Put stuff in CommonMidgameEval() which does not depend on whose turn it is.

SCORE ComputerChessPlayer::CommonMidgameEval ( ChessBoard &board )
{
   SCORE score = MaterialEval ( board.wmaterial, board.bmaterial );
   const SQUARE *b = board.board;

   // Check white castling...

   if ( board.white_flags & SF_KMOVED )
   {
      if ( board.white_flags & SF_KCASTLED )
      {
         score += HAS_KCASTLED_BONUS;
         score += WhiteOOSecurity ( b, cTRUE, board.inventory );
      }
      else if ( board.white_flags & SF_QCASTLED )
      {
         score += HAS_QCASTLED_BONUS;
         score += WhiteOOOSecurity ( b, cTRUE, board.inventory );
      }
   }
   else
   {
      // The white king has not yet moved

      if ( (board.white_flags & SF_KRMOVED) == 0 )
      {
         if ( (board.white_flags & SF_QRMOVED) == 0 )
         {
            // Might castle either way
            score += CAN_KQCASTLE_BONUS;

            int oo  = WhiteOOSecurity ( b, cFALSE, board.inventory );
            int ooo = WhiteOOOSecurity ( b, cFALSE, board.inventory );

            score += (oo > ooo) ? oo : ooo;
         }
         else
         {
            // Might castle kingside only
            score += CAN_KCASTLE_BONUS;
            score += WhiteOOSecurity ( b, cFALSE, board.inventory );
         }
      }
      else
      {
         // Might castle queenside only
         score += CAN_QCASTLE_BONUS;
         score += WhiteOOOSecurity ( b, cFALSE, board.inventory );
      }
   }

   // Check black castling...

   if ( board.black_flags & SF_KMOVED )
   {
      if ( board.black_flags & SF_KCASTLED )
      {
         score -= HAS_KCASTLED_BONUS;
         score -= BlackOOSecurity ( b, cTRUE, board.inventory );
      }
      else if ( board.black_flags & SF_QCASTLED )
      {
         score -= HAS_QCASTLED_BONUS;
         score -= BlackOOOSecurity ( b, cTRUE, board.inventory );
      }
   }
   else
   {
      // The black king has not yet moved

      if ( (board.black_flags & SF_KRMOVED) == 0 )
      {
         if ( (board.black_flags & SF_QRMOVED) == 0 )
         {
            // Might castle either way
            score -= CAN_KQCASTLE_BONUS;
            int oo  = BlackOOSecurity ( b, cFALSE, board.inventory );
            int ooo = BlackOOOSecurity ( b, cFALSE, board.inventory );
            score -= (oo > ooo) ? oo : ooo;
         }
         else
         {
            // Might castle kingside only
            score -= CAN_KCASTLE_BONUS;
            score -= BlackOOSecurity ( b, cFALSE, board.inventory );
         }
      }
      else
      {
         // Might castle queenside only
         score -= CAN_QCASTLE_BONUS;
         score -= BlackOOOSecurity ( b, cFALSE, board.inventory );
      }
   }

   // Do positional stuff...this is where indexed pieces would really help!

   for ( int ybase = OFFSET(2,2); ybase <= OFFSET(2,9); ybase += NORTH )
   {
      int x, ofs;

      for ( x=0, ofs=ybase; x < 8; x++, ofs++ )
      {
         switch ( NOINDEX ( b[ofs] ) )
         {
            case WPAWN:
                 score += WhitePawnBonus ( board, ofs, x, ybase );
                 break;

            case WKNIGHT:
                 score += KnightPosition [ ofs ];

                 if ( Distance ( ofs, board.bk_offset ) < 4 )
                 {
                    score += CTEK_KNIGHT;
                 }

                 score += WhiteKnightFork ( b + ofs );
                 break;

            case WBISHOP:
                 if ( ybase == OFFSET(2,2) )
                 {
                    score -= BISHOP_BACK_RANK;
                 }

                 score += WhiteBishopMobility ( b + ofs );

                 if ( Distance ( ofs, board.bk_offset ) < 4 )
                 {
            score += CTEK_BISHOP;
                 }
                 break;

            case WROOK:
                 score += WhiteRookBonus ( b, ofs );

                 if ( Distance ( ofs, board.bk_offset ) < 3 )
                 {
                    score += CTEK_ROOK;
                 }
                 break;

            case WQUEEN:
                 if ( Distance ( ofs, board.bk_offset ) < 3 )
                 {
                    score += CTEK_QUEEN;
                 }
                 break;

            case BPAWN:
                 score -= BlackPawnBonus ( board, ofs, x, ybase );
                 break;

            case BKNIGHT:
                 score -= KnightPosition [ 144 - ofs ];

                 if ( Distance ( ofs, board.wk_offset ) < 4 )
                 {
                    score -= CTEK_KNIGHT;
                 }

                 score -= BlackKnightFork ( b + ofs );
                 break;

            case BBISHOP:
                 if ( ybase == OFFSET(2,9) )
                 {
                    score += BISHOP_BACK_RANK;
                 }

                 score -= BlackBishopMobility ( b + ofs );

                 if ( Distance ( ofs, board.wk_offset ) < 4 )
                 {
                    score -= CTEK_BISHOP;
                 }
                 break;

            case BROOK:
                 score -= BlackRookBonus ( b, ofs );

                 if ( Distance ( ofs, board.wk_offset ) < 3 )
                 {
                    score -= CTEK_ROOK;
                 }
                 break;

            case BQUEEN:
                 if ( Distance ( ofs, board.wk_offset ) < 3 )
                 {
                    score -= CTEK_QUEEN;
                 }
                 break;
         }
      }
   }

   // This is a kludge to try to prevent computer from putting
   // pieces in front of QP or KP and blocking the other bishop
   // from development through the center.

   if ( b [ OFFSET(7,2) ] & WB_MASK )
   {
      if ( b [ OFFSET(6,4) ] != EMPTY && (b [ OFFSET(6,3) ] & WP_MASK) )
      {
         score -= CENTER_BLOCK_BISHOP1;
      }

      if ( b [ OFFSET(5,4) ] != EMPTY )
      {
         score -= CENTER_BLOCK_BISHOP2;
      }
   }

   if ( b [ OFFSET(4,2) ] & WB_MASK )
   {
      if ( b [ OFFSET(5,4) ] != EMPTY && (b [ OFFSET(5,3) ] & WP_MASK) )
      {
         score -= CENTER_BLOCK_BISHOP1;
      }

      if ( b [ OFFSET(6,4) ] != EMPTY )
      {
         score -= CENTER_BLOCK_BISHOP2;
      }
   }

   if ( b [ OFFSET(7,9) ] & BB_MASK )
   {
      if ( b [ OFFSET(6,7) ] != EMPTY && (b [ OFFSET(6,8) ] & BP_MASK) )
      {
         score += CENTER_BLOCK_BISHOP1;
      }

      if ( b [ OFFSET(5,7) ] != EMPTY )
      {
         score += CENTER_BLOCK_BISHOP2;
      }
   }

   if ( b [ OFFSET(4,9) ] & BB_MASK )
   {
      if ( b [ OFFSET(5,7) ] != EMPTY && (b [ OFFSET(5,8) ] & BP_MASK) )
      {
         score += CENTER_BLOCK_BISHOP1;
      }

      if ( b [ OFFSET(6,7) ] != EMPTY )
      {
         score += CENTER_BLOCK_BISHOP2;
      }
   }

   // Check for two-bishop synergy

   if ( board.inventory[WB_INDEX] == 2 )
   {
      score += TWO_BISHOP_SYNERGY;
   }

   if ( board.inventory[BB_INDEX] == 2 )
   {
      score -= TWO_BISHOP_SYNERGY;
   }

   return score;
}


SCORE ComputerChessPlayer::WhiteMidgameEval ( ChessBoard &board, int depth )
{
   ++evaluated;

   if ( board.IsDefiniteDraw() )
   {
      return DRAW;   // We have found a non-stalemate draw
   }

   SCORE score;

   if ( board.WhiteCanMove() )
   {
      score = CommonMidgameEval ( board );

      if ( board.white_in_check )
      {
         score -= CHECK_BONUS;
      }
   }
   else
   {
      if ( board.white_in_check )
      {
         return BLACK_WINS + WIN_POSTPONEMENT(depth);
      }
      else
      {
         return DRAW;      // This is a stalemate
      }
   }

   // This is a quiescence nudge for ending up with a tempo.
   score += TEMPO_BONUS;

   if ( score < 0 )
   {
      score += depth;
   }
   else if ( score > 0 )
   {
      score -= depth;
   }

   return score;
}


SCORE ComputerChessPlayer::BlackMidgameEval ( ChessBoard &board, int depth )
{
   ++evaluated;

   if ( board.IsDefiniteDraw() )
   {
      return DRAW;
   }

   SCORE score;

   if ( board.BlackCanMove() )
   {
      score = CommonMidgameEval ( board );

      if ( board.black_in_check )
      {
         score += CHECK_BONUS;
      }
   }
   else
   {
      if ( board.black_in_check )
      {
         return WHITE_WINS - WIN_POSTPONEMENT(depth);
      }
      else
      {
         return DRAW;
      }
   }

   // This is a quiescence nudge for ending up with a tempo.
   score -= TEMPO_BONUS;

   if ( score < 0 )
   {
      score += depth;
   }
   else if ( score > 0 )
   {
      score -= depth;
   }


   return score;
}


SCORE ComputerChessPlayer::WhitePawnBonus ( ChessBoard &b,
                        int         ofs,
                        int         x,
                        int         ybase )
{
   SCORE score = 0;

   if ( x == 0 || x == 7 )
   {
      score -= PAWN_SIDE_FILE;
   }

   // Look for pawns in front of us only.
   // We don't look for pawns behind us, because they would have
   // found us already.
   // Also, we stop as soon as we find a pawn in front of us,
   // because we will let that pawn find any ones ahead of it.
   // This is not just for efficiency; it changes the way things
   // are scored.

   int z;
   cBOOLEAN isPassedPawn = cTRUE;
   const SQUARE *board = b.board;

   for ( z = ofs + NORTH; (board[z] & OFFBOARD) == 0; z += NORTH )
   {
      if ( board[z] & WP_MASK )
      {
         isPassedPawn = cFALSE;
         score -= PAWN_DOUBLED;
         break;
      }

      if ( board[z] & BP_MASK )
      {
         isPassedPawn = cFALSE;
         break;
      }

      if ( (board[z+EAST] & BP_MASK) || (board[z+WEST] & BP_MASK) )
      {
         isPassedPawn = cFALSE;
      }
   }

   #if PAWN_SPLIT
   cBOOLEAN isSplitPawn = cTRUE;
   for ( z = OFFSET(XPART(ofs),3); z < OFFSET(2,9); z += NORTH )
   {
      if ( board[z+EAST] & WP_MASK || board[z+WEST] & WP_MASK )
      {
         isSplitPawn = cFALSE;
         break;
      }
   }

   if ( isSplitPawn )
   {
      score -= PAWN_SPLIT;
   }
   #endif  // PAWN_SPLIT

   if ( board [ofs + SOUTHWEST] & WP_MASK )
   {
      if ( board [ofs + SOUTHEAST] & WP_MASK )
      {
         score += isPassedPawn ? PASSED_PAWN_PROTECT2 : PAWN_PROTECT2;
      }
      else
      {
         score += isPassedPawn ? PASSED_PAWN_PROTECT1 : PAWN_PROTECT1;
      }
   }
   else if ( board [ofs + SOUTHEAST] & WP_MASK )
   {
      score += isPassedPawn ? PASSED_PAWN_PROTECT1 : PAWN_PROTECT1;
   }
   else if ( isPassedPawn )
   {
      score += PASSED_PAWN_ALONE;
   }

   if ( (board [ofs + NORTHEAST] & WB_MASK) ||
        (board [ofs + NORTHWEST] & WB_MASK) )
   {
      score += BISHOP_PROTECT_PAWN;
   }

   if ( (board[ofs+NORTHEAST] & (BN_MASK|BB_MASK|BR_MASK|BQ_MASK|BK_MASK)) &&
        (board[ofs+NORTHWEST] & (BN_MASK|BB_MASK|BR_MASK|BQ_MASK|BK_MASK)) )
   {
      score += PAWN_FORK;
   }

   if ( isPassedPawn )
   {
      switch ( ybase )
      {
         case OFFSET(2,6):  score += PASSED_3_FROM_PROM;   break;
         case OFFSET(2,7):  score += PASSED_2_FROM_PROM;   break;
         case OFFSET(2,8):  score += PASSED_1_FROM_PROM;   break;
      }
   }
   else
   {
      if ( ybase == OFFSET(2,7) )
      {
         score += BLOCKED_2_FROM_PROM;
      }
   }

   return score;
}


SCORE ComputerChessPlayer::BlackPawnBonus ( ChessBoard &b,
                        int         ofs,
                        int         x,
                        int         ybase )
{
   SCORE score = 0;

   if ( x == 0 || x == 7 )
   {
      score -= PAWN_SIDE_FILE;
   }

   // Look for pawns in front of us only.
   // We don't look for pawns behind us, because they would have
   // found us already.
   // Also, we stop as soon as we find a pawn in front of us,
   // because we will let that pawn find any ones ahead of it.
   // This is not just for efficiency; it changes the way things
   // are scored.

   int z;
   cBOOLEAN isPassedPawn = cTRUE;
   const SQUARE *board = b.board;

   for ( z = ofs + SOUTH; (board[z] & OFFBOARD) == 0; z += SOUTH )
   {
      if ( board[z] & BP_MASK )
      {
         isPassedPawn = cFALSE;
         score -= PAWN_DOUBLED;
         break;
      }

      if ( board[z] & WP_MASK )
      {
         isPassedPawn = cFALSE;
         break;
      }

      if ( (board[z+EAST] & WP_MASK) || (board[z+WEST] & WP_MASK) )
      {
         isPassedPawn = cFALSE;
      }
   }

   #if PAWN_SPLIT
   cBOOLEAN isSplitPawn = cTRUE;
   for ( z = OFFSET(XPART(ofs),8); z > OFFSET(9,2); z += SOUTH )
   {
      if ( (board[z+EAST] & BP_MASK) || (board[z+WEST] & BP_MASK) )
      {
         isSplitPawn = cFALSE;
         break;
      }
   }

   if ( isSplitPawn )
   {
      score -= PAWN_SPLIT;
   }
   #endif // PAWN_SPLIT

   if ( board [ofs + NORTHWEST] & BP_MASK )
   {
      if ( board [ofs + NORTHEAST] & BP_MASK )
      {
         score += isPassedPawn ? PASSED_PAWN_PROTECT2 : PAWN_PROTECT2;
      }
      else
      {
         score += isPassedPawn ? PASSED_PAWN_PROTECT1 : PAWN_PROTECT1;
      }
   }
   else if ( board [ofs + NORTHEAST] & BP_MASK )
   {
      score += isPassedPawn ? PASSED_PAWN_PROTECT1 : PAWN_PROTECT1;
   }
   else if ( isPassedPawn )
   {
      score += PASSED_PAWN_ALONE;
   }

   if ( (board [ofs + SOUTHEAST] & BB_MASK) ||
        (board [ofs + SOUTHWEST] & BB_MASK) )
   {
      score += BISHOP_PROTECT_PAWN;
   }

   if ( (board[ofs+SOUTHEAST] & (WN_MASK|WB_MASK|WR_MASK|WQ_MASK|WK_MASK)) &&
        (board[ofs+SOUTHWEST] & (WN_MASK|WB_MASK|WR_MASK|WQ_MASK|WK_MASK)) )
   {
      score += PAWN_FORK;
   }

   if ( isPassedPawn )
   {
      switch ( ybase )
      {
         case OFFSET(2,5):  score += PASSED_3_FROM_PROM;   break;
         case OFFSET(2,4):  score += PASSED_2_FROM_PROM;   break;
         case OFFSET(2,3):  score += PASSED_1_FROM_PROM;   break;
      }
   }
   else
   {
      if ( ybase == OFFSET(2,3) )
      {
         score += BLOCKED_2_FROM_PROM;
      }
   }

   return score;
}


/*--- end of file eval.cpp ---*/