/*===========================================================================

    search.cpp  -  Don Cross, May 1993.

    Contains min-max search algorithms with alpha-beta pruning for chess.

    Revision history:

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1993 October 16 [Don Cross]
     Fixed bug where I had a MakeBlackMove() paired with
     UnmakeWhiteMove().

1993 October 18 [Don Cross]
     Added the 'searchBias' stuff.  This is an attempt to allow
     some randomization of move selection.

1993 October 23 [Don Cross]
     Implemented timed search.
     Added check for draw by repetition of moves:
     {W+, B+, W-, B-} plies repeated 3 times.

1993 October 24 [Don Cross]
     Added call to ChessBoard::IsDefiniteDraw() to WhiteSearch() and
     BlackSearch() to make sure that draws by repetition are found
     anywhere in the search.

     Created a MoveList::AddMove(Move) function, and changed the
     pg.AddMove() line of code in the searchBias conditional to
     use it instead.  The old way was clobbering the bestmove.score,
     so searches were continuing even after a forced win was found.

1993 October 25 [Don Cross]
     Making the ComputerChessPlayer::CheckTimeLimit() member function
     adaptive to the processor speed.  It does this by increasing
     the timeCheckLimit until successive calls to ChessTime() always
     report a difference of at least 0.1 seconds.

1994 January 15 [Don Cross]
     Added ComputerChessPlayer::AbortSearch().
     This required a little reworking of logic in the CheckTimeLimit
     member function too.

1994 January 22 [Don Cross]
     Adding support for variable eval functions.

===========================================================================*/

#include "chess.h"


ComputerChessPlayer::ComputerChessPlayer ( ChessUI &ui ):
   ChessPlayer ( ui ),
   maxlevel ( 3 ),
   generated ( 0 ),
   visited ( 0 ),
   evaluated ( 0 ),
   searchBias ( 0 ),
   timedSearch ( cFALSE ),
   timeCheckLimit ( 100 ),
   whiteEval ( &ComputerChessPlayer::WhiteMidgameEval ),
   blackEval ( &ComputerChessPlayer::BlackMidgameEval )
{
   rootml.num = 0;
}


ComputerChessPlayer::~ComputerChessPlayer()
{
}


void ComputerChessPlayer::ChooseEvalFunctions ( ChessBoard &board )
{
   INT16 *inv = board.inventory;

   // If either side has a lone king, and the other
   // has rook(s) and/or queen(s) use endgame eval #1.

   if ( inv [WP_INDEX] == 0 &&
        inv [WN_INDEX] == 0 &&
        inv [WB_INDEX] == 0 &&
        inv [WR_INDEX] == 0 &&
        inv [WQ_INDEX] == 0 &&
        inv [BR_INDEX] + inv [BQ_INDEX] > 0 )
   {
      whiteEval = &ComputerChessPlayer::EndgameEval1;
      blackEval = &ComputerChessPlayer::EndgameEval1;
   }
   else if ( inv [BP_INDEX] == 0 &&
             inv [BN_INDEX] == 0 &&
             inv [BB_INDEX] == 0 &&
             inv [BR_INDEX] == 0 &&
             inv [BQ_INDEX] == 0 &&
             inv [WR_INDEX] + inv [WQ_INDEX] > 0 )
   {
      whiteEval = &ComputerChessPlayer::EndgameEval1;
      blackEval = &ComputerChessPlayer::EndgameEval1;
   }
   else
   {
      // Use the default midgame evaluator...

      whiteEval = &ComputerChessPlayer::WhiteMidgameEval;
      blackEval = &ComputerChessPlayer::BlackMidgameEval;
   }
}


cBOOLEAN ComputerChessPlayer::GetMove ( ChessBoard  &board,
                                        Move        &bestmove )
{
   searchAborted = cFALSE;
   if ( timedSearch )
   {
      stopTime = ChessTime() + timeLimit;
      timeCheckCounter = 0;
      prevTime = ChessTime();
   }

   ChooseEvalFunctions ( board );

   INT32 timeBefore = ChessTime();
   if ( board.WhiteToMove() )
   {
      GetWhiteMove ( board, bestmove );
   }
   else
   {
      GetBlackMove ( board, bestmove );
   }
   INT32 timeAfter = ChessTime();

   userInterface.DisplayMove ( board, bestmove );
   userInterface.ReportComputerStats ( timeAfter - timeBefore,
                                       visited,
                                       evaluated,
                                       generated,
                                       level+1,
                                       visnodes,
                                       gennodes );

   if ( board.WhiteToMove() && bestmove.score <= WHITE_RESIGN_THRESHOLD )
   {
      return cFALSE;   // resign
   }
   else if ( board.BlackToMove() && bestmove.score >= BLACK_RESIGN_THRESHOLD )
   {
      return cFALSE;   // resign
   }
   return cTRUE;   // keep on playing!
}


void ComputerChessPlayer::GetWhiteMove ( ChessBoard &board, Move &bestmove )
{
   board.GenWhiteMoves ( rootml, this );
   bestmove = rootml.m[0];   // just in case!

   gennodes[0] = generated = rootml.num;
   visited = evaluated = 0;

   for ( int i=0; i < NODES_ARRAY_SIZE; i++ )
   {
      visnodes[i] = gennodes[i] = 0;
   }

   if ( rootml.num == 1 )
   {
      // We have only one legal move...
      // no point thinking too hard about it!

      return;
   }

   if ( WhiteOpening ( board, bestmove ) )
   {
      return;   // We found a groovy opening move
   }

   for ( level=0; !searchAborted && level <= maxlevel; level++ )
   {
      Move thisBestmove;

      WhiteSearchRoot ( board, thisBestmove );

      if ( !searchAborted )
      {
         bestmove = thisBestmove;

         if ( searchBias )
         {
            MoveList pg;   // list of "pretty good" moves

            pg.num = 0;

            for ( int i=0; i < rootml.num; i++ )
            {
               if ( rootml.m[i].score > bestmove.score - searchBias )
               {
                  pg.AddMove ( rootml.m[i] );
               }
            }

            int r = ChessRandom ( pg.num );
            bestmove = pg.m[r];
         }

         if ( bestmove.score >= WON_FOR_WHITE )
         {
            break;
         }
      }
   }

   if ( bestmove.score >= WON_FOR_WHITE &&
        bestmove.score < WHITE_WINS - WIN_DELAY_PENALTY )
   {
      // If we get here, it means we predict a forced checkmate
      // for White!  We can calculate how many moves it will take
      // using the score.

      int plies = (WHITE_WINS - long(bestmove.score)) / WIN_DELAY_PENALTY;
      userInterface.PredictMate ( plies / 2 );
   }
}


void ComputerChessPlayer::GetBlackMove ( ChessBoard &board, Move &bestmove )
{
   board.GenBlackMoves ( rootml, this ); // Generate sorted list of legal moves
   bestmove = rootml.m[0];   // just in case!
   gennodes[0] = generated = rootml.num;

   visited = evaluated = 0;

   for ( int i=0; i < NODES_ARRAY_SIZE; i++ )
   {
      visnodes[i] = gennodes[i] = 0;
   }

   if ( rootml.num == 1 )
   {
      // We have only one legal move...
      // no point thinking too hard about it!

      return;
   }

   if ( BlackOpening ( board, bestmove ) )
   {
      return;   // We found a groovy opening move
   }

   for ( level=0; !searchAborted && level <= maxlevel; level++ )
   {
      Move thisBestmove;

      BlackSearchRoot ( board, thisBestmove );

      if ( !searchAborted )
      {
         bestmove = thisBestmove;

         if ( searchBias )
         {
            MoveList pg;   // list of "pretty good" moves

            pg.num = 0;

            for ( int i=0; i < rootml.num; i++ )
            {
               if ( rootml.m[i].score < bestmove.score + searchBias )
               {
                  pg.AddMove ( rootml.m[i] );
               }
            }

            int r = ChessRandom ( pg.num );
            bestmove = pg.m[r];
         }

         if ( bestmove.score <= WON_FOR_BLACK )
         {
            break;
         }
      }
   }

   if ( bestmove.score <= WON_FOR_BLACK &&
        bestmove.score > BLACK_WINS + WIN_DELAY_PENALTY )
   {
      // If we get here, it means we predict a forced checkmate
      // for Black!  We can calculate how many moves it will take
      // using the score.

      int plies = (long(bestmove.score) - BLACK_WINS) / WIN_DELAY_PENALTY;
      userInterface.PredictMate ( plies / 2 );
   }
}


int ComputerChessPlayer::WhiteSearchRoot ( ChessBoard &board,
                                           Move       &bestmove,
                                           SCORE      alpha,
                                           SCORE      beta )
{
   SCORE        score;
   SCORE        bestscore = NEGINF;
   int          i;
   Move        *move;
   UnmoveInfo   unmove;

   for ( i=0, move=&rootml.m[0];
         !searchAborted && i < rootml.num;
         i++, move++ )
   {
      // If this isn't the first move we are considering, and it
      // is a definite loser, just skip it!

      if ( i>0 && move->score <= WON_FOR_BLACK )
      {
         continue;
      }

      ++visited;
      ++visnodes[0];

      if ( level > 1 )
      {
         userInterface.DisplayCurrentMove ( board, *move, level );
      }

      board.MakeWhiteMove ( *move, unmove, cFALSE, cFALSE );
      if ( level > 0 )
      {
         score = BlackSearch ( board, 1, alpha, beta );
         if ( !searchAborted )
         {
            move->score = score;
         }
      }
      else
      {
         score = move->score = BlackQSearch ( board, 1, alpha, beta );
      }
      board.UnmakeWhiteMove ( *move, unmove );

      if ( score > bestscore )
      {
         bestmove = *move;
         bestscore = score;
         if ( level > 1 )
         {
            userInterface.DisplayBestMoveSoFar ( board, bestmove, level );
         }
      }

      if ( score > alpha )
      {
         alpha = score;
      }
   }

   rootml.WhiteSort();

   return bestscore;
}


int ComputerChessPlayer::BlackSearchRoot ( ChessBoard &board,
                                           Move       &bestmove,
                                           SCORE      alpha,
                                           SCORE      beta )
{
   SCORE        score;
   SCORE        bestscore = POSINF;
   int          i;
   Move        *move;
   UnmoveInfo   unmove;

   for ( i=0, move=&rootml.m[0];
         !searchAborted && i < rootml.num;
         i++, move++ )
   {
      // If this isn't the first move we are considering, and it
      // is a definite loser, just skip it!

      if ( i>0 && move->score >= WON_FOR_WHITE )
      {
         continue;
      }

      ++visited;
      ++visnodes[0];

      if ( level > 1 )
      {
         userInterface.DisplayCurrentMove ( board, *move, level );
      }

      board.MakeBlackMove ( *move, unmove, cFALSE, cFALSE );
      if ( level > 0 )
      {
         score = WhiteSearch ( board, 1, alpha, beta );
         if ( !searchAborted )
         {
            move->score = score;
         }
      }
      else
      {
         score = move->score = WhiteQSearch ( board, 1, alpha, beta );
      }
      board.UnmakeBlackMove ( *move, unmove );

      if ( score < bestscore )
      {
         bestmove = *move;
         bestscore = score;
         if ( level > 1 )
         {
            userInterface.DisplayBestMoveSoFar ( board, bestmove, level );
         }
      }

      if ( score < beta )
      {
         beta = score;
      }
   }

   rootml.BlackSort();

   return bestscore;
}


SCORE ComputerChessPlayer::WhiteSearch ( ChessBoard  &board,
                                         int          depth,
                                         SCORE        alpha,
                                         SCORE        beta )
{
   if ( CheckTimeLimit() )
   {
      return 0;
   }

   int          score;
   int          bestscore = NEGINF;
   int          i;
   MoveList     ml;
   UnmoveInfo   unmove;
   Move        *move;

   board.GenWhiteMoves ( ml, this );
   generated += ml.num;
   if ( depth < NODES_ARRAY_SIZE )
   {
      gennodes[depth] += ml.num;
   }

   if ( ml.num == 0 )
   {
      // This is the end of the game!
      return board.white_in_check ?
             (BLACK_WINS + WIN_POSTPONEMENT(depth)) :
             DRAW;
   }
   else if ( board.IsDefiniteDraw() )
   {
      return DRAW;
   }

   for ( i=0, move=&ml.m[0]; i < ml.num; i++, move++ )
   {
      ++visited;
      if ( depth < NODES_ARRAY_SIZE )
      {
         ++visnodes[depth];
      }

      board.MakeWhiteMove ( *move, unmove, cFALSE, cFALSE );

      if ( depth < level )
      {
         score = BlackSearch ( board, depth+1, alpha, beta );
      }
      else
      {
         score = BlackQSearch ( board, depth+1, alpha, beta );
      }

      board.UnmakeWhiteMove ( *move, unmove );

      if ( score > bestscore )
      {
         bestscore = score;
      }

      if ( score >= beta + searchBias )
      {
         break;
      }

      if ( score > alpha )
      {
         alpha = score;
      }
   }

   return bestscore;
}


SCORE ComputerChessPlayer::BlackSearch ( ChessBoard  &board,
                                         int          depth,
                                         SCORE        alpha,
                                         SCORE        beta )
{
   if ( CheckTimeLimit() )
   {
      return 0;
   }

   int          score;
   int          bestscore = POSINF;
   int          i;
   MoveList     ml;
   UnmoveInfo   unmove;
   Move        *move;

   board.GenBlackMoves ( ml, this );
   generated += ml.num;
   if ( depth < NODES_ARRAY_SIZE )
   {
      gennodes[depth] += ml.num;
   }

   if ( ml.num == 0 )
   {
      // This is the end of the game!
      return board.black_in_check ?
             (WHITE_WINS - WIN_POSTPONEMENT(depth)) :
             DRAW;
   }
   else if ( board.IsDefiniteDraw() )
   {
      return DRAW;
   }

   for ( i=0, move=&ml.m[0]; i < ml.num; i++, move++ )
   {
      ++visited;
      if ( depth < NODES_ARRAY_SIZE )
      {
         ++visnodes[depth];
      }

      board.MakeBlackMove ( *move, unmove, cFALSE, cFALSE );

      if ( depth < level )
      {
         score = WhiteSearch ( board, depth+1, alpha, beta );
      }
      else
      {
         score = WhiteQSearch ( board, depth+1, alpha, beta );
      }

      board.UnmakeBlackMove ( *move, unmove );

      if ( score < bestscore )
      {
         bestscore = score;
      }

      if ( score <= alpha - searchBias )
      {
         break;
      }

      if ( score < beta )
      {
         beta = score;
      }
   }

   return bestscore;
}


SCORE ComputerChessPlayer::WhiteQSearch ( ChessBoard  &board,
                                          int          depth,
                                          SCORE        alpha,
                                          SCORE        beta )
{
   if ( CheckTimeLimit() )
   {
      return 0;
   }

   SCORE       score;
   SCORE       bestscore = (this->*whiteEval) ( board, depth );
   MoveList    ml;
   UnmoveInfo  unmove;
   int         i;
   Move       *move;

   if ( bestscore < beta + searchBias )
   {
      board.GenWhiteCaptures ( ml, this );
      if ( depth < NODES_ARRAY_SIZE )
      {
         gennodes[depth] += ml.num;
      }

      for ( i=0, move=&ml.m[0]; i < ml.num; i++, move++ )
      {
         ++visited;
         if ( depth < NODES_ARRAY_SIZE )
         {
            ++visnodes[depth];
         }

         board.MakeWhiteMove ( *move, unmove, cFALSE, cFALSE );
         score = BlackQSearch ( board, depth+1, alpha, beta );
         board.UnmakeWhiteMove ( *move, unmove );

         if ( score > bestscore )
         {
            bestscore = score;
         }

         if ( score >= beta + searchBias )
         {
            break;
         }

         if ( score > alpha )
         {
            alpha = score;
         }
      }
   }

   return bestscore;
}


SCORE ComputerChessPlayer::BlackQSearch ( ChessBoard  &board,
                                          int          depth,
                                          SCORE        alpha,
                                          SCORE        beta )
{
   if ( CheckTimeLimit() )
   {
      return 0;
   }

   SCORE       score;
   SCORE       bestscore = (this->*blackEval) ( board, depth );
   MoveList    ml;
   UnmoveInfo  unmove;
   int         i;
   Move       *move;

   if ( bestscore > alpha - searchBias )
   {
      board.GenBlackCaptures ( ml, this );
      if ( depth < NODES_ARRAY_SIZE )
      {
         gennodes[depth] += ml.num;
      }

      for ( i=0, move=&ml.m[0]; i < ml.num; i++, move++ )
      {
         ++visited;
         if ( depth < NODES_ARRAY_SIZE )
         {
            ++visnodes[depth];
         }

         board.MakeBlackMove ( *move, unmove, cFALSE, cFALSE );
         score = WhiteQSearch ( board, depth+1, alpha, beta );
         board.UnmakeBlackMove ( *move, unmove );

         if ( score < bestscore )
         {
            bestscore = score;
         }

         if ( score <= alpha - searchBias )
         {
            break;
         }

         if ( score < beta )
         {
            beta = score;
         }
      }
   }

   return bestscore;
}


void ComputerChessPlayer::SetSearchDepth ( int NewSearchDepth )
{
   maxlevel = NewSearchDepth;
   timedSearch = cFALSE;
   searchAborted = cFALSE;
}


void ComputerChessPlayer::SetTimeLimit ( INT32 hundredthsOfSeconds )
{
   if ( hundredthsOfSeconds < 100 )
   {
      ChessFatal ( "Invalid search time for ComputerChessPlayer" );
   }

   maxlevel = NODES_ARRAY_SIZE;
   timedSearch = cTRUE;
   searchAborted = cFALSE;
   timeLimit = hundredthsOfSeconds;
}


void ComputerChessPlayer::AbortSearch()
{
   searchAborted = cTRUE;
}


cBOOLEAN ComputerChessPlayer::CheckTimeLimit()
{
   if ( searchAborted )
   {
      return cTRUE;
   }

   if ( timedSearch )
   {
      if ( ++timeCheckCounter >= timeCheckLimit )
      {
         timeCheckCounter = 0;

         INT32 now = ChessTime();

         if ( now >= stopTime )
         {
            if ( now - prevTime > 100 )
            {
               timeCheckLimit /= 2;
            }

            return searchAborted = cTRUE;
         }
         else if ( now - prevTime < 10 )   // less than 0.01 sec elapsed?
         {
            timeCheckLimit += 100;
         }

         prevTime = now;
      }
   }

   return cFALSE;
}


void ComputerChessPlayer::SetSearchBias ( SCORE newSearchBias )
{
   if ( newSearchBias >= 0 )
   {
      searchBias = newSearchBias;
   }
   else
   {
      ChessFatal ( "Negative search bias not allowed" );
   }
}


/*--- end of file search.cpp ---*/
