/*========================================================================

     opening.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Contains code for specially handling chess openings.
     These functions are called only by the search code,
     and only after the 'rootml' MoveList contained in
     class ComputerChessPlayer is filled out with a valid
     list of legal moves.

     ComputerChessPlayer::WhiteOpening() and
     ComputerChessPlayer::BlackOpening() return cTRUE when
     they have a definite opinion about what move to make.
     Otherwise, they return cFALSE, meaning that they don't
     know what to do in the current situation.

     Revision history:

1993 October 19 [Don Cross]
     Started writing.  The first version just knows about P-K4, P-K4
     kind of stuff.

1994 February 15 [Don Cross]
     Implementing true opening library using OBT (Open Book Translator).

========================================================================*/

#include "chess.h"

typedef unsigned long big;

extern unsigned long OpeningData[];

#define  BAD_MOVE_BIT     0x10000L
#define  BAD_MOVE_SCORE   (-100)


void Move::Fix ( unsigned long openBookData, ChessBoard &board )
{
   source = BYTE ( (openBookData >> 8) & 0x7F );
   dest   = BYTE ( openBookData & 0x7F );

   SQUARE piece = board.GetSquareContents ( source );

   if ( piece & (WK_MASK|BK_MASK) )
   {
      if ( dest - source == 2*EAST )
      {
         dest = SPECIAL_MOVE_KCASTLE;
      }
      else if ( dest - source == 2*WEST )
      {
         dest = SPECIAL_MOVE_QCASTLE;
      }
   }
   else if ( piece & WP_MASK )
   {
      if ( YPART(dest) == 9 )
      {
         // assume promotion to Queen

         switch ( dest - source )
         {
            case NORTH:      dest = SPECIAL_MOVE_PROMOTE_NORM;       break;
            case NORTHEAST:  dest = SPECIAL_MOVE_PROMOTE_CAP_EAST;   break;
            case NORTHWEST:  dest = SPECIAL_MOVE_PROMOTE_CAP_WEST;   break;
         }

         dest |= Q_INDEX;
      }
      else if ( (dest - source) != NORTH &&
                (dest - source) != 2*NORTH &&
                board.GetSquareContents(dest) == EMPTY )
      {
         // assume e.p. capture

         if ( dest - source == NORTHEAST )
         {
            dest = SPECIAL_MOVE_EP_EAST;
         }
         else
         {
            dest = SPECIAL_MOVE_EP_WEST;
         }
      }
   }
   else if ( piece & BP_MASK )
   {
      if ( YPART(dest) == 2 )
      {
         // assume promotion to Queen

         switch ( dest - source )
         {
            case SOUTH:      dest = SPECIAL_MOVE_PROMOTE_NORM;       break;
            case SOUTHEAST:  dest = SPECIAL_MOVE_PROMOTE_CAP_EAST;   break;
            case SOUTHWEST:  dest = SPECIAL_MOVE_PROMOTE_CAP_WEST;   break;
         }

         dest |= Q_INDEX;
      }
      else if ( (dest - source) != SOUTH &&
                (dest - source) != 2*SOUTH &&
                board.GetSquareContents(dest) == EMPTY )
      {
         // assume e.p. capture

         if ( dest - source == SOUTHEAST )
         {
            dest = SPECIAL_MOVE_EP_EAST;
         }
         else
         {
            dest = SPECIAL_MOVE_EP_WEST;
         }
      }
   }

   score  = (openBookData & BAD_MOVE_BIT) ? BAD_MOVE_SCORE : 0;
}


cBOOLEAN OB_FindMove ( Move move, big &oindex, ChessBoard &board )
{
   // Start with oindex given, and try to find this move.
   // If found, return cTRUE and set oindex to new branchlist offset.
   // Otherwise, return cFALSE.

   big numBranches = OpeningData [oindex];
   big bcount;
   big bindex = oindex + 1;

   for ( bcount=0; bcount < numBranches; bcount++ )
   {
      big branchSize = OpeningData [bindex];
      big moveData = OpeningData [bindex + 1];

      Move bookMove;
      bookMove.Fix ( moveData, board );

      if ( move == bookMove )
      {
         oindex = bindex + 2;
         return cTRUE;
      }

      bindex += branchSize;
   }

   return cFALSE;   // could not find move
}


cBOOLEAN OB_FindContinuation ( ChessBoard &board, big &oindex )
{
   // Start from beginning of game and try to find the continuation
   // that has been played so far in the opening book data.

   int currentPly = board.GetCurrentPlyNumber();
   int plyIndex;
   oindex = 0;

   static ChessBoard localBoard;
   localBoard.Init();

   for ( plyIndex=0; plyIndex < currentPly; plyIndex++ )
   {
      Move move = board.GetPastMove ( plyIndex );

      if ( !OB_FindMove ( move, oindex, localBoard ) )
      {
         return cFALSE;  // out of opening book
      }

      // Make the move in the local board...

      UnmoveInfo unmove;

      if ( localBoard.WhiteToMove() )
      {
         localBoard.MakeWhiteMove ( move, unmove, cTRUE, cTRUE );
      }
      else
      {
         localBoard.MakeBlackMove ( move, unmove, cTRUE, cTRUE );
      }
   }

   return cTRUE;
}


cBOOLEAN OB_PickMove ( big oindex, Move &move, ChessBoard &board )
{
   MoveList ml;
   ml.num = 0;

   big numBranches = OpeningData [oindex];
   big bcount;
   big branchIndex = oindex + 1;

   for ( bcount = 0; bcount < numBranches; bcount++ )
   {
      big branchSize = OpeningData [branchIndex];
      big moveData   = OpeningData [branchIndex + 1];

      if ( !(moveData & BAD_MOVE_BIT) )
      {
         Move bookMove;
         bookMove.Fix ( moveData, board );
         ml.AddMove ( bookMove );
      }

      branchIndex += branchSize;
   }

   if ( ml.num == 0 )
   {
      return cFALSE;
   }

   int r = ChessRandom ( ml.num );
   move = ml.m[r];

   return cTRUE;
}


cBOOLEAN ComputerChessPlayer::WhiteOpening ( ChessBoard &board,
                                             Move       &bestmove )
{
   if ( !openingBookSearchEnabled )
   {
      return cFALSE;
   }

   if ( board.HasBeenEdited() )
   {
      return cFALSE;
   }

   cBOOLEAN foundMove = cFALSE;

   big nodeIndex;
   if ( OB_FindContinuation ( board, nodeIndex ) )
   {
      if ( OB_PickMove ( nodeIndex, bestmove, board ) )
      {
         foundMove = cTRUE;
      }
   }

   if ( foundMove )
   {
      // Sanity check!
      if ( !rootml.IsLegal ( bestmove ) )
      {
         foundMove = cFALSE;
      }

      bestmove.score = 0;
   }

   return foundMove;
}


cBOOLEAN ComputerChessPlayer::BlackOpening ( ChessBoard &board,
                                             Move       &bestmove )
{
   if ( !openingBookSearchEnabled )
   {
      return cFALSE;
   }

   if ( board.HasBeenEdited() )
   {
      return cFALSE;
   }

   cBOOLEAN foundMove = cFALSE;

   big nodeIndex;
   if ( OB_FindContinuation ( board, nodeIndex ) )
   {
      if ( OB_PickMove ( nodeIndex, bestmove, board ) )
      {
         foundMove = cTRUE;
      }
   }

   if ( foundMove )
   {
      // Sanity check!
      if ( !rootml.IsLegal ( bestmove ) )
      {
         foundMove = cFALSE;
      }

      bestmove.score = 0;
   }

   return foundMove;
}


/*--- end of file opening.cpp ---*/
