/*=========================================================================

     os2pm.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Contains main() for a OS/2 Presentation Manager
     version of NewChess.

     Revision history:

1993 December 31 [Don Cross]:
     Fixed bug: Title bar text used to say it was White's turn
     when we restored a game from a file and it was either side's
     turn to move.

1994 January 12 [Don Cross]:
     Started adding application menu for Game, Edit, View.
     Here are the menu commands I intend to support:

     Game:
        New:   start over at the beginning of the game.
        Open:  read a game from a file and continue it.
        Save, Save As:  save the game to a file.

     Edit:  (most/all options will be disabled if not human's turn)
        Clear board:  make the board blank???
        Set up pieces:  go into "edit mode"???

1994 January 31 [Don Cross]
     Fixed RestoreBoardFromFile so that it calls
     ChessBoard::SaveSpecialMove when it finds an edit command
     in the game file.

1994 February 4 [Don Cross]
     Fixed bug with editing board when viewed from Black's side.
     Fixed bug with Empty side not disabling pieces in edit dialog.

1994 February 9 [Don Cross]
     Adding visual feedback for selecting squares.

=========================================================================*/

#define INCL_PM
#define INCL_DOSPROCESS
#include <os2.h>
#include <string.h>
#include <stdio.h>
#include <process.h>

#include "chess.h"
#include "os2chess.h"
#include "os2pmch.h"

cBOOLEAN Global_ResetGameFlag  = cFALSE;
cBOOLEAN Global_ClearBoardFlag = cFALSE;
cBOOLEAN Global_ViewDebugFlag  = cFALSE;

BoardEditRequest Global_BoardEditRequest;

char TitleBarText_WhitesTurn[] = CHESS_PROGRAM_NAME " - White's turn";
char TitleBarText_BlacksTurn[] = CHESS_PROGRAM_NAME " - Black's turn";
char TitleBarText_GameOver[]   = CHESS_PROGRAM_NAME " - game over";

char GameSave_filename[] = "chess.gam";

void ChessThreadFunc ( void * );

struct ChessThreadParms
{
   HWND  boardClientHwnd;
   HWND  boardFrameHwnd;
   HAB   hab;

   ChessThreadParms ( HWND _clientHwnd, HWND _frameHwnd, HAB _hab ):
      boardClientHwnd ( _clientHwnd ),
      boardFrameHwnd ( _frameHwnd ),
      hab ( _hab )
   {
   }
};

MRESULT EXPENTRY _export ChessWndProc ( HWND, ULONG, MPARAM, MPARAM );

static HAB myHab;
static cBOOLEAN globalAbortFlag = cFALSE;
static cBOOLEAN globalImmedQuitFlag = cFALSE;
static int SoundEnableFlag = 1;
HWND   global_hwndClient;
HWND   global_FrameHwnd;

DefPlayerInfo DefPlayer;


static void EnableEditPieces ( HWND hwnd, cBOOLEAN enable )
{
   WinEnableControl ( hwnd, EDIT_PAWN,   enable );
   WinEnableControl ( hwnd, EDIT_KNIGHT, enable );
   WinEnableControl ( hwnd, EDIT_BISHOP, enable );
   WinEnableControl ( hwnd, EDIT_ROOK,   enable );
   WinEnableControl ( hwnd, EDIT_QUEEN,  enable );
   WinEnableControl ( hwnd, EDIT_KING,   enable );
}


MRESULT EXPENTRY _export EditSquare_DlgProc
                          ( HWND     hwnd,
                            ULONG    msg,
                            MPARAM   mp1,
                            MPARAM   mp2 )
{
   static SQUARE piece = EMPTY;
   static SQUARE side = EMPTY;
   MRESULT result = FALSE;
   static SQUARE *squarePtr = 0;
   static SHORT lastPiece = EDIT_PAWN;
   static SHORT lastSide = EDIT_EMPTY;

   switch ( msg )
   {
      case WM_INITDLG:
      {
         squarePtr = (SQUARE *) ( mp2 );

         WinSendDlgItemMsg ( hwnd, lastPiece, BM_SETCHECK,
                             MPFROM2SHORT(1,0), 0 );

         WinSendDlgItemMsg ( hwnd, lastSide, BM_SETCHECK,
                             MPFROM2SHORT(1,0), 0 );

         EnableEditPieces ( hwnd, lastSide != EDIT_EMPTY );
      }
      break;

      case WM_COMMAND:
      {
         switch ( COMMANDMSG(&msg)->cmd )
         {
            case DID_OK:
            {
               if ( squarePtr )
               {
                  if ( side == EMPTY )
                  {
                     *squarePtr = EMPTY;
                  }
                  else
                  {
                     *squarePtr = PieceLookupTable.sval [ piece | side ];
                  }
               }

               WinDismissDlg ( hwnd, DID_OK );
            }
            break;

            case DID_CANCEL:
            {
               WinDismissDlg ( hwnd, DID_CANCEL );
            }
            break;
         }
      }
      break;

      case WM_CONTROL:
      {
         SHORT b = SHORT1FROMMP(mp1);

         switch ( b )
         {
            case EDIT_PAWN:    piece = P_INDEX;   lastPiece = b;  break;
            case EDIT_KNIGHT:  piece = N_INDEX;   lastPiece = b;  break;
            case EDIT_BISHOP:  piece = B_INDEX;   lastPiece = b;  break;
            case EDIT_ROOK:    piece = R_INDEX;   lastPiece = b;  break;
            case EDIT_QUEEN:   piece = Q_INDEX;   lastPiece = b;  break;
            case EDIT_KING:    piece = K_INDEX;   lastPiece = b;  break;

            case EDIT_EMPTY:
            {
               side = EMPTY;
               lastSide = b;
               EnableEditPieces ( hwnd, FALSE );
            }
            break;

            case EDIT_WHITE:
            {
               side = WHITE_IND;
               lastSide = b;
               EnableEditPieces ( hwnd, TRUE );
            }
            break;

            case EDIT_BLACK:
            {
               side = BLACK_IND;
               lastSide = b;
               EnableEditPieces ( hwnd, TRUE );
            }
            break;
         }
      }
      break;

      default:
      {
         result = WinDefDlgProc ( hwnd, msg, mp1, mp2 );
      }
      break;
   }

   return result;
}


void AbortProgram ( HWND hwndClient, HAB hab )
{
   PERRINFO  pErrInfoBlk;
   PSZ       pszOffSet;
   PSZ       pszErrMsg;

   ChessBeep ( 100, 10 );
   ChessBeep ( 440, 110 );

   if ( (pErrInfoBlk = WinGetErrorInfo(hab)) != 0 )
   {
      pszOffSet = ((PSZ)pErrInfoBlk) + pErrInfoBlk->offaoffszMsg;
      pszErrMsg = ((PSZ)pErrInfoBlk) + *((PSHORT)pszOffSet);

      WinMessageBox ( HWND_DESKTOP,         // Parent window is desk top
                      HWND_DESKTOP,         // Owner window is desktop
                      (PSZ)pszErrMsg,       // PMWIN Error message
                      CHESS_PROGRAM_NAME,   // Title bar message
                      666,                  // Message identifier
                      MB_MOVEABLE | MB_CUACRITICAL | MB_CANCEL ); // Flags

      WinFreeErrorInfo ( pErrInfoBlk );
   }

   if ( hwndClient )
   {
      WinPostMsg ( hwndClient, WM_QUIT, (ULONG)0, (ULONG)0 );
   }
}


static cBOOLEAN EditSquareDialog ( HWND clientHwnd, SQUARE &square )
{
   ULONG result = WinDlgBox ( HWND_DESKTOP,
                              clientHwnd,
                              EditSquare_DlgProc,
                              0L,
                              IDD_EDIT_SQUARE,
                              MPARAM(&square) );

   cBOOLEAN choseSomething = cFALSE;

   switch ( result )
   {
      case DID_ERROR:
      {
         AbortProgram ( clientHwnd, myHab );
      }
      break;

      case DID_OK:
      {
         choseSomething = cTRUE;
      }
      break;
   }

   return choseSomething;
}


MRESULT EXPENTRY _export PromotePawn_DlgProc
                          ( HWND     hwnd,
                            ULONG    msg,
                            MPARAM   mp1,
                            MPARAM   mp2 )
{
   MRESULT result = FALSE;
   MRESULT rook_rb, bishop_rb, knight_rb;
   static SQUARE *promPtr;

   switch ( msg )
   {
      case WM_INITDLG:
      {
         // Turn on the 'queen' radio button...

         WinSendDlgItemMsg ( hwnd, RB_QUEEN, BM_SETCHECK,
                             MPFROM2SHORT(1,0), 0 );

         // Place to put promoted pawn data...
         promPtr = (SQUARE *) mp2;
      }
      break;

      case WM_COMMAND:
      {
         switch ( COMMANDMSG(&msg)->cmd )
         {
            case DID_OK:
            {
               rook_rb   = WinSendDlgItemMsg ( hwnd, RB_ROOK, BM_QUERYCHECK, 0, 0 );
               bishop_rb = WinSendDlgItemMsg ( hwnd, RB_BISHOP, BM_QUERYCHECK, 0, 0 );
               knight_rb = WinSendDlgItemMsg ( hwnd, RB_KNIGHT, BM_QUERYCHECK, 0, 0 );

               if ( rook_rb )
               {
                  *promPtr = R_INDEX;
               }
               else if ( bishop_rb )
               {
                  *promPtr = B_INDEX;
               }
               else if ( knight_rb )
               {
                  *promPtr = N_INDEX;
               }
               else // assume the queen radio button is selected
               {
                  *promPtr = Q_INDEX;
               }

               WinDismissDlg ( hwnd, DID_OK );
            }
            break;
         }
         break;
      }

      default:
      {
         result = WinDefDlgProc ( hwnd, msg, mp1, mp2 );
      }
      break;
   }

   return result;
}


MRESULT EXPENTRY _export DefineChessPlayers_DlgProc
                          ( HWND     hwnd,
                            ULONG    msg,
                            MPARAM   mp1,
                            MPARAM   mp2 )
{
   MRESULT result = FALSE;
   MRESULT query;

   switch ( msg )
   {
      case WM_INITDLG:
      {
         // Turn on the 'human' radio button for white player...
         WinSendDlgItemMsg ( hwnd, RB_WHUMAN, BM_SETCHECK,
                             MPFROM2SHORT(1,0), 0 );

         // Turn on the 'computer' radio button for black player...
         WinSendDlgItemMsg ( hwnd, RB_BCOMPUTER, BM_SETCHECK,
                             MPFROM2SHORT(1,0), 0 );

         // Turn on the 'enable sound' check box by default...
         WinSendDlgItemMsg ( hwnd, CB_ENABLE_SOUNDS, BM_SETCHECK,
                             MPFROM2SHORT(1,0), 0 );

         // Turn on the 'seconds' button in the computer search
         // for both players...
         WinSendDlgItemMsg ( hwnd, RB_W_IS_SECONDS, BM_SETCHECK,
                             MPFROM2SHORT(1,0), 0 );

         WinSendDlgItemMsg ( hwnd, RB_B_IS_SECONDS, BM_SETCHECK,
                             MPFROM2SHORT(1,0), 0 );
      }
      break;

      case WM_CONTROL:
      {
         switch ( SHORT1FROMMP(mp1) )
         {
            case RB_WHUMAN:
                 // Disable the white computer player's think time box
                 WinEnableControl ( hwnd, TB_WTIME, 0 );

                 // Disable 'seconds' and 'plies' radio buttons for search
                 WinEnableControl ( hwnd, RB_W_IS_SECONDS, 0 );
                 WinEnableControl ( hwnd, RB_W_IS_PLIES, 0 );

                 // Disable the white computer player's bias box
                 WinEnableControl ( hwnd, TB_WBIAS, 0 );
                 break;

            case RB_WCOMPUTER:
                 // Enable the white computer player's think time box
                 WinEnableControl ( hwnd, TB_WTIME, 1 );

                 // Enable 'seconds' and 'plies' radio buttons for search
                 WinEnableControl ( hwnd, RB_W_IS_SECONDS, 1 );
                 WinEnableControl ( hwnd, RB_W_IS_PLIES, 1 );

                 // Enable the white computer player's bias box
                 WinEnableControl ( hwnd, TB_WBIAS, 1 );
                 break;

            case RB_BHUMAN:
                 // Disable the black computer player's think time box
                 WinEnableControl ( hwnd, TB_BTIME, 0 );

                 // Disable 'seconds' and 'plies' radio buttons for search
                 WinEnableControl ( hwnd, RB_B_IS_SECONDS, 0 );
                 WinEnableControl ( hwnd, RB_B_IS_PLIES, 0 );

                 // Disable the black computer player's bias box
                 WinEnableControl ( hwnd, TB_BBIAS, 0 );
                 break;

            case RB_BCOMPUTER:
                 // Enable the black computer player's think time box
                 WinEnableControl ( hwnd, TB_BTIME, 1 );

                 // Enable 'seconds' and 'plies' radio buttons for search
                 WinEnableControl ( hwnd, RB_B_IS_SECONDS, 1 );
                 WinEnableControl ( hwnd, RB_B_IS_PLIES, 1 );

                 // Enable the black computer player's bias box
                 WinEnableControl ( hwnd, TB_BBIAS, 1 );
                 break;
         }
      }
      break;

      case WM_COMMAND:
      {
         switch ( COMMANDMSG(&msg)->cmd )
         {
            case DID_OK:
            {
               // Get the noisy flag...
               SoundEnableFlag =
                  WinQueryButtonCheckstate ( hwnd,
                                             CB_ENABLE_SOUNDS );

               // Figure out whether white is human or computer
               int checked = WinQueryButtonCheckstate ( hwnd,
                                                        RB_WHUMAN );

               DefPlayer.whiteType = checked ?
                                     DefPlayerInfo::humanPlayer :
                                     DefPlayerInfo::computerPlayer;

               // Think time for white
               SHORT value;

               BOOL validNumber = WinQueryDlgItemShort ( hwnd,
                                                         TB_WTIME,
                                                         &value,
                                                         0 );

               DefPlayer.whiteThinkTime = validNumber ? value : 3;

               checked = WinQueryButtonCheckstate ( hwnd,
                                                    RB_W_IS_SECONDS );

               DefPlayer.whiteUseSeconds = checked ? 1 : 0;

               // Figure out whether black is human or computer

               checked = WinQueryButtonCheckstate ( hwnd, RB_BHUMAN );

               DefPlayer.blackType = checked ?
                                     DefPlayerInfo::humanPlayer :
                                     DefPlayerInfo::computerPlayer;

               // Think time for black

               validNumber = WinQueryDlgItemShort ( hwnd,
                                                    TB_BTIME,
                                                    &value,
                                                    0 );

               DefPlayer.blackThinkTime = validNumber ? value : 3;

               checked = WinQueryButtonCheckstate ( hwnd,
                                                    RB_B_IS_SECONDS );

               DefPlayer.blackUseSeconds = checked ? 1 : 0;

               // Get white's search bias...

               validNumber = WinQueryDlgItemShort ( hwnd,
                                                    TB_WBIAS,
                                                    &value,
                                                    0 );

               DefPlayer.whiteSearchBias = validNumber ? value : 0;

               // Get black's search bias...

               validNumber = WinQueryDlgItemShort ( hwnd,
                                                    TB_BBIAS,
                                                    &value,
                                                    0 );

               DefPlayer.blackSearchBias = validNumber ? value : 0;

               checked = WinQueryButtonCheckstate ( hwnd,
                                                    CB_RESTORE_GAME );

               DefPlayer.restoreGameFlag = checked ? 1 : 0;

               WinDismissDlg ( hwnd, DID_OK );
            }
            break;

            case DID_CANCEL:
            {
               WinDismissDlg ( hwnd, DID_CANCEL );
            }
            break;
         }
      }
      break;

      default:
      {
         result = WinDefDlgProc ( hwnd, msg, mp1, mp2 );
      }
      break;
   }

   return result;
}

static cBOOLEAN DefinePlayerDialog ( HWND hwndClient )
{
   ULONG fred;

   fred = WinDlgBox
               ( HWND_DESKTOP,               // parent is desk top
                 hwndClient,                 // owner is main frame
                 DefineChessPlayers_DlgProc,
                 0L,                         // load from resource file
                 IDD_DEFINE_PLAYERS,         // dialog resource id
                 0 );                        // dialog parameter pointer

   if ( fred == DID_ERROR )
   {
      AbortProgram ( hwndClient, myHab );
   }

   return (fred == DID_OK) ? cTRUE : cFALSE;
}



static void SetChessWindowSize ( HWND hwndFrame, cBOOLEAN showDebugInfo )
{
   RECTL rectDesktop;
   WinQueryWindowRect ( HWND_DESKTOP, &rectDesktop );

   int newDX = CHESS_WINDOW_DX;
   int newDY = CHESS_WINDOW_DY;

   if ( showDebugInfo )
   {
      newDX += DEBUG_WINDOW_DX;
   }

   RECTL newFrameRect;
   newFrameRect.xLeft = (rectDesktop.xRight - newDX) / 2;
   newFrameRect.xRight = newFrameRect.xLeft + newDX;
   newFrameRect.yBottom = (rectDesktop.yTop - newDY) / 2;
   newFrameRect.yTop = newFrameRect.yBottom + newDY;
   WinCalcFrameRect ( hwndFrame, &newFrameRect, FALSE );

   WinSetWindowPos ( hwndFrame, HWND_TOP,
                     newFrameRect.xLeft,
                     newFrameRect.yBottom,
                     newFrameRect.xRight - newFrameRect.xLeft,
                     newFrameRect.yTop - newFrameRect.yBottom,
                     SWP_ACTIVATE | SWP_MOVE | SWP_SIZE | SWP_SHOW );
}



int main()
{
   static CHAR chessWindowClass[] = "DonChessWindow";

   static ULONG frameFlags = FCF_TITLEBAR | FCF_SYSMENU | FCF_MINBUTTON |
                             FCF_SHELLPOSITION | FCF_TASKLIST |
                             FCF_DLGBORDER | FCF_MENU;

   HAB hab = WinInitialize ( 0 );
   HMQ hmq = WinCreateMsgQueue ( hab, 0 );
   QMSG qmsg;

   myHab = hab;

   WinRegisterClass ( hab,
                      chessWindowClass,
                      ChessWndProc,
                      CS_SIZEREDRAW,
                      sizeof (ChessUI_os2_pm *) );

   HWND  hwndClient;
   HWND  hwndFrame = WinCreateStdWindow ( HWND_DESKTOP,
                                          0,
                                          &frameFlags,
                                          chessWindowClass,
                                          TitleBarText_WhitesTurn,
                                          0L,    // style of client window
                                          0,  // module handle for resources
                                          DDC_MENU_RESOURCE,
                                          &hwndClient );

   global_hwndClient = hwndClient;
   global_FrameHwnd = hwndFrame;


   WinSendMsg ( hwndFrame,
                WM_SETICON,
                MPFROMLONG(WinQuerySysPointer(HWND_DESKTOP,SPTR_APPICON,FALSE)),
                0 );

   SetChessWindowSize ( hwndFrame, cFALSE );

   // Start ChessGame thread here

   ChessThreadParms threadParms ( hwndClient, hwndFrame, hab );

   int chessThreadID = _beginthread ( ChessThreadFunc,   // thread function
                                      32u * 1024u,       // stack size
                                      &threadParms );    // thread parameter

   if ( chessThreadID >= 0 )
   {
      for(;;)
      {
         while ( WinGetMsg ( hab, &qmsg, 0, 0, 0 ) && !globalAbortFlag )
         {
            WinDispatchMsg ( hab, &qmsg );
         }

         if ( globalAbortFlag || globalImmedQuitFlag )
         {
            break;
         }

         USHORT choice = WinMessageBox
            ( HWND_DESKTOP,
              hwndClient,
              "Really quit chess program?",
              CHESS_PROGRAM_NAME,
              0,
              MB_YESNO | MB_QUERY | MB_APPLMODAL | MB_MOVEABLE );

         if ( choice == MBID_YES )
         {
            break;
         }

         WinCancelShutdown ( hmq, FALSE );
      }

      DosKillThread ( chessThreadID );
   }
   else
   {
      // Put error code here!

      ChessBeep ( 880, 200 );
   }

   WinDestroyWindow ( hwndFrame );
   WinDestroyMsgQueue ( hmq );
   WinTerminate ( hab );

   return 0;
}


static void SetStaticColors ( HWND hwnd, LONG fg, LONG bg )
{
   WinSetPresParam ( hwnd, PP_FOREGROUNDCOLORINDEX, sizeof(LONG), &fg );
   WinSetPresParam ( hwnd, PP_BACKGROUNDCOLORINDEX, sizeof(LONG), &bg );
}


static void CheckHumanMoveState ( cBOOLEAN   insideHumanMove,
                                  HWND       hwndGameMenu,
                                  HWND       hwndEditMenu,
                                  HWND       hwndViewMenu )
{
   if ( insideHumanMove )
   {
      if ( hwndGameMenu )
      {
         WinEnableMenuItem ( hwndGameMenu, 101, TRUE );    // enable Game/New
      }

      if ( hwndEditMenu )
      {
         WinEnableMenuItem ( hwndEditMenu, 206, TRUE );   // enable Clear Board
         WinEnableMenuItem ( hwndEditMenu, 207, TRUE );   // enable Edit Mode
      }

      if ( hwndViewMenu )
      {
      }
   }
   else
   {
      if ( hwndGameMenu )
      {
         WinEnableMenuItem ( hwndGameMenu, 101, FALSE );  // disable Game/New
      }

      if ( hwndEditMenu )
      {
         WinEnableMenuItem ( hwndEditMenu, 206, FALSE );  // Clear Board
         WinEnableMenuItem ( hwndEditMenu, 207, FALSE );  // Edit Mode
      }

      if ( hwndViewMenu )
      {
      }
   }
}


static void SendEditRequest ( int x, int y, SQUARE s )
{
   // If the following flag is set, it means that an edit
   // is still pending.  We don't want to get confused, so
   // we ignore mouse clicks until then.

   if ( !Global_BoardEditRequest.isValid )
   {
      Global_BoardEditRequest.x = x;
      Global_BoardEditRequest.y = y;
      Global_BoardEditRequest.square = s;
      Global_BoardEditRequest.isValid = cTRUE;
   }
}


MRESULT EXPENTRY _export ChessWndProc ( HWND     hwnd,
                                        ULONG    msg,
                                        MPARAM   mp1,
                                        MPARAM   mp2 )
{
   MRESULT result = FALSE;
   char buffer [128];
   static HWND hwndGameMenu = 0;
   static HWND hwndEditMenu = 0;
   static HWND hwndViewMenu = 0;

   static cBOOLEAN insideHumanMove = cFALSE;

   switch ( msg )
   {
      case WM_INITMENU:
      {
         switch ( SHORT1FROMMP(mp1) )
         {
            case 100:   // Game menu
            {
               hwndGameMenu = HWNDFROMMP ( mp2 );
            }
            break;

            case 200:   // Edit menu
            {
               hwndEditMenu = HWNDFROMMP ( mp2 );
            }
            break;

            case 300:   // View menu
            {
               hwndViewMenu = HWNDFROMMP ( mp2 );
            }
            break;
         }

         CheckHumanMoveState ( insideHumanMove,
                               hwndGameMenu,
                               hwndEditMenu,
                               hwndViewMenu );
      }
      break;

      case WM_CREATE:
      if ( !DefinePlayerDialog(hwnd) )
      {
         globalAbortFlag = cTRUE;
      }
      else
      {
         HWND sHwnd;
         int textWidth = 180;
         int textSep = 2;
         int textHeight = WinQuerySysValue ( HWND_DESKTOP,
                                             SV_CYTITLEBAR );

         int textX = CHESS_BITMAP_DX*8 + 10;
         int textY = CHESS_BITMAP_DY*8 - textHeight - textSep;
         ULONG textID;

         for ( textID = FIRST_DEBUG_STATIC_ID;
               textID <= LAST_DEBUG_STATIC_ID;
               textID++ )
         {
            sHwnd = WinCreateWindow ( hwnd,
                              WC_STATIC,
                              "",
                              WS_VISIBLE | SS_TEXT | DT_VCENTER,
                              textX,                    // x1
                              textY,                    // y1
                              textWidth,                // dx
                              textHeight,               // dy
                              hwnd,
                              HWND_TOP,
                              textID,
                              0,
                              0 );

            SetStaticColors ( sHwnd, CLR_YELLOW, CLR_BLUE );
            textY -= (textHeight + textSep);
         }
      }
      break;

      case WM_COMMAND:   // Mostly handles program's menus
      {
         switch ( COMMANDMSG(&msg)->cmd )
         {
            case 101:   // Game/New
            {
               Global_ResetGameFlag = cTRUE;
            }
            break;

            case 102:   // Game/Open
            {
            }
            break;

            case 103:   // Game/Save
            {
            }
            break;

            case 104:   // Game/Save as
            {
            }
            break;

            case 105:   // Game/Exit
            {
               WinSendMsg ( hwnd, WM_CLOSE, 0, 0 );
            }
            break;

            case 201:   // Edit/Undo Move
            {
            }
            break;

            case 202:   // Edit/Redo Move
            {
            }
            break;

            case 206:   // Edit/Clear board
            {
               Global_ClearBoardFlag = cTRUE;
            }
            break;

            case 207:   // Edit/Edit mode
            {
               int editMode = WinIsMenuItemChecked ( hwndEditMenu, 207 );
               editMode ^= MIA_CHECKED;
               WinCheckMenuItem ( hwndEditMenu, 207, editMode );
            }
            break;

            case 301:   // View/Rotate Board
            {
               TheBoardDisplayBuffer.toggleView();
               TheBoardDisplayBuffer.freshenBoard();
            }
            break;

            case 302:   // View/Freshen
            {
               TheBoardDisplayBuffer.freshenBoard();
            }
            break;

            case 303:   // View/Debug Info
            {
               int viewDebugInfo = WinIsMenuItemChecked ( hwndViewMenu, 303 );
               viewDebugInfo ^= MIA_CHECKED;
               WinCheckMenuItem ( hwndViewMenu, 303, viewDebugInfo );

               if ( viewDebugInfo & MIA_CHECKED )
               {
                  // Make frame window bigger, to see debug info
                  Global_ViewDebugFlag = cTRUE;
               }
               else
               {
                  // Make frame window smaller, to hide debug info
                  Global_ViewDebugFlag = cFALSE;
               }

               SetChessWindowSize ( global_FrameHwnd, Global_ViewDebugFlag );
            }
            break;
         }
      }
      break;

      case WM_ERASEBACKGROUND:
      {
         if ( !globalAbortFlag )
         {
            WinFillRect ( HPS(mp1), PRECTL(mp2), CLR_BLUE );
         }
      }
      break;

      case WM_PAINT:
      {
         if ( !globalAbortFlag )
         {
            RECTL rectl;

            HPS hps = WinBeginPaint ( hwnd, NULLHANDLE, &rectl );
            TheBoardDisplayBuffer.draw ( hps,
                                         SQUARE_CHESSX ( rectl.xLeft ),
                                         SQUARE_CHESSX ( rectl.xRight ),
                                         SQUARE_CHESSY ( rectl.yBottom ),
                                         SQUARE_CHESSY ( rectl.yTop ) );
            WinEndPaint ( hps );
         }
      }
      break;

      case WM_BUTTON1CLICK:
      {
         if ( insideHumanMove )
         {
            int mouseX = SHORT1FROMMP ( mp1 );
            int mouseY = SHORT2FROMMP ( mp1 );

            int x = SQUARE_CHESSX ( mouseX );
            int y = SQUARE_CHESSY ( mouseY );
            if ( !TheBoardDisplayBuffer.queryWhiteView() )
            {
               // Need to rotate the coords...

               x = 7 - x;
               y = 7 - y;
            }

            int editMode = WinIsMenuItemChecked ( hwndEditMenu, 207 );
            if ( editMode )
            {
               SQUARE square;

               if ( EditSquareDialog ( hwnd, square ) )
               {
                  SendEditRequest ( x, y, square );
               }
            }
            else
            {
               TheBoardDisplayBuffer.squareSelectedNotify ( x, y );
            }
         }
      }
      break;

      case WM_BUTTON2CLICK:
      {
         if ( insideHumanMove )
         {
            int mouseX = SHORT1FROMMP ( mp1 );
            int mouseY = SHORT2FROMMP ( mp1 );

            int editMode = WinIsMenuItemChecked ( hwndEditMenu, 207 );
            if ( editMode )
            {
               int x = SQUARE_CHESSX ( mouseX );
               int y = SQUARE_CHESSY ( mouseY );

               if ( !TheBoardDisplayBuffer.queryWhiteView() )
               {
                  // Need to rotate the coords...

                  x = 7 - x;
                  y = 7 - y;
               }

               SendEditRequest ( x, y, Global_BoardEditRequest.square );
            }
         }
      }
      break;

      case WM_DDC_ENTER_HUMAN_MOVE:
      {
         // Getting here means that we have just begun reading
         // a human player's move.  There are certain menu options
         // whose enable state should be changed now.

         insideHumanMove = cTRUE;
         CheckHumanMoveState ( insideHumanMove,
                               hwndGameMenu,
                               hwndEditMenu,
                               hwndViewMenu );
      }
      break;

      case WM_DDC_LEAVE_HUMAN_MOVE:
      {
         // Getting here means we are just now finishing with
         // a human player's move.

         insideHumanMove = cFALSE;
         CheckHumanMoveState ( insideHumanMove,
                               hwndGameMenu,
                               hwndEditMenu,
                               hwndViewMenu );
      }
      break;

      case WM_DDC_FATAL:
      {
         char *errorText = (char *)(mp1);

         WinMessageBox ( HWND_DESKTOP,
                         hwnd,
                         errorText,
                         CHESS_PROGRAM_NAME " fatal error",
                         0,
                         MB_MOVEABLE | MB_CUACRITICAL | MB_OK );

         WinPostMsg ( hwnd, WM_QUIT, ULONG(0), ULONG(0) );
      }
      break;

      case WM_DDC_PREDICT_MATE:
      {
         sprintf ( buffer, "Mate in %d", int(mp1) );
         WinMessageBox ( HWND_DESKTOP,
                         hwnd,
                         buffer,
                         CHESS_PROGRAM_NAME,
                         0,
                         MB_MOVEABLE | MB_OK );

         *(int *)(mp2) = 1;   // signals that message box is finished
      }
      break;

      case WM_DDC_PROMOTE_PAWN:
      {
         SQUARE *promPtr = (SQUARE *)mp1;
         SQUARE localProm;

         WinDlgBox ( HWND_DESKTOP,
                     hwnd,
                     PromotePawn_DlgProc,
                     0L,
                     IDD_PROMOTE_PAWN,
                     &localProm );

         if ( localProm != R_INDEX &&
              localProm != B_INDEX &&
              localProm != N_INDEX &&
              localProm != Q_INDEX )
         {
            localProm = Q_INDEX;
         }

         *promPtr = localProm;
      }
      break;

      case WM_DDC_GAME_RESULT:
      {
         ChessUI_os2_pm::gameReport *report =
            (ChessUI_os2_pm::gameReport *)(mp1);

         char *message;

         switch ( report->winner )
         {
            case SIDE_WHITE:
            {
               message = report->resignation ? "Black resigns" : "White wins";
            }
            break;

            case SIDE_BLACK:
            {
               message = report->resignation ? "White resigns" : "Black wins";
            }
            break;

            case SIDE_NEITHER:
            {
               message = "This game is a draw";
            }
            break;

            default:
            {
               message = "??? I'm confused ???";
            }
            break;
         }

         WinMessageBox ( HWND_DESKTOP,
                         hwnd,
                         message,
                         CHESS_PROGRAM_NAME,
                         0,
                         MB_MOVEABLE | MB_OK );
      }
      break;

      default:
      {
         result = WinDefWindowProc ( hwnd, msg, mp1, mp2 );
      }
      break;
   }

   return result;
}


static void RestoreBoardFromFile ( FILE *f, ChessBoard &board )
{
   board.Init();

   Move move;
   UnmoveInfo unmove;

   while ( fread ( &move, sizeof(Move), 1, f ) == 1 )
   {
      if ( (move.dest & SPECIAL_MOVE_MASK) == SPECIAL_MOVE_EDIT )
      {
         board.EditCommand ( move );
         board.SaveSpecialMove ( move );
      }
      else
      {
         if ( board.WhiteToMove() )
         {
            board.MakeWhiteMove ( move, unmove, cTRUE, cTRUE );
         }
         else
         {
            board.MakeBlackMove ( move, unmove, cTRUE, cTRUE );
         }
      }
   }
}


void ChessThreadFunc ( void *_parms )
{
   ChessThreadParms  *threadParms = (ChessThreadParms *)(_parms);

   HMQ hmq = WinCreateMsgQueue ( threadParms->hab, 0 );

   ChessBoard  theBoard;

   if ( DefPlayer.restoreGameFlag )
   {
      FILE *f = fopen ( GameSave_filename, "rb" );
      if ( f )
      {
         RestoreBoardFromFile ( f, theBoard );
         fclose(f);

         // When we get here, the title bar will say it's White's turn.
         // If the file says it's actually Black's turn, we need to
         // change the title bar now.

         if ( theBoard.BlackToMove() )
         {
            WinSetWindowText ( threadParms->boardFrameHwnd,
                               TitleBarText_BlacksTurn );
         }
      }
   }

   ChessUI_os2_pm  theUserInterface ( threadParms->boardClientHwnd,
                                      threadParms->boardFrameHwnd );

   TheBoardDisplayBuffer.update ( theBoard );

   ChessGame theGame ( theBoard, theUserInterface );

   theGame.Play();

   WinDestroyMsgQueue ( hmq );

   globalImmedQuitFlag = cTRUE;
}


void ChessFatal ( const char *message )
{
   WinPostMsg ( global_hwndClient,
                ULONG(WM_DDC_FATAL),
                MPARAM(message),
                MPARAM(0) );
}


void ChessBeep ( int freq, int duration )
{
   if ( SoundEnableFlag )
   {
      DosBeep ( freq, duration );
   }
}


/*--- end of file os2pm.cpp ---*/
