/*=========================================================================

     portable.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Contains main() for a completely portable (stdio.h UI)
     version of NewChess.

     Revision history:

1993 May [Don Cross]
     Started writing.

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1995 March 26 [Don Cross]
     Added self-play capability.

1995 April 2 [Don Cross]
     Added command line option to load game from file.
     This enables Dr. Chenard to play against itself in
     an arbitrary position.
     Made separate functions LoadGame and SaveGame.

1996 February 28 [Don Cross]
     Put in advertising for my email address and web page.

1996 July 28 [Don Cross]
     Made it so that when you pass a game filename as argv[1],
     Chenard will use it whether or not the file already exists.
     If the file exists, Chenard loads the game state from it.
     Whether or not it exists, the filename is used for autosave.
     If no filename is provided, Chenard now always starts new
     game and autosaves to "chess.gam".
     These changes make it more convenient to play games on ChessLive.

1996 August 23 [Don Cross]
     Replacing old memory-based learning tree with disk-based tree file

=========================================================================*/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "chess.h"
#include "uistdio.h"
#include "lrntree.h"


static void Advertise()
{
    printf ( "\nThe Portable Chess Program v 1.132" );

    #ifdef _WIN32
        printf ( " (Win32 text version)" );
    #endif

    printf ( "\n" );
    printf ( "by Don Cross <dcross@intersrv.com>, 22 June 1997.\n" );
    printf ( "http://www.intersrv.com/~dcross\n\n" );
}


int main ( int argc, char *argv[] )
{
    Advertise();

    if ( argc == 1 ||
         (argc>1 && argv[1][0] != '-') )
    {
        ChessBoard     theBoard;
        ChessUI_stdio  theUserInterface;

        const char *gameFilename = "chess.gam";
        if ( argc == 2 )
        {
            gameFilename = argv[1];
        }
        else
        {
            remove ( gameFilename );   // start over!
        }

        if ( gameFilename )
        {
            // Load the game from the given filename.
            if ( LoadGame ( theBoard, gameFilename ) )
            {
                printf ( "Continuing game in file '%s'\n", gameFilename );
            }
            else
            {
                printf ( "Will auto-save new game to '%s'\n", gameFilename );
            }
        }

        ChessGame theGame ( theBoard, theUserInterface );

        if ( gameFilename )
        {
            if ( !theGame.AutoSaveToFile ( gameFilename ) )
            {
                fprintf (
                    stderr,
                    "Error:  Auto-save '%s' option has failed (out of memory?)\x07\n",
                    gameFilename );

                return 1;
            }
            theUserInterface.SetScoreDisplay ( cTRUE );
        }

        theGame.Play();
    }
    else if ( argc == 2 )
    {
        if ( strncmp ( argv[1], "-t", 2 ) == 0 )
        {
            // Tree trainer

            int timeToThink = atoi(argv[1]+2);
            if ( timeToThink < 1 )
            {
                timeToThink = 10;
            }

            ChessBoard theBoard;
            ChessUI_stdio theUserInterface;

            ComputerChessPlayer *whitePlayer =
                new ComputerChessPlayer ( theUserInterface );

            ComputerChessPlayer *blackPlayer =
                new ComputerChessPlayer ( theUserInterface );

            if ( !whitePlayer || !blackPlayer )
            {
                ChessFatal ( "Out of memory!\n" );
            }

            whitePlayer->SetTimeLimit ( INT32(timeToThink) * INT32(100) );
            blackPlayer->SetTimeLimit ( INT32(timeToThink) * INT32(100) );

            whitePlayer->SetSearchBias ( 1 );
            blackPlayer->SetSearchBias ( 1 );

            whitePlayer->SetOpeningBookEnable ( cFALSE );
            blackPlayer->SetOpeningBookEnable ( cFALSE );

            whitePlayer->SetTrainingEnable ( cFALSE );
            blackPlayer->SetTrainingEnable ( cFALSE );

            ChessGame theGame (
                theBoard,
                theUserInterface,
                cFALSE,        // white is played by a ComputerChessPlayer object
                cFALSE,        // black is played by a ComputerChessPlayer object
                whitePlayer,
                blackPlayer );

            theUserInterface.SetScoreDisplay ( cTRUE );

            Learn_Output = 1;

            LearnTree tree;
            tree.trainer (
                whitePlayer,
                blackPlayer,
                theBoard,
                theUserInterface,
                INT32(timeToThink) * INT32(100) );
        }
        else if ( strncmp ( argv[1], "-s", 2 ) == 0 )
        {
            // Self play

            int timeToThink = atoi(argv[1]+2);
            if ( timeToThink < 1 )
            {
                timeToThink = 10;
            }

            ChessBoard theBoard;
            ChessUI_stdio theUserInterface;

            ComputerChessPlayer *whitePlayer =
                new ComputerChessPlayer ( theUserInterface );

            ComputerChessPlayer *blackPlayer =
                new ComputerChessPlayer ( theUserInterface );

            if ( !whitePlayer || !blackPlayer )
            {
                ChessFatal ( "Out of memory!\n" );
            }

            whitePlayer->SetTimeLimit ( INT32(timeToThink) * INT32(100) );
            blackPlayer->SetTimeLimit ( INT32(timeToThink) * INT32(100) );

            whitePlayer->SetSearchBias ( 1 );
            blackPlayer->SetSearchBias ( 1 );

            theUserInterface.SetScoreDisplay ( cTRUE );

            unsigned long uniqueTag = 1;
            extern const char *LogFilename;
            const char *tagFilename = "chenard.id";
            FILE *tagSaveFile = fopen ( tagFilename, "rt" );
            if ( tagSaveFile )
            {
                if ( fscanf ( tagSaveFile, "%lu", &uniqueTag ) != 1 )
                {
                    uniqueTag = 1;
                }

                fclose ( tagSaveFile );
                tagSaveFile = 0;
            }

            tagSaveFile = fopen ( tagFilename, "wt" );
            if ( !tagSaveFile )
            {
                char err [128];
                sprintf ( err, "Cannot open unique tag file '%s' for write!\n", tagFilename );
                ChessFatal ( err );
            }

            fprintf ( tagSaveFile, "%lu\n", 1+uniqueTag );
            fclose ( tagSaveFile );

            static char uniqueFilename [256];
            sprintf ( uniqueFilename, "self%04lu.txt", uniqueTag );
            LogFilename = uniqueFilename;

            ChessGame theGame (
                theBoard,
                theUserInterface,
                cFALSE,        // white is played by a ComputerChessPlayer object
                cFALSE,        // black is played by a ComputerChessPlayer object
                whitePlayer,
                blackPlayer );

            // When playing against self, improve training by telling
            // one of the players (randomly) to not use opening book.

            if ( ChessRandom(2) )
            {
                whitePlayer->SetOpeningBookEnable ( cFALSE );
            }
            else
            {
                blackPlayer->SetOpeningBookEnable ( cFALSE );
            }

            theGame.Play();
            Advertise();
        }
        else
        {
            fprintf ( stderr, "Use:  %s [-s<time_per_move_in_seconds>]\n", argv[0] );
            fprintf ( stderr, "The option '-s' causes Dr. Chenard to play against itself.\n" );
            return 1;
        }
    }

    return 0;
}

/*--- end of file portable.cpp ---*/
