/*=========================================================================

     morder.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Contains move-ordering heuristics for the min-max search.

     Revision history:

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1994 February 3 [Don Cross]
     Adding BestPath support.

1994 February 5 [Don Cross]
     Added "killer move" heuristics.

1994 February 10 [Don Cross]
     Adding piece indexing.

1994 February 20 [Don Cross]
     Adding most-probable capture combination min-maxing.

=========================================================================*/

#include "chess.h"

#define  PREV_SQUARE_BONUS          5
#define  CHECK_BONUS              140
#define  PIECE_ATTACKED_BY_PAWN    70
#define  KILLER_MOVE_BONUS        150

#define  WHITE_BEST_PATH     (20000)
#define  BLACK_BEST_PATH    (-20000)

#define  ENEMY_MOBILITY(n)   (n)

#define  KNIGHT_MOVE    3
#define  BISHOP_MOVE    3
#define  ROOK_MOVE      5
#define  QUEEN_MOVE     9
#define  KING_MOVE     15


//---------------------------------------------------------------------------

struct AttackRay
{
   int    length;     // how many pieces there are in the ray
   int    pos;        // how many have been used so far in min-max
   SCORE  val [9];    // raw piece values (except for kings=-30000,30000)
};


struct AttackMass
{
   int         rayCount;
   AttackRay   ray [9];

   cBOOLEAN remove ( SCORE &score );
};


cBOOLEAN AttackMass::remove ( SCORE &score )
{
   // pick smallest value in front of any ray...

   score = 31000;
   SCORE x;
   int bestr = -1;

   for ( int r=0; r < rayCount; r++ )
   {
      int pos = ray[r].pos;
      if ( pos < ray[r].length &&
           (x = ray[r].val[pos]) < score )
      {
         score = x;
         bestr = r;
      }
   }

   if ( bestr > -1 )
   {
      ++ray[bestr].pos;
      return cTRUE;
   }
   else
   {
      return cFALSE;
   }
}


#define  KING_TOKEN    (30000)


static void FindAttacks ( AttackMass &wa, AttackMass &ba,
                          const SQUARE * const p )
{
   wa.rayCount = 0;
   ba.rayCount = 0;

   SQUARE x;

   int    wlen = 0;
   int    blen = 0;
   SCORE *wval = &(wa.ray[wa.rayCount].val[0]);
   SCORE *bval = &(ba.ray[ba.rayCount].val[0]);

   // Look for knights first

   if ( (x = p[OFFSET(1,2)]) & (BN_MASK | WN_MASK) )
   {
      if ( x & WN_MASK )
      {
         wval [wlen++] = KNIGHT_VAL;
      }
      else
      {
         bval [blen++] = KNIGHT_VAL;
      }
   }

   if ( (x = p[OFFSET(1,-2)]) & (BN_MASK | WN_MASK) )
   {
      if ( x & WN_MASK )
      {
         wval [wlen++] = KNIGHT_VAL;
      }
      else
      {
         bval [blen++] = KNIGHT_VAL;
      }
   }

   if ( (x = p[OFFSET(-1,2)]) & (BN_MASK | WN_MASK) )
   {
      if ( x & WN_MASK )
      {
         wval [wlen++] = KNIGHT_VAL;
      }
      else
      {
         bval [blen++] = KNIGHT_VAL;
      }
   }

   if ( (x = p[OFFSET(-1,-2)]) & (BN_MASK | WN_MASK) )
   {
      if ( x & WN_MASK )
      {
         wval [wlen++] = KNIGHT_VAL;
      }
      else
      {
         bval [blen++] = KNIGHT_VAL;
      }
   }

   if ( (x = p[OFFSET(2,1)]) & (BN_MASK | WN_MASK) )
   {
      if ( x & WN_MASK )
      {
         wval [wlen++] = KNIGHT_VAL;
      }
      else
      {
         bval [blen++] = KNIGHT_VAL;
      }
   }

   if ( (x = p[OFFSET(2,-1)]) & (BN_MASK | WN_MASK) )
   {
      if ( x & WN_MASK )
      {
         wval [wlen++] = KNIGHT_VAL;
      }
      else
      {
         bval [blen++] = KNIGHT_VAL;
      }
   }

   if ( (x = p[OFFSET(-2,1)]) & (BN_MASK | WN_MASK) )
   {
      if ( x & WN_MASK )
      {
         wval [wlen++] = KNIGHT_VAL;
      }
      else
      {
         bval [blen++] = KNIGHT_VAL;
      }
   }

   if ( (x = p[OFFSET(-2,-1)]) & (BN_MASK | WN_MASK) )
   {
      if ( x & WN_MASK )
      {
         wval [wlen++] = KNIGHT_VAL;
      }
      else
      {
         bval [blen++] = KNIGHT_VAL;
      }
   }

   // If we found any knights, allocate their rays...

   if ( wlen > 0 )
   {
      wa.ray[wa.rayCount].pos = 0;
      wa.ray[wa.rayCount++].length = wlen;
      wlen = 0;
      wval = &(wa.ray[wa.rayCount].val[0]);
   }

   if ( blen > 0 )
   {
      ba.ray[ba.rayCount].pos = 0;
      ba.ray[ba.rayCount++].length = blen;
      blen = 0;
      bval = &(ba.ray[ba.rayCount].val[0]);
   }

   // ORTHOGONALS...North first

   const SQUARE *y;

   if ( (x = *(y = p + NORTH)) & (WK_MASK | BK_MASK) )
   {
      if ( x & WK_MASK )
      {
         wval[wlen++] = KING_TOKEN;
      }
      else
      {
         bval[blen++] = KING_TOKEN;
      }
   }
   else
   {
      while ( !(x & OFFBOARD) )
      {
         if ( x & WHITE_MASK )
         {
            while ( (x & (WR_MASK | WQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  wval[wlen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += NORTH);
            }
            break;
         }
         else if ( x & BLACK_MASK )
         {
            while ( (x & (BR_MASK | BQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  bval[blen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += NORTH);
            }
            break;
         }

         x = *(y += NORTH);
      }
   }

   if ( wlen > 0 )
   {
      wa.ray[wa.rayCount].pos = 0;
      wa.ray[wa.rayCount++].length = wlen;
      wlen = 0;
      wval = &(wa.ray[wa.rayCount].val[0]);
   }

   if ( blen > 0 )
   {
      ba.ray[ba.rayCount].pos = 0;
      ba.ray[ba.rayCount++].length = blen;
      blen = 0;
      bval = &(ba.ray[ba.rayCount].val[0]);
   }

   // SOUTH

   if ( (x = *(y = p + SOUTH)) & (WK_MASK | BK_MASK) )
   {
      if ( x & WK_MASK )
      {
         wval[wlen++] = KING_TOKEN;
      }
      else
      {
         bval[blen++] = KING_TOKEN;
      }
   }
   else
   {
      while ( !(x & OFFBOARD) )
      {
         if ( x & WHITE_MASK )
         {
            while ( (x & (WR_MASK | WQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  wval[wlen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += SOUTH);
            }
            break;
         }
         else if ( x & BLACK_MASK )
         {
            while ( (x & (BR_MASK | BQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  bval[blen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += SOUTH);
            }
            break;
         }

         x = *(y += SOUTH);
      }
   }

   if ( wlen > 0 )
   {
      wa.ray[wa.rayCount].pos = 0;
      wa.ray[wa.rayCount++].length = wlen;
      wlen = 0;
      wval = &(wa.ray[wa.rayCount].val[0]);
   }

   if ( blen > 0 )
   {
      ba.ray[ba.rayCount].pos = 0;
      ba.ray[ba.rayCount++].length = blen;
      blen = 0;
      bval = &(ba.ray[ba.rayCount].val[0]);
   }

   // EAST

   if ( (x = *(y = p + EAST)) & (WK_MASK | BK_MASK) )
   {
      if ( x & WK_MASK )
      {
         wval[wlen++] = KING_TOKEN;
      }
      else
      {
         bval[blen++] = KING_TOKEN;
      }
   }
   else
   {
      while ( !(x & OFFBOARD) )
      {
         if ( x & WHITE_MASK )
         {
            while ( (x & (WR_MASK | WQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  wval[wlen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += EAST);
            }
            break;
         }
         else if ( x & BLACK_MASK )
         {
            while ( (x & (BR_MASK | BQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  bval[blen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += EAST);
            }
            break;
         }

         x = *(y += EAST);
      }
   }

   if ( wlen > 0 )
   {
      wa.ray[wa.rayCount].pos = 0;
      wa.ray[wa.rayCount++].length = wlen;
      wlen = 0;
      wval = &(wa.ray[wa.rayCount].val[0]);
   }

   if ( blen > 0 )
   {
      ba.ray[ba.rayCount].pos = 0;
      ba.ray[ba.rayCount++].length = blen;
      blen = 0;
      bval = &(ba.ray[ba.rayCount].val[0]);
   }

   // WEST

   if ( (x = *(y = p + WEST)) & (WK_MASK | BK_MASK) )
   {
      if ( x & WK_MASK )
      {
         wval[wlen++] = KING_TOKEN;
      }
      else
      {
         bval[blen++] = KING_TOKEN;
      }
   }
   else
   {
      while ( !(x & OFFBOARD) )
      {
         if ( x & WHITE_MASK )
         {
            while ( (x & (WR_MASK | WQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  wval[wlen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += WEST);
            }
            break;
         }
         else if ( x & BLACK_MASK )
         {
            while ( (x & (BR_MASK | BQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  bval[blen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += WEST);
            }
            break;
         }

         x = *(y += WEST);
      }
   }

   if ( wlen > 0 )
   {
      wa.ray[wa.rayCount].pos = 0;
      wa.ray[wa.rayCount++].length = wlen;
      wlen = 0;
      wval = &(wa.ray[wa.rayCount].val[0]);
   }

   if ( blen > 0 )
   {
      ba.ray[ba.rayCount].pos = 0;
      ba.ray[ba.rayCount++].length = blen;
      blen = 0;
      bval = &(ba.ray[ba.rayCount].val[0]);
   }

   // Diagonals: NORTHEAST first

   if ( (x = *(y = p + NORTHEAST)) & (WK_MASK | BK_MASK) )
   {
      if ( x & WK_MASK )
      {
         wval[wlen++] = KING_TOKEN;
      }
      else
      {
         bval[blen++] = KING_TOKEN;
      }
   }
   else
   {
      if ( x & BP_MASK )
      {
         bval[blen++] = PAWN_VAL;
         x = *(y += NORTHEAST);
      }

      while ( !(x & OFFBOARD) )
      {
         if ( x & WHITE_MASK )
         {
            while ( (x & (WB_MASK | WQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  wval[wlen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += NORTHEAST);
            }
            break;
         }
         else if ( x & BLACK_MASK )
         {
            while ( (x & (BB_MASK | BQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  bval[blen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += NORTHEAST);
            }
            break;
         }

         x = *(y += NORTHEAST);
      }
   }

   if ( wlen > 0 )
   {
      wa.ray[wa.rayCount].pos = 0;
      wa.ray[wa.rayCount++].length = wlen;
      wlen = 0;
      wval = &(wa.ray[wa.rayCount].val[0]);
   }

   if ( blen > 0 )
   {
      ba.ray[ba.rayCount].pos = 0;
      ba.ray[ba.rayCount++].length = blen;
      blen = 0;
      bval = &(ba.ray[ba.rayCount].val[0]);
   }

   // NORTHWEST

   if ( (x = *(y = p + NORTHWEST)) & (WK_MASK | BK_MASK) )
   {
      if ( x & WK_MASK )
      {
         wval[wlen++] = KING_TOKEN;
      }
      else
      {
         bval[blen++] = KING_TOKEN;
      }
   }
   else
   {
      if ( x & BP_MASK )
      {
         bval[blen++] = PAWN_VAL;
         x = *(y += NORTHWEST);
      }

      while ( !(x & OFFBOARD) )
      {
         if ( x & WHITE_MASK )
         {
            while ( (x & (WB_MASK | WQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  wval[wlen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += NORTHWEST);
            }
            break;
         }
         else if ( x & BLACK_MASK )
         {
            while ( (x & (BB_MASK | BQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  bval[blen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += NORTHWEST);
            }
            break;
         }

         x = *(y += NORTHWEST);
      }
   }

   if ( wlen > 0 )
   {
      wa.ray[wa.rayCount].pos = 0;
      wa.ray[wa.rayCount++].length = wlen;
      wlen = 0;
      wval = &(wa.ray[wa.rayCount].val[0]);
   }

   if ( blen > 0 )
   {
      ba.ray[ba.rayCount].pos = 0;
      ba.ray[ba.rayCount++].length = blen;
      blen = 0;
      bval = &(ba.ray[ba.rayCount].val[0]);
   }

   // SOUTHWEST

   if ( (x = *(y = p + SOUTHWEST)) & (WK_MASK | BK_MASK) )
   {
      if ( x & WK_MASK )
      {
         wval[wlen++] = KING_TOKEN;
      }
      else
      {
         bval[blen++] = KING_TOKEN;
      }
   }
   else
   {
      if ( x & WP_MASK )
      {
         wval[wlen++] = PAWN_VAL;
         x = *(y += SOUTHWEST);
      }

      while ( !(x & OFFBOARD) )
      {
         if ( x & WHITE_MASK )
         {
            while ( (x & (WB_MASK | WQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  wval[wlen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += SOUTHWEST);
            }
            break;
         }
         else if ( x & BLACK_MASK )
         {
            while ( (x & (BB_MASK | BQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  bval[blen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += SOUTHWEST);
            }
            break;
         }

         x = *(y += SOUTHWEST);
      }
   }

   if ( wlen > 0 )
   {
      wa.ray[wa.rayCount].pos = 0;
      wa.ray[wa.rayCount++].length = wlen;
      wlen = 0;
      wval = &(wa.ray[wa.rayCount].val[0]);
   }

   if ( blen > 0 )
   {
      ba.ray[ba.rayCount].pos = 0;
      ba.ray[ba.rayCount++].length = blen;
      blen = 0;
      bval = &(ba.ray[ba.rayCount].val[0]);
   }

   // SOUTHEAST

   if ( (x = *(y = p + SOUTHEAST)) & (WK_MASK | BK_MASK) )
   {
      if ( x & WK_MASK )
      {
         wval[wlen++] = KING_TOKEN;
      }
      else
      {
         bval[blen++] = KING_TOKEN;
      }
   }
   else
   {
      if ( x & WP_MASK )
      {
         wval[wlen++] = PAWN_VAL;
         x = *(y += SOUTHEAST);
      }

      while ( !(x & OFFBOARD) )
      {
         if ( x & WHITE_MASK )
         {
            while ( (x & (WB_MASK | WQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  wval[wlen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += SOUTHEAST);
            }
            break;
         }
         else if ( x & BLACK_MASK )
         {
            while ( (x & (BB_MASK | BQ_MASK)) || x == EMPTY )
            {
               if ( x != EMPTY )
               {
                  bval[blen++] = RAW_PIECE_VALUE(x);
               }
               x = *(y += SOUTHEAST);
            }
            break;
         }

         x = *(y += SOUTHEAST);
      }
   }

   if ( wlen > 0 )
   {
      wa.ray[wa.rayCount].pos = 0;
      wa.ray[wa.rayCount++].length = wlen;
   }

   if ( blen > 0 )
   {
      ba.ray[ba.rayCount].pos = 0;
      ba.ray[ba.rayCount++].length = blen;
   }
}


static SCORE CalcWhiteAttack ( const SQUARE *p )
{
   AttackMass  wa, ba;
   FindAttacks ( wa, ba, p );
   SCORE ply [32];
   int depth = 0;
   SCORE score;
   SCORE hotSpot = RAW_PIECE_VALUE(*p);

   ply[depth++] = 0;   // nothing new captured (yet)

   while ( wa.remove(score) )
   {
      ply[depth] = ply[depth-1] + hotSpot;
      hotSpot = score;
      ++depth;

      if ( ba.remove(score) )
      {
         // remove illegal captures with white king
         if ( hotSpot == KING_TOKEN )
         {
            --depth;
            break;
         }

         ply[depth] = ply[depth-1] - hotSpot;
         hotSpot = score;
         ++depth;
      }
      else
      {
         break;
      }
   }

   score = ply [--depth];
   int optDepth = depth;

   while ( --depth >= 0 )
   {
      // compare ply[depth] with score

      if ( depth & 1 )
      {
         // min

         if ( ply[depth] < score )
         {
            score = ply[depth];
            optDepth = depth;
         }
      }
      else
      {
         // max

         if ( ply[depth] > score )
         {
            score = ply[depth];
            optDepth = depth;
         }
      }
   }

   return 10 * score + optDepth;
}


static SCORE CalcBlackAttack ( const SQUARE *p )
{
   AttackMass  wa, ba;
   FindAttacks ( wa, ba, p );
   SCORE ply [32];
   int depth = 0;
   SCORE score;
   SCORE hotSpot = RAW_PIECE_VALUE(*p);

   ply[depth++] = 0;   // nothing new captured (yet)

   while ( ba.remove(score) )
   {
      ply[depth] = ply[depth-1] - hotSpot;
      hotSpot = score;
      ++depth;

      if ( wa.remove(score) )
      {
         // remove illegal captures with black king
         if ( hotSpot == KING_TOKEN )
         {
            --depth;
            break;
         }

         ply[depth] = ply[depth-1] + hotSpot;
         hotSpot = score;
         ++depth;
      }
      else
      {
         break;
      }
   }

   score = ply [--depth];
   int optDepth = depth;

   while ( --depth >= 0 )
   {
      // compare ply[depth] with score

      if ( depth & 1 )
      {
         // max

         if ( ply[depth] > score )
         {
            score = ply[depth];
            optDepth = depth;
         }
      }
      else
      {
         // min

         if ( ply[depth] < score )
         {
            score = ply[depth];
            optDepth = depth;
         }
      }
   }

   return 10 * score - optDepth;
}


//---------------------------------------------------------------------------


void ComputerChessPlayer::WhiteMoveOrdering ( const ChessBoard &board,
                                              Move &move,
                                              const UnmoveInfo &unmove,
                                              int depth,
                                              cBOOLEAN bestPathFlag )
{
   // First, check for best path.  It overrides everything!

   if ( bestPathFlag &&
        depth <= currentBestPath.depth &&
        currentBestPath.m[depth] == move )
   {
      move.score = WHITE_BEST_PATH;
      return;
   }

   move.score = MaterialEval ( board.wmaterial, board.bmaterial );

   // "killer move" heuristic...

   if ( depth > 0 &&
        depth <= nextBestPath[depth-1].depth &&
        nextBestPath[depth-1].m[depth] == move )
   {
      move.score += KILLER_MOVE_BONUS;
   }

#if 0
   if ( move.dest == unmove.prev_move.dest )
   {
      move.score += PREV_SQUARE_BONUS;
   }

   if ( board.black_in_check )
   {
      move.score += CHECK_BONUS;
   }
   else
   {
      if ( depth < level )
      {
         int n = board.NumBlackMoves();
         move.score -= ENEMY_MOBILITY(n);
      }
   }
#endif
}




void ComputerChessPlayer::BlackMoveOrdering ( const ChessBoard &board,
                                              Move &move,
                                              const UnmoveInfo &unmove,
                                              int depth,
                                              cBOOLEAN bestPathFlag )
{
   // First, check for best path.  It overrides everything!

   if ( bestPathFlag &&
        depth <= currentBestPath.depth &&
        currentBestPath.m[depth] == move )
   {
      move.score = BLACK_BEST_PATH;
      return;
   }

   move.score = MaterialEval ( board.wmaterial, board.bmaterial );

   // "killer move" heuristic...

   if ( depth > 0 &&
        depth <= nextBestPath[depth-1].depth &&
        nextBestPath[depth-1].m[depth] == move )
   {
      move.score -= KILLER_MOVE_BONUS;
   }

#if 0
   if ( move.dest == unmove.prev_move.dest )
   {
      move.score -= PREV_SQUARE_BONUS;
   }

   if ( board.white_in_check )
   {
      move.score -= CHECK_BONUS;
   }
   else
   {
      if ( depth < level )
      {
         int n = board.NumWhiteMoves();
         move.score += ENEMY_MOBILITY(n);
      }
   }
#endif
}


//----------------------------------------------------------------------


void MoveList::WhiteSort()
{
   if ( num > 1 )
   {
      SCORE bestscore;
      int   besti, i;
      int   limit = num - 1;

      for ( int dest=0; dest < limit; dest++ )
      {
         bestscore = m [besti = dest].score;
         for ( i=dest + 1; i < num; i++ )
         {
            if ( m[i].score > bestscore )
            {
               bestscore = m[i].score;
               besti = i;
            }
         }

         if ( besti != dest )
         {
            Move temp = m[dest];
            m[dest] = m[besti];
            m[besti] = temp;
         }
      }
   }
}


void MoveList::BlackSort()
{
   if ( num > 1 )
   {
      SCORE bestscore;
      int   besti, i;
      int   limit = num - 1;

      for ( int dest=0; dest < limit; dest++ )
      {
         bestscore = m [besti = dest].score;
         for ( i=dest + 1; i < num; i++ )
         {
            if ( m[i].score < bestscore )
            {
               bestscore = m[i].score;
               besti = i;
            }
         }

         if ( besti != dest )
         {
            Move temp = m[dest];
            m[dest] = m[besti];
            m[besti] = temp;
         }
      }
   }
}


/*--- end of file morder.cpp ---*/
