/*===========================================================================

     search.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Contains min-max search algorithms with alpha-beta pruning for chess.

     Revision history:

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1993 October 16 [Don Cross]
     Fixed bug where I had a MakeBlackMove() paired with
     UnmakeWhiteMove().

1993 October 18 [Don Cross]
     Added the 'searchBias' stuff.  This is an attempt to allow
     some randomization of move selection.

1993 October 23 [Don Cross]
     Implemented timed search.
     Added check for draw by repetition of moves:
     {W+, B+, W-, B-} plies repeated 3 times.

1993 October 24 [Don Cross]
     Added call to ChessBoard::IsDefiniteDraw() to WhiteSearch() and
     BlackSearch() to make sure that draws by repetition are found
     anywhere in the search.

     Created a MoveList::AddMove(Move) function, and changed the
     pg.AddMove() line of code in the searchBias conditional to
     use it instead.  The old way was clobbering the bestmove.score,
     so searches were continuing even after a forced win was found.

1993 October 25 [Don Cross]
     Making the ComputerChessPlayer::CheckTimeLimit() member function
     adaptive to the processor speed.  It does this by increasing
     the timeCheckLimit until successive calls to ChessTime() always
     report a difference of at least 0.1 seconds.

1994 January 15 [Don Cross]
     Added ComputerChessPlayer::AbortSearch().
     This required a little reworking of logic in the CheckTimeLimit
     member function too.

1994 January 22 [Don Cross]
     Adding support for variable eval functions.

1994 February 2 [Don Cross]
     Split off from search.cpp to add BestPath stuff.

1994 February 7 [Don Cross]
     Improving search behavior for time limits, and preparing
     to add search windows.

1994 February 14 [Don Cross]
     Made quiescence search call regular search when a side is
     in check.

1994 February 24 [Don Cross]
     Added "killer history" idea.

1994 March 1 [Don Cross]
     Added hash function and code to help find repeated board positions.

1994 March 2 [Don Cross]
     Adding conditional code to dump search to a text file.
     Need to tell the compiler to define DUMP_SEARCH to do this.

1996 July 27 [Don Cross]
     Adding experiment with looking at moves which cause check in the
     quiescence search.

1996 August 4 [Don Cross]
     Adding code to recycle best path from previous search so that
     we can start at a deeper search.

1996 August 8 [Don Cross]
     Adding "obvious move" code to shorten search when one move is
     much better than all others after a certain fraction of the
     total search time.

1996 August 23 [Don Cross]
     Replacing old memory-based learning tree with disk-based
     class LearnTree.

1996 August 26 [Don Cross]
     Extending quiescence search when own side is in check to a
     full width search at that depth, with a cutoff extended
     depth of 'ESCAPE_CHECK_DEPTH' to avoid infinite recursion
     from perpetual checks.

     Bugs in extended quiescence search which cause infinite recursion,
     which in turn uncovered bugs with checking array indices in best 
     path stuff.  Fixed 'em!

===========================================================================*/

#include <stdio.h>

#include "chess.h"
#include "lrntree.h"
#include "profiler.h"


#define  ESCAPE_CHECK_DEPTH   1
#define  MAX_CHECK_DEPTH      1

#define  HASH_HIST_MAX         2560
#define  HASH_HIST_FUNC(h,d)     26


ComputerChessPlayer::ComputerChessPlayer ( ChessUI &ui ):
    ChessPlayer ( ui ),
    maxlevel ( 3 ),
    maxCheckDepth ( MAX_CHECK_DEPTH ),
    generated ( 0 ),
    visited ( 0 ),
    evaluated ( 0 ),
    searchBias ( 0 ),
    timedSearch ( cFALSE ),
    timeCheckLimit ( 100 ),
    whiteEval ( &ComputerChessPlayer::WhiteMidgameEval ),
    blackEval ( &ComputerChessPlayer::BlackMidgameEval ),
    eachBestPathCount ( 0 ),
    expectedNextBoardHash ( 0 ),
    prevCompletedLevel ( 0 ),
    openingBookSearchEnabled ( cTRUE ),
    trainingEnabled ( WACKY_GAME==0 ),
    whiteHist ( new SCORE [4096] ),
    blackHist ( new SCORE [4096] ),
	allowResignation ( cFALSE )
{
    rootml.num = 0;

    if ( !whiteHist || !blackHist )
    {
        ChessFatal ( "Out of memory creating ComputerChessPlayer" );
    }

    ResetHistoryBuffers();
}


ComputerChessPlayer::~ComputerChessPlayer()
{
    delete[] whiteHist;
    delete[] blackHist;
}


void ComputerChessPlayer::ChooseEvalFunctions ( ChessBoard &board )
{
    INT16 *inv = board.inventory;

    // If either side has a lone king, and the other
    // has rook(s) and/or queen(s) use endgame eval #1.

    if ( inv [WP_INDEX] == 0 &&
         inv [WN_INDEX] == 0 &&
         inv [WB_INDEX] == 0 &&
         inv [WR_INDEX] == 0 &&
         inv [WQ_INDEX] == 0 &&
         inv [BR_INDEX] + inv [BQ_INDEX] > 0 )
    {
        whiteEval = &ComputerChessPlayer::EndgameEval1;
        blackEval = &ComputerChessPlayer::EndgameEval1;
    }
    else if ( inv [BP_INDEX] == 0 &&
              inv [BN_INDEX] == 0 &&
              inv [BB_INDEX] == 0 &&
              inv [BR_INDEX] == 0 &&
              inv [BQ_INDEX] == 0 &&
              inv [WR_INDEX] + inv [WQ_INDEX] > 0 )
    {
        whiteEval = &ComputerChessPlayer::EndgameEval1;
        blackEval = &ComputerChessPlayer::EndgameEval1;
    }
    else
    {
        // Use the default midgame evaluator...

        whiteEval = &ComputerChessPlayer::WhiteMidgameEval;
        blackEval = &ComputerChessPlayer::BlackMidgameEval;
    }
}


void ComputerChessPlayer::ResetHistoryBuffers()
{
	for ( int i=0; i<4096; i++ )
		whiteHist[i] = blackHist[i] = 0;
}


cBOOLEAN ComputerChessPlayer::GetMove (
    ChessBoard  &board,
    Move        &bestmove,
    INT32       &timeSpent )
{
	PROFILER_ENTER(PX_SEARCH);

    searchAborted = cFALSE;
    timeSpent = 0;
    INT32 timeBefore = ChessTime();
    if ( timedSearch )
    {
        prevTime = timeBefore;
        stopTime = prevTime + timeLimit;
        timeCheckCounter = 0;
    }

    ChooseEvalFunctions ( board );

    if ( board.WhiteToMove() )
        GetWhiteMove ( board, bestmove );
    else
        GetBlackMove ( board, bestmove );

    INT32 timeAfter = ChessTime();
    timeSpent = timeAfter - timeBefore;

    userInterface.DisplayMove ( board, bestmove );
    userInterface.ReportComputerStats (
        timeSpent,
        visited,
        evaluated,
        generated,
        level+1,
        visnodes,
        gennodes );

    FindPrevBestPath ( bestmove );
    if ( currentBestPath.depth > 1 )
    {
        // Keep hash code of expected board position if opponent makes the
        // move we think he will make.  This helps us to know whether we
        // can recycle best path information on our next search.
        UnmoveInfo unmove[2];
        board.MakeMove ( bestmove, unmove[0] );
        board.MakeMove ( currentBestPath.m[1], unmove[1] );
        expectedNextBoardHash = board.Hash();
        board.UnmakeMove ( currentBestPath.m[1], unmove[1] );
        board.UnmakeMove ( bestmove, unmove[0] );
    }
    else
    {
        expectedNextBoardHash = 0;   // will never match any board hash
    }

	PROFILER_EXIT();

	if ( allowResignation )
	{
        if ( board.WhiteToMove() &&
             bestmove.score <= WHITE_RESIGN_THRESHOLD )
        {
            return cFALSE;   // resign
        }
        else if ( board.BlackToMove() &&
                bestmove.score >= BLACK_RESIGN_THRESHOLD )
        {
            return cFALSE;   // resign
        }
	}

    return cTRUE;   // keep on playing!
}


void ComputerChessPlayer::GetWhiteMove (
    ChessBoard &board,
    Move &bestmove )
{
    moveOrder_bestPathFlag = cFALSE;
    board.GenWhiteMoves ( rootml, this );
    bestmove = rootml.m[0];   // just in case!

    gennodes[0] = generated = rootml.num;
    visited = evaluated = 0;

    for ( int i=0; i < NODES_ARRAY_SIZE; i++ )
        visnodes[i] = gennodes[i] = 0;

    for ( i=0; i < 4096; i++ )
    {
        whiteHist[i] /= 2;
        blackHist[i] /= 2;
    }

    if ( rootml.num == 1 )
        return;   // only one legal move exists, so just do it!

    if ( openingBookSearchEnabled && WhiteOpening ( board, bestmove ) )
    {
        userInterface.ReportSpecial ( "opening" );
        return;   // We found a groovy opening move
    }

    LearnTree tree;
    if ( timedSearch && trainingEnabled && tree.familiarPosition ( board, bestmove, timeLimit, rootml ) )
    {
		char buffer [64];
		sprintf ( buffer, "experience (%6d)", int(bestmove.score) );
		userInterface.ReportSpecial ( buffer );
        return;   // We found a move we know is best from a previous (longer) search
    }

    // See if we can recycle best path info from the previous move's search.
    UINT32 hash = board.Hash();
    int startLevel = 0;
    if ( hash==expectedNextBoardHash && currentBestPath.depth>=2 && prevCompletedLevel>1 )
    {
        // Strip the top two plies because they are in the past!
        for ( int i=2; i <= currentBestPath.depth; i++ )
            currentBestPath.m[i-2] = currentBestPath.m[i];

        currentBestPath.depth -= 2;

        eachBestPathCount = 1;
        eachBestPath[0] = currentBestPath;
        startLevel = prevCompletedLevel - 1;
        rootml.SendToFront ( currentBestPath.m[0] );
        bestmove = rootml.m[0];
    }
    else
    {
        eachBestPathCount = 0;
        currentBestPath.depth = 0;
    }

    nextBestPath[0].depth = 0;
    prevCompletedLevel = 0;
    for ( level=startLevel; !searchAborted && level <= maxlevel; level++ )
    {
        WhiteSearchRoot ( board, bestmove );
        if ( !searchAborted )
            prevCompletedLevel = level;

        if ( searchBias )
        {
            MoveList pg;   // list of "pretty good" moves
            pg.num = 0;
            for ( int i=0; i < rootml.num; i++ )
            {
                if ( rootml.m[i].score > bestmove.score - searchBias )
                    pg.AddMove ( rootml.m[i] );
            }

            int r = ChessRandom ( pg.num );
            bestmove = pg.m[r];
        }

        if ( bestmove.score >= WON_FOR_WHITE )
            break;   // We have found a forced win!

        if ( bestmove.score <= WON_FOR_BLACK )
            break;   // Time to give up - no deeper search will improve things.
    }

    if ( bestmove.score >= WON_FOR_WHITE &&
         bestmove.score < WHITE_WINS - WIN_DELAY_PENALTY )
    {
        // If we get here, it means we predict a forced checkmate
        // for White!  We can calculate how many moves it will take
        // using the score.

        int plies = (WHITE_WINS - long(bestmove.score)) / WIN_DELAY_PENALTY;
        userInterface.PredictMate ( plies / 2 );
    }

    if ( timedSearch && trainingEnabled )
    {
        LearnTree tree;
        tree.rememberPosition ( board, bestmove, timeLimit );
    }
}


void ComputerChessPlayer::GetBlackMove (
    ChessBoard &board,
    Move &bestmove )
{
    moveOrder_bestPathFlag = cFALSE;
    board.GenBlackMoves ( rootml, this ); // Generate sorted list of legal moves
    bestmove = rootml.m[0];   // just in case!
    gennodes[0] = generated = rootml.num;

    visited = evaluated = 0;

    for ( int i=0; i < NODES_ARRAY_SIZE; i++ )
        visnodes[i] = gennodes[i] = 0;

    for ( i=0; i < 64*64; i++ )
    {
        whiteHist[i] /= 2;
        blackHist[i] /= 2;
    }

    if ( rootml.num == 1 )
        return;   // only one legal move exists, so just do it!

    if ( openingBookSearchEnabled && BlackOpening ( board, bestmove ) )
    {
		userInterface.ReportSpecial ( "opening" );
        return;   // We found a groovy opening move
    }

    LearnTree tree;
    if ( timedSearch && trainingEnabled && tree.familiarPosition ( board, bestmove, timeLimit, rootml ) )
    {
		char buffer [64];
		sprintf ( buffer, "experience (%6d)", int(bestmove.score) );
		userInterface.ReportSpecial ( buffer );
        return;   // We found a move we know is best from a previous (longer) search
    }

    // See if we can recycle best path info from the previous move's search.
    UINT32 hash = board.Hash();
    int startLevel = 0;
    if ( hash==expectedNextBoardHash && currentBestPath.depth>=2 && prevCompletedLevel>1 )
    {
        // Strip the top two plies because they are in the past!
        for ( int i=2; i <= currentBestPath.depth; i++ )
            currentBestPath.m[i-2] = currentBestPath.m[i];

        currentBestPath.depth -= 2;

        eachBestPathCount = 1;
        eachBestPath[0] = currentBestPath;
        startLevel = prevCompletedLevel - 1;
        rootml.SendToFront ( currentBestPath.m[0] );
        bestmove = rootml.m[0];
    }
    else
    {
        eachBestPathCount = 0;
        currentBestPath.depth = 0;
    }

    nextBestPath[0].depth = 0;
    prevCompletedLevel = 0;
    for ( level=startLevel; !searchAborted && level <= maxlevel; level++ )
    {
        BlackSearchRoot ( board, bestmove );
        if ( !searchAborted )
            prevCompletedLevel = level;

        if ( searchBias )
        {
            MoveList pg;   // list of "pretty good" moves
            pg.num = 0;
            for ( int i=0; i < rootml.num; i++ )
            {
                if ( rootml.m[i].score < bestmove.score + searchBias )
                    pg.AddMove ( rootml.m[i] );
            }

            int r = ChessRandom ( pg.num );
            bestmove = pg.m[r];
        }

        if ( bestmove.score <= WON_FOR_BLACK )
            break;

        if ( bestmove.score >= WON_FOR_WHITE )
            break;
    }

    if ( bestmove.score <= WON_FOR_BLACK &&
         bestmove.score > BLACK_WINS + WIN_DELAY_PENALTY )
    {
        // If we get here, it means we predict a forced checkmate
        // for Black!  We can calculate how many moves it will take
        // using the score.

        int plies = (long(bestmove.score) - BLACK_WINS) / WIN_DELAY_PENALTY;
        userInterface.PredictMate ( plies / 2 );
    }

    if ( timedSearch && trainingEnabled )
    {
        LearnTree tree;
        tree.rememberPosition ( board, bestmove, timeLimit );
    }
}


void ComputerChessPlayer::FindPrevBestPath ( Move m )
{
    // This function tries to find the previous top-level move
    // in the best path table and copy it into currentBestPath.

    for ( int i=0; i < eachBestPathCount; i++ )
    {
        if ( eachBestPath[i].depth > 0 )
        {
            if ( eachBestPath[i].m[0] == m )
            {
                currentBestPath = eachBestPath[i];
                return;
            }
        }
    }

    currentBestPath.depth = 0;   // keep search from using best path
}


BestPath *ComputerChessPlayer::SaveTLMBestPath ( Move move )
{
    // Try to find move in existing TLM BestPaths...

    for ( int i=0; i < eachBestPathCount; i++ )
    {
        if ( eachBestPath[i].m[0] == move )
        {
            eachBestPath[i] = nextBestPath[1];
            eachBestPath[i].m[0] = move;
            return & ( eachBestPath[i] );
        }
    }

    // Need to add a new one!

    if ( eachBestPathCount >= MAX_MOVES )
    {
        ChessFatal ( "BestPath top-level-move overflow!" );
        return 0;
    }

    i = eachBestPathCount++;
    eachBestPath[i] = nextBestPath[1];
    eachBestPath[i].m[0] = move;
    return & ( eachBestPath[i] );
}


void ComputerChessPlayer::FoundBestMove (
    Move bestmove,
    int depth )
{
    if ( depth < MAX_BESTPATH_DEPTH-1 )
    {
        // Insert the bestmove found at this depth...

        nextBestPath[depth].m[depth] = bestmove;

        // Copy best path from layer below to this layer.

        int copyDepth = nextBestPath[depth].depth =
                        nextBestPath[depth+1].depth;

        Move *source = &nextBestPath[depth+1].m[0];
        Move *dest   = &nextBestPath[depth].m[0];

        for ( int d=depth+1; d <= copyDepth; d++ )
        {
            dest[d] = source[d];
        }
    }
}


SCORE ComputerChessPlayer::WhiteSearchRoot (
    ChessBoard &board,
    Move       &bestmove )
{
    SCORE   score;
    SCORE   bestscore = NEGINF;
    SCORE   alpha = MIN_WINDOW;   // Best so far for White
    SCORE   beta  = MAX_WINDOW;   // Best so far for Black

    int  i;
    Move        *move;
    UnmoveInfo   unmove;

    for ( i=0, move=&rootml.m[0];
          !searchAborted && i < rootml.num;
          i++, move++ )
    {
        // If this isn't the first move we are considering, and it
        // is a definite loser, just skip it!

        if ( i>0 && move->score <= WON_FOR_BLACK )
        {
            continue;
        }

        ++visited;
        ++visnodes[0];

        if ( level > 1 )
        {
            userInterface.DisplayCurrentMove ( board, *move, level );
        }

        FindPrevBestPath ( *move );

        userInterface.DebugPly ( 0, board, *move );
        board.MakeWhiteMove ( *move, unmove, cFALSE, cFALSE );

        if ( level > 0 )
        {
            score = BlackSearch ( board, 1, alpha, beta, cTRUE );
            if ( !searchAborted )
            {
                move->score = score;
            }
        }
        else
        {
            score = move->score = BlackQSearch ( board, 1, alpha, beta, cTRUE );
        }
        board.UnmakeWhiteMove ( *move, unmove );

        BestPath *path = 0;
        if ( !searchAborted )
        {
            path = SaveTLMBestPath ( *move );
        }

        if ( searchAborted )
        {
            if ( i > 0 )
            {
                // Invalidate this and all the remaining moves,
                // to keep search bias from picking from them...

                while ( i < rootml.num )
                {
                    rootml.m[i++].score = NEGINF;
                }
            }
        }
        else
        {
            if ( score > bestscore )
            {
                bestmove = *move;
                bestscore = score;
                if ( level > 1 )
                {
                    userInterface.DisplayBestMoveSoFar (
                        board, bestmove, level );

                    if ( path )
                    {
                        userInterface.DisplayBestPath ( board, *path );
                    }
                }
            }
        }

        if ( score > alpha )
        {
            alpha = score;
        }
    }

    rootml.WhiteSort();

    userInterface.DebugExit ( 0, board, bestscore );
    return bestscore;
}


SCORE ComputerChessPlayer::BlackSearchRoot (
    ChessBoard &board,
    Move       &bestmove )
{
    SCORE   score;
    SCORE   bestscore = POSINF;
    SCORE   alpha = MIN_WINDOW;   // Best so far for White
    SCORE   beta  = MAX_WINDOW;   // Best so far for Black

    int  i;
    Move        *move;
    UnmoveInfo   unmove;

    for ( i=0, move=&rootml.m[0];
          !searchAborted && i < rootml.num;
          i++, move++ )
    {
        // If this isn't the first move we are considering, and it
        // is a definite loser, just skip it!

        if ( i>0 && move->score >= WON_FOR_WHITE )
        {
            continue;
        }

        ++visited;
        ++visnodes[0];

        if ( level > 1 )
        {
            userInterface.DisplayCurrentMove ( board, *move, level );
        }

        FindPrevBestPath ( *move );

        userInterface.DebugPly ( 0, board, *move );
        board.MakeBlackMove ( *move, unmove, cFALSE, cFALSE );

        if ( level > 0 )
        {
            score = WhiteSearch ( board, 1, alpha, beta, cTRUE );
            if ( !searchAborted )
            {
                move->score = score;
            }
        }
        else
        {
            score = move->score = WhiteQSearch ( board, 1, alpha, beta, cTRUE );
        }
        board.UnmakeBlackMove ( *move, unmove );

        BestPath *path = 0;
        if ( !searchAborted )
        {
            path = SaveTLMBestPath ( *move );
        }

        if ( searchAborted )
        {
            if ( i > 0 )
            {
                // Invalidate all the remaining moves, to keep search
                // bias from picking from them...

                while ( i < rootml.num )
                {
                    rootml.m[i++].score = POSINF;
                }
            }
        }
        else
        {
            if ( score < bestscore )
            {
                bestmove = *move;
                bestscore = score;
                if ( level > 1 )
                {
                    userInterface.DisplayBestMoveSoFar ( board, bestmove, level );
                    if ( path )
                    {
                        userInterface.DisplayBestPath ( board, *path );
                    }
                }
            }
        }

        if ( score < beta )
        {
            beta = score;
        }
    }

    rootml.BlackSort();

    userInterface.DebugExit ( 0, board, bestscore );
    return bestscore;
}


SCORE ComputerChessPlayer::WhiteSearch (
    ChessBoard  &board,
    int          depth,
    SCORE        alpha,
    SCORE        beta,
    cBOOLEAN     bestPathFlag )
{
    if ( CheckTimeLimit() )
        return NEGINF;

    int          score;
    int          bestscore = NEGINF;
    int          i;
    MoveList     ml;
    UnmoveInfo   unmove;
    Move        *move;
    Move        *bestMove = 0;

    moveOrder_bestPathFlag = bestPathFlag;
    moveOrder_depth = depth;
    board.GenWhiteMoves ( ml, this );
    generated += ml.num;
    if ( depth < NODES_ARRAY_SIZE )
        gennodes[depth] += ml.num;

    if ( depth < MAX_BESTPATH_DEPTH )
        nextBestPath[depth].depth = depth - 1;

    if ( ml.num == 0 )
    {
        // This is the end of the game!
        bestscore = (board.flags & SF_WCHECK) ?
                       (BLACK_WINS + WIN_POSTPONEMENT(depth)) :
                       DRAW;

        userInterface.DebugExit ( depth, board, bestscore );
        return bestscore;
    }
    else if ( board.IsDefiniteDraw(3) )
    {
        bestscore = DRAW;
        userInterface.DebugExit ( depth, board, bestscore );
        return bestscore;
    }

    for ( i=0, move=&ml.m[0]; i < ml.num; i++, move++ )
    {
        ++visited;
        if ( depth < NODES_ARRAY_SIZE )
            ++visnodes[depth];

        cBOOLEAN nextBestPathFlag =
            bestPathFlag &&
            depth <= currentBestPath.depth &&
            *move == currentBestPath.m[depth];

        userInterface.DebugPly ( depth, board, *move );
        board.MakeWhiteMove ( *move, unmove, cFALSE, cFALSE );

        if ( depth < level )
            score = BlackSearch ( board, depth+1, alpha, beta, nextBestPathFlag );
        else
            score = BlackQSearch ( board, depth+1, alpha, beta, nextBestPathFlag );

        board.UnmakeWhiteMove ( *move, unmove );

        if ( score > bestscore )
        {
            bestscore = score;
            bestMove = move;
            FoundBestMove ( *move, depth );
        }

        if ( score >= beta + searchBias )
            break;   // PRUNE: Black has better choices than getting here.

        if ( score > alpha )
            alpha = score;
    }

    if ( bestMove )
    {
        SCORE *h = &whiteHist[bestMove->whiteHash()];
        if ( *h < HASH_HIST_MAX )
        {
            *h += HASH_HIST_FUNC(*h,depth);
        }
    }

    userInterface.DebugExit ( depth, board, bestscore );
    return bestscore;
}


SCORE ComputerChessPlayer::BlackSearch (
    ChessBoard  &board,
    int          depth,
    SCORE        alpha,
    SCORE        beta,
    cBOOLEAN     bestPathFlag )
{
    if ( CheckTimeLimit() )
        return POSINF;

    int          score;
    int          bestscore = POSINF;
    int          i;
    MoveList     ml;
    UnmoveInfo   unmove;
    Move        *move;
    Move        *bestMove = 0;

    moveOrder_bestPathFlag = bestPathFlag;
    moveOrder_depth = depth;
    board.GenBlackMoves ( ml, this );
    generated += ml.num;
    if ( depth < NODES_ARRAY_SIZE )
        gennodes[depth] += ml.num;

    if ( depth < MAX_BESTPATH_DEPTH )
        nextBestPath[depth].depth = depth - 1;

    if ( ml.num == 0 )
    {
        // This is the end of the game!
        bestscore = (board.flags & SF_BCHECK) ?
                       (WHITE_WINS - WIN_POSTPONEMENT(depth)) :
                       DRAW;

        userInterface.DebugExit ( depth, board, bestscore );
        return bestscore;
    }
    else if ( board.IsDefiniteDraw(3) )
    {
        bestscore = DRAW;
        userInterface.DebugExit ( depth, board, bestscore );
        return bestscore;
    }

    for ( i=0, move=&ml.m[0]; i < ml.num; i++, move++ )
    {
        ++visited;
        if ( depth < NODES_ARRAY_SIZE )
            ++visnodes[depth];

        cBOOLEAN nextBestPathFlag =
            bestPathFlag &&
            depth <= currentBestPath.depth &&
            *move == currentBestPath.m[depth];

        userInterface.DebugPly ( depth, board, *move );
        board.MakeBlackMove ( *move, unmove, cFALSE, cFALSE );

        if ( depth < level )
            score = WhiteSearch ( board, depth+1, alpha, beta, nextBestPathFlag );
        else
            score = WhiteQSearch ( board, depth+1, alpha, beta, nextBestPathFlag );

        board.UnmakeBlackMove ( *move, unmove );

        if ( score < bestscore )
        {
            bestscore = score;
            bestMove  = move;
            FoundBestMove ( *move, depth );
        }

        if ( score <= alpha - searchBias )
            break;   // PRUNE: White has better choices than getting here.

        if ( score < beta )
            beta = score;
    }

    if ( bestMove )
    {
        SCORE *h = &blackHist [bestMove->blackHash()];
        if ( *h < HASH_HIST_MAX )
        {
            *h += HASH_HIST_FUNC(*h,depth);
        }
    }

    userInterface.DebugExit ( depth, board, bestscore );
    return bestscore;
}


SCORE ComputerChessPlayer::WhiteQSearch (
    ChessBoard  &board,
    int          depth,
    SCORE        alpha,
    SCORE        beta,
    cBOOLEAN     bestPathFlag )
{
    if ( CheckTimeLimit() )
        return NEGINF;

    SCORE       score;
    SCORE       bestscore = (this->*whiteEval) ( board, depth, beta );
    MoveList    ml;
    UnmoveInfo  unmove;
    int         i;
    Move       *move;

    // Assume we have already hit bottom of next best path...

    if ( depth < MAX_BESTPATH_DEPTH )
        nextBestPath[depth].depth = depth - 1;

    const cBOOLEAN escapeCheck = 
        (board.flags & SF_WCHECK) && depth < level + ESCAPE_CHECK_DEPTH;

    if ( bestscore < beta + searchBias || escapeCheck )
    {
        moveOrder_bestPathFlag = bestPathFlag;
        moveOrder_depth = depth;

        if ( escapeCheck )
            board.GenWhiteMoves ( ml, this );
        else if ( depth <= level + maxCheckDepth )
            GenWhiteCapturesAndChecks ( board, ml );
        else
            board.GenWhiteCaptures ( ml, this );

        if ( depth < NODES_ARRAY_SIZE )
            gennodes[depth] += ml.num;

        generated += ml.num;
        for ( i=0, move=&ml.m[0]; i < ml.num; i++, move++ )
        {
            ++visited;
            if ( depth < NODES_ARRAY_SIZE )
                ++visnodes[depth];

            cBOOLEAN nextBestPathFlag =
                bestPathFlag &&
                depth <= currentBestPath.depth &&
                *move == currentBestPath.m[depth];

            userInterface.DebugPly ( depth, board, *move );
            board.MakeWhiteMove ( *move, unmove, cFALSE, cFALSE );

            if ( (board.flags & SF_BCHECK) || escapeCheck )
                score = BlackSearch ( board, depth+1, alpha, beta, nextBestPathFlag );
            else
                score = BlackQSearch ( board, depth+1, alpha, beta, nextBestPathFlag );

            board.UnmakeWhiteMove ( *move, unmove );

            if ( score > bestscore )
            {
                bestscore = score;
                FoundBestMove ( *move, depth );
            }

            if ( score >= beta + searchBias )
                break;      // PRUNE

            if ( score > alpha )
                alpha = score;
        }
    }

    userInterface.DebugExit ( depth, board, bestscore );
    return bestscore;
}


SCORE ComputerChessPlayer::BlackQSearch (
    ChessBoard  &board,
    int          depth,
    SCORE        alpha,
    SCORE        beta,
    cBOOLEAN     bestPathFlag )
{
    if ( CheckTimeLimit() )
        return POSINF;

    SCORE       score;
    SCORE       bestscore = (this->*blackEval) ( board, depth, alpha );
    MoveList    ml;
    UnmoveInfo  unmove;
    int         i;
    Move       *move;

    if ( depth < MAX_BESTPATH_DEPTH )
        nextBestPath[depth].depth = depth - 1;

    const cBOOLEAN escapeCheck =
        (board.flags & SF_BCHECK) && depth < level + ESCAPE_CHECK_DEPTH;

    if ( bestscore > alpha - searchBias || escapeCheck )
    {
        moveOrder_bestPathFlag = bestPathFlag;
        moveOrder_depth = depth;

        if ( escapeCheck )
            board.GenBlackMoves ( ml, this );
        else if ( depth <= level + maxCheckDepth )
            GenBlackCapturesAndChecks ( board, ml );
        else
            board.GenBlackCaptures ( ml, this );

        if ( depth < NODES_ARRAY_SIZE )
            gennodes[depth] += ml.num;

        generated += ml.num;
        for ( i=0, move=&ml.m[0]; i < ml.num; i++, move++ )
        {
            ++visited;
            if ( depth < NODES_ARRAY_SIZE )
                ++visnodes[depth];

            cBOOLEAN nextBestPathFlag =
                bestPathFlag &&
                depth <= currentBestPath.depth &&
                *move == currentBestPath.m[depth];

            userInterface.DebugPly ( depth, board, *move );
            board.MakeBlackMove ( *move, unmove, cFALSE, cFALSE );

            if ( (board.flags & SF_WCHECK) || escapeCheck )
                score = WhiteSearch ( board, depth+1, alpha, beta, nextBestPathFlag );
            else
                score = WhiteQSearch ( board, depth+1, alpha, beta, nextBestPathFlag );

            board.UnmakeBlackMove ( *move, unmove );

            if ( score < bestscore )
            {
                bestscore = score;
                FoundBestMove ( *move, depth );
            }

            if ( score <= alpha - searchBias )
                break;      // PRUNE

            if ( score < beta )
                beta = score;
        }
    }

    userInterface.DebugExit ( depth, board, bestscore );
    return bestscore;
}


void ComputerChessPlayer::GenWhiteCapturesAndChecks (
    ChessBoard &board,
    MoveList &ml )
{
	PROFILER_ENTER(PX_GENCAPS);
    MoveList all;
    board.GenWhiteMoves ( all, this );

    // Keep only moves which cause check or are captures in 'ml'
    ml.num = 0;
    for ( int i=0; i<all.num; i++ )
    {
        Move move = all.m[i];
        if ( move.source & CAUSES_CHECK_BIT )
            ml.AddMove ( move );
        else if ( move.dest > OFFSET(9,9) )
        {
            int special = move.dest & SPECIAL_MOVE_MASK;
            if ( special != SPECIAL_MOVE_KCASTLE &&
                 special != SPECIAL_MOVE_QCASTLE )
                ml.AddMove ( move );
        }
        else if ( board.board[move.dest] != EMPTY )
            ml.AddMove ( move );
    }
	PROFILER_EXIT();
}


void ComputerChessPlayer::GenBlackCapturesAndChecks (
    ChessBoard &board,
    MoveList &ml )
{
	PROFILER_ENTER(PX_GENCAPS);
    MoveList all;
    board.GenBlackMoves ( all, this );

    // Keep only moves which cause check or are captures in 'ml'
    ml.num = 0;
    for ( int i=0; i<all.num; i++ )
    {
        Move move = all.m[i];
        if ( move.source & CAUSES_CHECK_BIT )
            ml.AddMove ( move );
        else if ( move.dest > OFFSET(9,9) )
        {
            int special = move.dest & SPECIAL_MOVE_MASK;
            if ( special != SPECIAL_MOVE_KCASTLE &&
                 special != SPECIAL_MOVE_QCASTLE )
                ml.AddMove ( move );
        }
        else if ( board.board[move.dest] != EMPTY )
            ml.AddMove ( move );
    }
	PROFILER_EXIT();
}


void ComputerChessPlayer::SetSearchDepth ( int NewSearchDepth )
{
    maxlevel = NewSearchDepth;
    timedSearch = cFALSE;
    searchAborted = cFALSE;
}


void ComputerChessPlayer::SetTimeLimit ( INT32 hundredthsOfSeconds )
{
    if ( hundredthsOfSeconds < 100 )
    {
        ChessFatal ( "Invalid search time for ComputerChessPlayer" );
    }

    maxlevel = NODES_ARRAY_SIZE - 1;
    timedSearch = cTRUE;
    searchAborted = cFALSE;
    timeLimit = hundredthsOfSeconds;
}


void ComputerChessPlayer::AbortSearch()
{
    searchAborted = cTRUE;
}


cBOOLEAN ComputerChessPlayer::CheckTimeLimit()
{
    if ( searchAborted )
    {
        return cTRUE;
    }

    if ( timedSearch )
    {
        if ( ++timeCheckCounter >= timeCheckLimit )
        {
            timeCheckCounter = 0;

            INT32 now = ChessTime();

            if ( now >= stopTime )
            {
                if ( now - prevTime > 100 )
                {
                    timeCheckLimit /= 2;
                }

                return searchAborted = cTRUE;
            }
            else if ( now - prevTime < 10 )   // less than 0.01 sec elapsed?
            {
                timeCheckLimit += 100;
            }

            prevTime = now;
        }
    }

    return cFALSE;
}


void ComputerChessPlayer::SetSearchBias ( SCORE newSearchBias )
{
    if ( newSearchBias >= 0 )
    {
        searchBias = newSearchBias;
    }
    else
    {
        ChessFatal ( "Negative search bias not allowed" );
    }
}


/*--- end of file search.cpp ---*/
