/*==========================================================================

    transpos.cpp  -  Copyright (C) 1996 by Don Cross

    email:   dcross@intersrv.com
    WWW:     http://www.intersrv.com/~dcross/

    Contains transposition table for chess search.

    Revision history:

1996 July 23 [Don Cross]
     Started writing

==========================================================================*/

#include "chess.h"

#if USE_TRANSPOSITION

#define TRANSPOSE_POOL_SIZE   20000

//--------------------------------------------------------------------

TranspositionEntry::TranspositionEntry():
    white_flags ( 0xFFFF ),
    black_flags ( 0xFFFF ),
    next ( 0 )
{
    bestmove.source = bestmove.dest = BYTE(0);
    bestmove.score = 0;

    for ( int i=0; i<64; i++ )
        board[i] = 0;
}


void TranspositionEntry::assign (
    const ChessBoard &_board,
    Move _bestmove,
    int _fwDepth )
{
    white_flags = _board.white_flags;
    black_flags = _board.black_flags;
    bestmove    = _bestmove;
    fwDepth     = _fwDepth;

    BYTE *p = board;
    for ( int y=OFFSET(2,2); y <= OFFSET(2,9); y += NORTH )
    {
        for ( int x=0; x<8; x++ )
        {
            *p++ = SPIECE_INDEX ( _board.board[x+y] );
        }
    }
}


void TranspositionEntry::update (
    Move _bestmove,
    int _fwDepth )
{
    bestmove = _bestmove;
    fwDepth = _fwDepth;
}


int TranspositionEntry::operator== (
    const ChessBoard &_board )
{
    if ( white_flags != _board.white_flags )
        return 0;

    if ( black_flags != _board.black_flags )
        return 0;

    const BYTE *p = board;
    for ( int y=OFFSET(2,2); y <= OFFSET(2,9); y += NORTH )
    {
        for ( int x=0; x<8; x++ )
        {
            if ( *p++ != SPIECE_INDEX ( _board.board[x+y] ) )
                return 0;
        }
    }

    return 1;
}


//--------------------------------------------------------------------


TranspositionTable::TranspositionTable ( unsigned _tableSize ):
    tableSize ( _tableSize ),
    table ( new TranspositionPtr [_tableSize] )
{
    if ( !table || pool.SetNumItems(TRANSPOSE_POOL_SIZE) != DDC_SUCCESS )
        ChessFatal ( "Out of memory in TranspositionTable constructor" );

    for ( unsigned i=0; i < _tableSize; i++ )
    {
        table[i] = 0;
    }
}


TranspositionTable::~TranspositionTable()
{
    if ( table )
    {
        delete[] table;
        table = 0;
    }
}


TranspositionPtr TranspositionTable::find (
    const ChessBoard &_board ) const
{
    unsigned index = hash(_board);

    for ( TranspositionPtr p = table[index]; p; p = p->getNext() )
    {
        if ( *p == _board )
        {
            return p;
        }
    }

    return 0;
}


void TranspositionTable::insert (
    const ChessBoard &_board,
    Move _bestmove,
    int _fwDepth )
{
    unsigned index = hash(_board);

    // See if we can find this board position in the table already.

    TranspositionPtr p;
    for ( p = table[index]; p; p = p->getNext() )
    {
        if ( *p == _board )
        {
            // We found it there already.  Modify if necessary, then leave.
            if ( _fwDepth > p->queryDepth() )
            {
                p->update ( _bestmove, _fwDepth );
            }

            return;
        }
    }

    // If we get here, it means that we need to add the new position.

    p = pool.CheckOut();
    if ( p )
    {
        p->assign ( _board, _bestmove, _fwDepth );
        p->linkTo ( table[index] );
        table[index] = p;
    }
}


void TranspositionTable::reset()
{
    // Return all entries to the freelist, which makes the table empty.

    for ( int i=0; i < tableSize; i++ )
    {
        TranspositionPtr p = table[i];
        while ( p )
        {
            TranspositionPtr next = p->getNext();
            pool.CheckIn(p);
            p = next;
        }
        table[i] = 0;
    }
}


unsigned TranspositionTable::hash ( const ChessBoard &_board ) const
{
    return unsigned ( _board.Hash() % UINT32(tableSize) );
}


//--------------------------------------------------------------------


#endif // USE_TRANSPOSITION

/*--- end of file transpos.cpp ---*/