/*==========================================================================

    uios2pm.cpp  -  Copyright (C) 1993-1996 by Don Cross
    email: dcross@intersrv.com
    WWW:   http://www.intersrv.com/~dcross/

    Contains a specialized user interface class for new chess.
    This one is for OS/2 Presentation Manager.

    Revision history:

1993 July 14 [Don Cross]
     Started writing!

1993 December 31 [Don Cross]
     Added code to save the whole game to the file 'chess.gam'
     each time the board is displayed in ChessUI_os2_pm::DrawBoard().
     Changed the special "save game" command to write to 'save.gam'.
     This way, it is easier to quit from a game, then restore its
     last position some other time.

1994 January 15 [Don Cross]
     Added support for allowing the user to re-start the game.

1994 January 30 [Don Cross]
     Giving back SPECIAL_MOVE_EDIT signals now when board edit commands
     are processed.

==========================================================================*/

#define  INCL_PM
#define  INCL_DOSPROCESS
#include <os2.h>

#include <stdio.h>

#include "chess.h"        // Universal chess header file
#include "os2chess.h"     // Resource definitions
#include "os2pmch.h"      // Special stuff for PM version of chess

ChessUI_os2_pm::ChessUI_os2_pm ( HWND _clientHwnd, HWND _frameHwnd ):
   hwnd ( _clientHwnd ),
   frameHwnd ( _frameHwnd )
{
}


ChessUI_os2_pm::~ChessUI_os2_pm()
{
}


ChessPlayer *ChessUI_os2_pm::CreatePlayer ( ChessSide side )
{
   ChessPlayer *player = 0;

   static int doneWhite = 0;
   static int whiteIsHuman = 0;
   static int blackIsHuman = 0;

   int searchDepth;
   int searchBias;
   int useSeconds;

   DefPlayerInfo::Type playerType;

   if ( side == SIDE_WHITE )
   {
      doneWhite = 1;
      playerType = DefPlayer.whiteType;
      searchDepth = DefPlayer.whiteThinkTime;
      searchBias = DefPlayer.whiteSearchBias;
      useSeconds = DefPlayer.whiteUseSeconds;
   }
   else
   {
      playerType = DefPlayer.blackType;
      searchDepth = DefPlayer.blackThinkTime;
      searchBias = DefPlayer.blackSearchBias;
      useSeconds = DefPlayer.blackUseSeconds;
   }

   switch ( playerType )
   {
      case DefPlayerInfo::computerPlayer:
      {
         ComputerChessPlayer *puter = new ComputerChessPlayer (*this);
         if ( puter )
         {
            player = puter;
            if ( useSeconds )
            {
               puter->SetTimeLimit ( INT32(searchDepth) * 100 );
            }
            else
            {
               puter->SetSearchDepth ( searchDepth );
            }
            puter->SetSearchBias ( searchBias );
         }
      }
      break;

      case DefPlayerInfo::humanPlayer:
      {
         player = new HumanChessPlayer (*this);
         if ( side == SIDE_WHITE )
         {
            whiteIsHuman = 1;
         }
         else
         {
            blackIsHuman = 1;
         }
      }
      break;
   }

   if ( doneWhite && side==SIDE_BLACK && !whiteIsHuman && blackIsHuman )
   {
      TheBoardDisplayBuffer.setView ( 0 );
   }

   return player;
}


static void SaveGameToFile ( const char        *filename,
                             const ChessBoard  &board,
                             cBOOLEAN           shouldBeepIfSuccessful )
{
   FILE *f = fopen ( filename, "wb" );

   if ( f )
   {
      int ply = board.GetCurrentPlyNumber();

      for ( int p=0; p < ply; p++ )
      {
         Move move = board.GetPastMove(p);

         fwrite ( &move, sizeof(Move), 1, f );
      }

      fclose(f);

      if ( shouldBeepIfSuccessful )
      {
         ChessBeep ( 1000, 100 );
      }
   }
}


cBOOLEAN ChessUI_os2_pm::ReadMove ( ChessBoard &board,
                                    int        &source,
                                    int        &dest )
{
   // Let the client window know we are starting to read
   // a human player's move...
   WinSendMsg ( hwnd, WM_DDC_ENTER_HUMAN_MOVE, 0, 0 );

   TheBoardDisplayBuffer.startReadingMove ( board.WhiteToMove() );

   cBOOLEAN itIsTimeToCruise = cFALSE;
   Move edit;
   edit.score = 0;

   while ( TheBoardDisplayBuffer.isReadingMove() )
   {
      DosSleep ( 20 );

      if ( Global_BoardEditRequest.isValid )
      {
         SQUARE square = Global_BoardEditRequest.square;
         int    x = Global_BoardEditRequest.x;
         int    y = Global_BoardEditRequest.y;

         board.SetSquareContents ( square, x, y );
         edit.dest = dest = SPECIAL_MOVE_EDIT | ConvertSquareToNybble(square);
         edit.source = source = OFFSET(x+2,y+2);

         board.SaveSpecialMove ( edit );

         itIsTimeToCruise = cTRUE;

         Global_BoardEditRequest.isValid = cFALSE;
      }
      else if ( Global_ClearBoardFlag )
      {
         USHORT choice = WinMessageBox
            ( HWND_DESKTOP,
              frameHwnd,
              "Do you really want to clear the board?",
              CHESS_PROGRAM_NAME,
              0,
              MB_MOVEABLE | MB_YESNO | MB_QUERY | MB_APPLMODAL );

         if ( choice == MBID_YES )
         {
            board.ClearEverythingButKings();
            itIsTimeToCruise = cTRUE;
            edit.source = source = 0;        // signal to clear everything!
            edit.dest = dest = SPECIAL_MOVE_EDIT;
            board.SaveSpecialMove ( edit );
         }

         Global_ClearBoardFlag = cFALSE;
      }
      else if ( Global_ResetGameFlag )
      {
         Global_ResetGameFlag = cFALSE;

         USHORT choice = WinMessageBox
            ( HWND_DESKTOP,
              frameHwnd,
              "Do you really want to start a new game?",
              CHESS_PROGRAM_NAME,
              0,
              MB_MOVEABLE | MB_YESNO | MB_QUERY | MB_APPLMODAL );

         if ( choice == MBID_YES )
         {
            board.Init();
            itIsTimeToCruise = cTRUE;
            dest = SPECIAL_MOVE_NULL;
            source = 0;
         }
      }

      if ( itIsTimeToCruise )
      {
         TheBoardDisplayBuffer.update ( board );
         DrawBoard ( board );
         WinSendMsg ( hwnd, WM_DDC_LEAVE_HUMAN_MOVE, 0, 0 );
         return cTRUE;
      }
   }

   TheBoardDisplayBuffer.copyMove ( source, dest );

   WinSendMsg ( hwnd, WM_DDC_LEAVE_HUMAN_MOVE, 0, 0 );

   return cTRUE;
}


SQUARE ChessUI_os2_pm::PromotePawn ( int, ChessSide side )
{
   volatile SQUARE prom = EMPTY;

   WinPostMsg ( hwnd,
                WM_DDC_PROMOTE_PAWN,
                MPARAM(&prom),
                MPARAM(side) );

   while ( prom == EMPTY )
   {
      DosSleep ( 100 );
   }

   return prom;
}


void ChessUI_os2_pm::DrawBoard ( const ChessBoard &board )
{
   TheBoardDisplayBuffer.update ( board );
   TheBoardDisplayBuffer.deselectSquare();
   TheBoardDisplayBuffer.freshenBoard();
   SaveGameToFile ( GameSave_filename, board, cFALSE );
   DosSleep ( 100 );   // Helps the board to "breathe"
}


void ChessUI_os2_pm::Resign ( ChessSide iGiveUp )
{
   gameReport report;

   report.winner = (iGiveUp == SIDE_WHITE) ? SIDE_BLACK : SIDE_WHITE;
   report.resignation = cTRUE;

   WinPostMsg ( hwnd,
                WM_DDC_GAME_RESULT,
                MPARAM(&report),
                MPARAM(0) );

   WinSetWindowText ( frameHwnd, TitleBarText_GameOver );
}


void ChessUI_os2_pm::ReportEndOfGame ( ChessSide winner )
{
   gameReport report;

   report.winner = winner;
   report.resignation = cFALSE;

   WinPostMsg ( hwnd,
                WM_DDC_GAME_RESULT,
                MPARAM(&report),
                MPARAM(0) );

   WinSetWindowText ( frameHwnd, TitleBarText_GameOver );
}


void ChessUI_os2_pm::PredictMate ( int n )
{
   DosSleep ( 200 );
   int messageProcessed = 0;

   WinPostMsg ( hwnd,
                WM_DDC_PREDICT_MATE,
                MPARAM(n),
                MPARAM(&messageProcessed) );

   while ( !messageProcessed )
   {
      DosSleep ( 200 );
   }
}


static void FlashSquare ( int ofs, SQUARE before, SQUARE after )
{
   const int x = XPART(ofs) - 2;
   const int y = YPART(ofs) - 2;

   TheBoardDisplayBuffer.setSquareContents ( x, y, after );
   TheBoardDisplayBuffer.freshenSquare ( x, y );

   for ( int i=0; i < 2; i++ )
   {
      DosSleep ( 180 );

      TheBoardDisplayBuffer.setSquareContents ( x, y, before );
      TheBoardDisplayBuffer.freshenSquare ( x, y );
      DosSleep ( 180 );

      TheBoardDisplayBuffer.setSquareContents ( x, y, after );
      TheBoardDisplayBuffer.freshenSquare ( x, y );
   }
}


void ChessUI_os2_pm::DisplayMove ( ChessBoard &board, Move move )
{
   ChessBeep ( 1400, 50 );

   int ofs1, ofs2;
   move.actualOffsets ( board, ofs1, ofs2 );

   // Show it being picked up...
   FlashSquare ( ofs1, board.GetSquareContents(ofs1), EMPTY );

   // Temporarily make move in board to show it being put down...
   UnmoveInfo unmove;
   if ( board.WhiteToMove() )
   {
      board.MakeWhiteMove ( move, unmove, cTRUE, cTRUE );
   }
   else
   {
      board.MakeBlackMove ( move, unmove, cTRUE, cTRUE );
   }

   TheBoardDisplayBuffer.update ( board );
   TheBoardDisplayBuffer.freshenSquare ( XPART(ofs1)-2, YPART(ofs1)-2 );
   FlashSquare ( ofs2, EMPTY, board.GetSquareContents(ofs2) );

   // Undo the move because it's done officially by the ChessGame object.
   if ( board.BlackToMove() )
   {
      board.UnmakeWhiteMove ( move, unmove );
   }
   else
   {
      board.UnmakeBlackMove ( move, unmove );
   }
}


void ChessUI_os2_pm::RecordMove ( ChessBoard  &board,
                                  Move         move,
                                  INT32        thinkTime )
{
   char moveString [MAX_MOVE_STRLEN + 1];
   char displayString [30 + MAX_MOVE_STRLEN + 1];
   int  wm = board.WhiteToMove();

   FormatChessMove ( board, move, moveString );
   sprintf ( displayString,
             "%s: %s (%0.2lf)",
             wm ? "White" : "Black",
             moveString,
             double(thinkTime) / 100.0 );

   WinSetDlgItemText ( hwnd,
                       wm ? STATIC_ID_WHITES_MOVE : STATIC_ID_BLACKS_MOVE,
                       displayString );

   if ( wm )
   {
      // It will be black's turn real soon now.
      WinSetWindowText ( frameHwnd, TitleBarText_BlacksTurn );
   }
   else
   {
      // It will be white's turn real soon now.
      WinSetWindowText ( frameHwnd, TitleBarText_WhitesTurn );
   }
}


void ChessUI_os2_pm::DisplayBestMoveSoFar ( const ChessBoard &board,
                                            Move              bestMove,
                                            int )
{
   char moveString [MAX_MOVE_STRLEN + 1];
   char displayString [20 + MAX_MOVE_STRLEN + 1];

   FormatChessMove ( board, bestMove, moveString );
   sprintf ( displayString,
             "%6d [%s]",
             int ( bestMove.score ),
             moveString );

   WinSetDlgItemText ( hwnd, STATIC_ID_BEST_SO_FAR, displayString );
}


void ChessUI_os2_pm::DisplayCurrentMove ( const ChessBoard &board,
                                          Move              move,
                                          int               level )
{
   extern cBOOLEAN Global_ViewDebugFlag;

   if ( Global_ViewDebugFlag )
   {
      char moveString [MAX_MOVE_STRLEN + 1];
      char displayString [20 + MAX_MOVE_STRLEN + 1];
   
      FormatChessMove ( board, move, moveString );
      sprintf ( displayString,
                "%2d: %6d [%s]",
                level,
                int ( move.score ),
                moveString );
   
      WinSetDlgItemText ( hwnd, STATIC_ID_CURRENT_MOVE, displayString );
   }
}


void ChessUI_os2_pm::ReportComputerStats ( INT32   thinkTime,
                                           UINT32  nodesVisited,
                                           UINT32  nodesEvaluated,
                       UINT32  /*nodesGenerated*/,
                                           int     fwSearchDepth,
                                           UINT32  nvis [NODES_ARRAY_SIZE],
                                           UINT32  ngen [NODES_ARRAY_SIZE] )
{
   char displayString [80];

   sprintf ( displayString, "visited=%lu", long(nodesVisited) );
   WinSetDlgItemText ( hwnd, STATIC_ID_NODES_VISITED, displayString );

   sprintf ( displayString, "evaluated=%lu", long(nodesEvaluated) );
   WinSetDlgItemText ( hwnd, STATIC_ID_NODES_EVALUATED, displayString );

   if ( thinkTime > 0 )
   {
      sprintf ( displayString, "vis/sec=%0.0lf",
                double(nodesVisited) / double(thinkTime) * 100.0 );
      WinSetDlgItemText ( hwnd, STATIC_ID_VIS_PER_SECOND, displayString );

      sprintf ( displayString, "eval/sec=%0.0lf",
                double(nodesEvaluated) / double(thinkTime) * 100.0 );
      WinSetDlgItemText ( hwnd, STATIC_ID_EVAL_PER_SECOND, displayString );
   }
   else
   {
      WinSetDlgItemText ( hwnd, STATIC_ID_VIS_PER_SECOND, "" );
      WinSetDlgItemText ( hwnd, STATIC_ID_EVAL_PER_SECOND, "" );
   }

   UINT32  fwvis = 0;
   UINT32  fwgen = 0;

   for ( int i=0; i < fwSearchDepth; i++ )
   {
      fwvis += nvis[i];
      fwgen += ngen[i];
   }

   if ( fwgen > 0 )
   {
      sprintf ( displayString, "fwvis/fwgen=%0.4lf",
                double(fwvis) / double(fwgen) );

      WinSetDlgItemText ( hwnd, STATIC_ID_FWPR, displayString );
   }
   else
   {
      WinSetDlgItemText ( hwnd, STATIC_ID_FWPR, "" );
   }
}


/*--- end of file uios2pm.cpp ---*/
