/*==========================================================================

      uistdio.cpp   -   Copyright (C) 1993-1996 by Don Cross
      email: dcross@intersrv.com
      WWW:   http://www.intersrv.com/~dcross/

      Contains the stdio.h portable chess user interface class
      (ChessUI_stdio).

      Revision history:

1996 July 23 [Don Cross]
     Added ReportTranspositionStats member function.

1996 July 21 [Don Cross]
     Added "time" command to change ComputerChessPlayer time limit in middle of game.

1996 July 20 [Don Cross]
     Added the "rotate" command to show the board rotated on the screen.

1995 December 30 [Don Cross]
     Fixed bug in ChessUI_stdio::DisplayBestPath.
     Was not displaying path[depth]; instead, was
     stopping at depth-1.

1995 July 13 [Don Cross]
     Improved display of current move and best move so far.

1995 April 17 [Don Cross]
     Added the ability to enable/disable showing eval scores from UI.
     By default, this ability will be disabled (ChessUI_stdio::showScores == cFALSE),
     so that opponents cannot glean unfair information from the UI.
     For debug purposes, though, it can be enabled.

1995 March 26 [Don Cross]
     Added the command line option "-s <MaxTimePerMove>", which causes Chenard to play 
     a game against itself and save the result in a unique file with filename of the
     form "SELFxxxx.TXT", where xxxx is a four-digit decimal expression of a 
     positive integer.  (Leading zeroes are used when necessary.)

1995 February 26 [Don Cross]
     Added the following UI commands: new, clear, edit.
     new:    Resets the board to start-of-game.
     clear:  Removes every piece from the board except the two kings.
     edit:   Allows a piece to be placed or removed from a single square on the board.

1995 February 12 [Don Cross]
     Added support for showing both white's and black's view 
     of the board.

1994 November 28 [Don Cross]
     Added simple game log functionality.

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1993 October 25 [Don Cross]
     Using timed search instead of fixed full-width search.

==========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "chess.h"
#include "uistdio.h"
#include "lrntree.h"

const char * LogFilename = "gamelog.txt";

ChessUI_stdio::ChessUI_stdio():
    ChessUI(),
    whiteIsHuman ( cFALSE ),
    blackIsHuman ( cFALSE ),
    showScores ( cFALSE ),
    numPlayersCreated ( 0 ),
    rotateBoard ( cFALSE ),
    whitePlayer ( 0 ),
    blackPlayer ( 0 )
{
    remove ( LogFilename );
}


ChessUI_stdio::~ChessUI_stdio()
{
}


ChessPlayer *ChessUI_stdio::CreatePlayer ( ChessSide side )
{
    char *sideName;

    switch ( side )
    {
        case SIDE_WHITE:
            sideName = "White";
            break;

        case SIDE_BLACK:
            sideName = "Black";
            break;

        default:
            ChessFatal ( "Invalid call to ChessUI_stdio::CreatePlayer()" );
    }

    printf ( "Should %s be played by Human or Computer (h/c)? ", sideName );

    ChessPlayer *NewPlayer = 0;
    char UserString[80];

    while ( fgets ( UserString, sizeof(UserString), stdin ) )
    {
        if ( UserString[0] == 'h' || UserString[0] == 'H' )
        {
            NewPlayer = new HumanChessPlayer ( *this );
            if ( side == SIDE_WHITE )
            {
                whiteIsHuman = cTRUE;
            }
            else
            {
                blackIsHuman = cTRUE;
            }
            break;
        }
        else if ( UserString[0] == 'c' || UserString[0] == 'C' )
        {
            ComputerChessPlayer *GoodGuy = new ComputerChessPlayer ( *this );
            NewPlayer = GoodGuy;
            if ( GoodGuy )
            {
                printf ( "Enter think time in seconds [10]: " );
                UserString[0] = '\0';
                fgets ( UserString, sizeof(UserString), stdin );

                if ( UserString[0] == 'p' )
                {
                    int PliesDeep = 0;
                    if ( sscanf ( UserString+1, "%d", &PliesDeep ) != 1 )
                    {
                        PliesDeep = 3;
                    }
                    GoodGuy->SetSearchDepth ( PliesDeep );
                    GoodGuy->SetSearchBias ( 1 );
                }
                else
                {
                    int ThinkTime;
                    if ( !(sscanf ( UserString, "%d", &ThinkTime ) == 1 &&
                           ThinkTime > 0) )
                    {
                        ThinkTime = 10;
                    }
                    GoodGuy->SetTimeLimit ( INT32(ThinkTime) * INT32(100) );
                    GoodGuy->SetSearchBias ( 1 );
                }
            }
            break;
        }
        else
        {
            printf ( "\n??? Please enter 'h' or 'c': " );
        }
    }

    if ( !NewPlayer )
    {
        ChessFatal ( "Failure to create a ChessPlayer object!" );
    }

    if ( ++numPlayersCreated == 2 )
    {
        if ( !whiteIsHuman && !blackIsHuman )
        {
            // The computer is playing against itself.
            // Go ahead and enable display of min-max scores.

            showScores = cTRUE;
        }
    }

    if ( side == SIDE_WHITE )
        whitePlayer = NewPlayer;
    else
    blackPlayer = NewPlayer;

    return NewPlayer;
}


static void TrimString ( char *s )
{
    while ( *s )
    {
        if ( *s == '\r' || *s == '\n' )
        {
            *s = '\0';
        }
        else
        {
            ++s;
        }
    }
}


cBOOLEAN ChessUI_stdio::ReadMove ( ChessBoard &board,
                                   int &source, int &dest )
{
    char s[80];
    char Prompt[80];
    int newTimeLimit = 0;

    sprintf ( Prompt, "%s move> ", board.WhiteToMove() ? "White" : "Black" );
    printf ( "%s", Prompt );

    while ( fgets ( s, sizeof(s), stdin ) )
    {
        char temp[20];

        TrimString(s);

        if ( s[0] >= 'a' &&
             s[0] <= 'h' &&
             s[1] >= '1' &&
             s[1] <= '8' &&
             s[2] >= 'a' &&
             s[2] <= 'h' &&
             s[3] >= '1' &&
             s[3] <= '8' )
        {
            source = OFFSET ( s[0]-'a'+2, s[1]-'1'+2 );
            dest   = OFFSET ( s[2]-'a'+2, s[3]-'1'+2 );
            return cTRUE;
        }
        else if ( strcmp ( s, "quit" ) == 0 )    // forces an immediate draw
        {
        LearnTree tree;
            tree.learnFromGame ( board, SIDE_NEITHER );
            return cFALSE;
        }
        else if ( strcmp ( s, "resign" ) == 0 ) // speeds teaching of winning behavior
        {
            ChessSide winner = board.WhiteToMove() ? SIDE_BLACK : SIDE_WHITE;
        LearnTree tree;
            tree.learnFromGame ( board, winner );
            return cFALSE;
        }
        else if ( strcmp ( s, "iwin" ) == 0 )   // sounds funny, but important for teaching
        {
            ChessSide winner = board.WhiteToMove() ? SIDE_WHITE : SIDE_BLACK;
        LearnTree tree;
            tree.learnFromGame ( board, winner );
            return cFALSE;
        }
        else if ( strcmp ( s, "clear" ) == 0 )
        {
            board.ClearEverythingButKings();
            DrawBoard ( board );
        }
        else if ( strcmp ( s, "new" ) == 0 )
        {
            board.Init();
            DrawBoard ( board );
        }
        else if ( strcmp ( s, "rotate" ) == 0 )
        {
            rotateBoard = !rotateBoard;
            DrawBoard ( board );
        }
        else if ( sscanf ( s, "load %s", temp ) == 1 )
        {
            cBOOLEAN imWhite = board.WhiteToMove();
            if ( LoadGame(board,temp) )
            {
                if ( board.WhiteToMove() != imWhite )
                {
                    // Make a NULL move to get things rolling.
                    // (kind of like "pass")

                    source = dest = 0;
                }
                printf ( "Successfully loaded game from file '%s'\n", temp );
                DrawBoard ( board );
                return cTRUE;
            }
            else
            {
                fprintf ( stderr, "Error opening file '%s' for read!\x07\n", temp );
            }
        }
        else if ( sscanf ( s, "save %s", temp ) == 1 )
        {
            if ( SaveGame ( board, temp ) )
            {
                printf ( "Game successfully saved to file '%s'\n", temp );
            }
            else
            {
                fprintf ( stderr, "Error opening file '%s' for write!\x07\n", temp );
            }
        }
        else if ( strcmp ( s, "scores" ) == 0 )
        {
            showScores = !showScores;
            printf ( "Display of min-max scores and best path info now %s.\n",
                (showScores ? "ENABLED" : "DISABLED") );
        }
        else if ( sscanf ( s, "time %d", &newTimeLimit ) == 1 )
        {
            if ( newTimeLimit > 0 )
            {
                if ( !whiteIsHuman && whitePlayer )
                {
                    ComputerChessPlayer *cp = (ComputerChessPlayer *)whitePlayer;
                    cp->SetTimeLimit ( INT32(100) * INT32(newTimeLimit) );
                    printf ( "Set White's search time limit to %d seconds\n", newTimeLimit );
                }

                if ( !blackIsHuman && blackPlayer )
                {
                    ComputerChessPlayer *cp = (ComputerChessPlayer *)blackPlayer;
                    cp->SetTimeLimit ( INT32(100) * INT32(newTimeLimit) );
                    printf ( "Set Black's search time limit to %d seconds\n", newTimeLimit );
                }
            }
            else
            {
                printf ( "!!! Ignoring invalid time limit\n" );
            }
        }
        else if ( sscanf ( s, "edit %[a-zA-Z1-8.]", temp ) == 1 && strlen(temp)==3 )
        {
            int x = temp[1] - 'a';
            int y = temp[2] - '1';
            if ( x >= 0 && x <= 7 && y >= 0 && y <= 7 )
            {
                SQUARE square = 0xFFFFFFFF;
                switch ( temp[0] )
                {
                    case 'P':   square = WPAWN;     break;
                    case 'N':   square = WKNIGHT;   break;
                    case 'B':   square = WBISHOP;   break;
                    case 'R':   square = WROOK;     break;
                    case 'Q':   square = WQUEEN;    break;
                    case 'K':   square = WKING;     break;
                    case 'p':   square = BPAWN;     break;
                    case 'n':   square = BKNIGHT;   break;
                    case 'b':   square = BBISHOP;   break;
                    case 'r':   square = BROOK;     break;
                    case 'q':   square = BQUEEN;    break;
                    case 'k':   square = BKING;     break;
                    case '.':   square = EMPTY;     break;
                }

                if ( square != 0xFFFFFFFF )
                {
                    board.SetSquareContents ( square, x, y );
                    Move edit;
                    edit.dest = dest = SPECIAL_MOVE_EDIT | ConvertSquareToNybble(square);
                    edit.source = source = OFFSET(x+2,y+2);
                    board.SaveSpecialMove ( edit );
                    return cTRUE;
                }
                else
                {
                    printf ( "Invalid square contents in edit command\n" );
                }
            }
            else
            {
                printf ( "Invalid coordinates in edit command\n" );
            }
        }
        else if ( ParseFancyMove(s,board,source,dest) )
        {
            return cTRUE;
        }
        else
        {
            printf ( "??? " );
        }

        printf ( "%s", Prompt );
        fflush ( stdout );
        fflush ( stdin );
    }

    return cFALSE;
}



void ChessUI_stdio::DrawBoard ( const ChessBoard &board )
{
    cBOOLEAN whiteView = cFALSE;
    if ( board.WhiteToMove() )
    {
        whiteView = whiteIsHuman;
    }
    else
    {
        whiteView = !blackIsHuman;
    }

    if ( rotateBoard )
    {
    whiteView = !whiteView;
    }

    int x, y;

    printf ( "\n    +--------+\n" );

    if ( whiteView )
    {
        for ( y=7; y >= 0; y-- )
        {
            printf ( "    |" );
            for ( x=0; x <= 7; x++ )
            {
                SQUARE square = board.GetSquareContents ( x, y );
                printf ( "%c", PieceRepresentation(square) );
            }

            printf ( "|%d\n", y + 1 );
        }
        printf ( "    +--------+\n" );
        printf ( "     abcdefgh\n\n\n" );
    }
    else
    {
        for ( y=0; y <= 7; y++ )
        {
            printf ( "    |" );
            for ( x=7; x >= 0; x-- )
            {
                SQUARE square = board.GetSquareContents ( x, y );
                printf ( "%c", PieceRepresentation(square) );
            }

            printf ( "|%d\n", y + 1 );
        }
        printf ( "    +--------+\n" );
        printf ( "     hgfedcba\n\n\n" );
    }
}


void ChessUI_stdio::RecordMove ( ChessBoard  &board,
                                 Move         move,
                                 INT32        thinkTime )
{
    char movestr [MAX_MOVE_STRLEN + 1];
    FormatChessMove ( board, move, movestr );
    int whiteMove = board.WhiteToMove();
    printf ( "\n%s%s  (%0.2lf seconds)\n",
    whiteMove ? "White: " : "Black: ",
    movestr,
    double(thinkTime) / 100.0 );

    FILE *log = fopen ( LogFilename, "at" );
    if ( log )
    {
        static unsigned MoveNumber = 1;
        if ( whiteMove )
        {
            fprintf ( log, "%4u. %-20s", MoveNumber++, movestr );
        }
        else
        {
            fprintf ( log, "%s\n", movestr );
        }

        fclose (log);
    }
    else
    {
        fprintf ( stderr, "*** Warning:  Cannot append to file '%s'\n", LogFilename );
    }
}


void ChessUI_stdio::DisplayMove ( ChessBoard &, Move )
{
}


void ChessUI_stdio::ReportEndOfGame ( ChessSide winner )
{
    switch ( winner )
    {
        case SIDE_WHITE:
            printf ( "White wins.\n" );
            break;

        case SIDE_BLACK:
            printf ( "Black wins.\n" );
            break;

        case SIDE_NEITHER:
            printf ( "Game over.\n" );
            break;

        default:
            printf ( "The game is over, and I don't know why!\n" );
    }
}


static int ConvertCharToPromPiece ( char x )
{
    switch ( x )
    {
        case 'b':
        case 'B':
            return B_INDEX;

        case 'n':
        case 'N':
            return N_INDEX;

        case 'r':
        case 'R':
            return R_INDEX;

        case 'q':
        case 'Q':
            return Q_INDEX;
    }

    return 0;
}


static int AskPawnPromote()
{
    char UserString[80];

    printf ( "Promote pawn to what kind of piece (Q,R,B,N)? " );

    while ( fgets ( UserString, sizeof(UserString), stdin ) )
    {
        int piece = ConvertCharToPromPiece ( UserString[0] );
        if ( piece )
        {
            return piece;
        }

        printf ( "Try again (Q,R,B,N): " );
    }

    ChessFatal ( "Hit EOF on stdin in AskPawnPromote()" );
    return 0;   // make the compiler happy
}



SQUARE ChessUI_stdio::PromotePawn ( int, ChessSide )
{
    return AskPawnPromote();
}


void ChessUI_stdio::DisplayCurrentMove ( const ChessBoard &b,
                                         Move              m,
                                         int               level )
{
    char moveString [MAX_MOVE_STRLEN+1];
    FormatChessMove ( b, m, moveString );
    if ( showScores )
    {
        printf ( 
            "\rdepth=%d  s=%6d m=[%s]               ",
            level,
            m.score,
            moveString );
    }
    else
    {
        printf ( 
            "\rdepth=%d  m=[%s]               ",
            level,
            moveString );
    }
}


void ChessUI_stdio::DisplayBestMoveSoFar ( const ChessBoard &b, 
                                           Move  m,
                                           int   level )
{
    char moveString [MAX_MOVE_STRLEN+1];
    FormatChessMove ( b, m, moveString );

    if ( showScores )
    {
        printf ( "\rBest so far: depth=%d  score=%d  move=[%s]              \n",
                level,
                m.score,
                moveString );
    }
    else
    {
        printf ( "\rBest so far: depth=%d  move=[%s]                \n",
                level,
                moveString );
    }
}


void ChessUI_stdio::DisplayBestPath ( const ChessBoard &_board,
                                      const BestPath &path )
{
    if ( showScores )
    {
        char moveString [MAX_MOVE_STRLEN + 1];
        ChessBoard board = _board;

        printf ( "{ " );
        for ( int i=0; i <= path.depth; i++ )
        {
            Move move = path.m[i];
            UnmoveInfo unmove;

            FormatChessMove ( board, move, moveString );
            board.MakeMove ( move, unmove );

            if ( i > 0 )
            {
                printf ( ", " );
            }
            printf ( "%s", moveString );
        }
        printf ( " }\n\n" );
    }
}


void ChessUI_stdio::PredictMate ( int numMoves )
{
   printf ( "Mate in %d\n", numMoves );
}


void ChessUI_stdio::ReportComputerStats ( 
    INT32   /*thinkTime*/,
    UINT32  nodesVisited,
    UINT32  nodesEvaluated,
    UINT32  nodesGenerated,
    int     /*fwSearchDepth*/,
    UINT32  /*vis*/ [NODES_ARRAY_SIZE],
    UINT32  /*gen*/ [NODES_ARRAY_SIZE] )
{
    printf ( 
        "\ngenerated=%lu, visited=%lu, evaluated=%lu\n",
        nodesGenerated, 
        nodesVisited, 
        nodesEvaluated );
}


void ChessFatal ( const char *message )
{
    fprintf ( stderr, "Fatal chess error: %s\n", message );
    exit(1);
}


/*--- end of file uistdio.cpp ---*/
