/*==========================================================================

    uiwin32.cpp  -  Copyright (C) 1993-1996 by Don Cross
    email: dcross@intersrv.com
    WWW:   http://www.intersrv.com/~dcross/

    Contains a specialized user interface class for new chess.
    This one is for OS/2 Presentation Manager.

    Revision history:

1993 July 14 [Don Cross]
     Started writing!

1993 December 31 [Don Cross]
     Added code to save the whole game to the file 'chess.gam'
     each time the board is displayed in ChessUI_win32_gui::DrawBoard().
     Changed the special "save game" command to write to 'save.gam'.
     This way, it is easier to quit from a game, then restore its
     last position some other time.

1994 January 15 [Don Cross]
     Added support for allowing the user to re-start the game.

1994 January 30 [Don Cross]
     Giving back SPECIAL_MOVE_EDIT signals now when board edit commands
     are processed.

1996 July 29 [Don Cross]
     Starting porting from uios2pm.cpp to Win32 environment.

1996 August 28 [Don Cross]
     Added automatic benchmark facility.

1997 January 30 [Don Cross]
     Adding support for undo move and redo move.

1997 March 8 [Don Cross]
	Added option to allow computer to resign.
	Added command for human to resign.

==========================================================================*/

#include <stdio.h>

#include "chess.h"
#include "winchess.h"
#include "winguich.h"
#include "profiler.h"


cBOOLEAN Global_UserResign = cFALSE;
cBOOLEAN Global_SpeakMovesFlag = cTRUE;
extern int Global_AnalysisType;


ChessUI_win32_gui::ChessUI_win32_gui ( HWND _hwnd ):
    hwnd ( _hwnd ),
    whiteIsHuman (cFALSE),
    blackIsHuman (cFALSE),
    whitePlayer (0),
    blackPlayer (0),
    tacticalBenchmarkMode (cFALSE),
    benchmarkPlayer (0),
    enableMateAnnounce (cTRUE)
{
}


ChessUI_win32_gui::~ChessUI_win32_gui()
{
}


ChessPlayer *ChessUI_win32_gui::CreatePlayer ( ChessSide side )
{
   static int doneWhite = 0;

   ChessPlayer *player = 0;
   int searchDepth;
   int searchBias;
   int useSeconds;

   DefPlayerInfo::Type playerType;

   if ( side == SIDE_WHITE )
   {
      doneWhite = 1;
      playerType = DefPlayer.whiteType;
      searchDepth = DefPlayer.whiteThinkTime;
      searchBias = DefPlayer.whiteSearchBias;
      useSeconds = DefPlayer.whiteUseSeconds;
   }
   else
   {
      playerType = DefPlayer.blackType;
      searchDepth = DefPlayer.blackThinkTime;
      searchBias = DefPlayer.blackSearchBias;
      useSeconds = DefPlayer.blackUseSeconds;
   }

   switch ( playerType )
   {
      case DefPlayerInfo::computerPlayer:
      {
         ComputerChessPlayer *puter = new ComputerChessPlayer (*this);
         if ( puter )
         {
            player = puter;
            if ( useSeconds )
            {
               puter->SetTimeLimit ( INT32(searchDepth) * 100 );
            }
            else
            {
               puter->SetSearchDepth ( searchDepth );
            }
            puter->SetSearchBias ( searchBias );
			extern cBOOLEAN Global_AllowResignFlag;
			puter->setResignFlag ( Global_AllowResignFlag );
         }
      }
      break;

      case DefPlayerInfo::humanPlayer:
      {
         player = new HumanChessPlayer (*this);
         if ( side == SIDE_WHITE )
            whiteIsHuman = cTRUE;
         else
            blackIsHuman = cTRUE;
      }
      break;
   }

   if ( doneWhite && side==SIDE_BLACK && !whiteIsHuman && blackIsHuman )
   {
      TheBoardDisplayBuffer.setView ( 0 );
   }

   if ( side == SIDE_WHITE )
       whitePlayer = player;
   else
       blackPlayer = player;

   return player;
}


static void LoadGameFromFile ( FILE *f, ChessBoard &board )
{
   board.Init();

   Move move;
   UnmoveInfo unmove;

   while ( fread ( &move, sizeof(Move), 1, f ) == 1 )
   {
      if ( (move.dest & SPECIAL_MOVE_MASK) == SPECIAL_MOVE_EDIT )
      {
         board.EditCommand ( move );
         board.SaveSpecialMove ( move );
      }
      else
      {
         board.MakeMove ( move, unmove );
      }
   }
}


static void LoadGameFromHardcode ( unsigned long *h, int n, ChessBoard &board )
{
   board.Init();

   for ( int i=0; i<n; i++ )
   {
      Move move = *((Move *)&h[i]);    // WARNING:  Not very portable!
      if ( (move.dest & SPECIAL_MOVE_MASK) == SPECIAL_MOVE_EDIT )
      {
         board.EditCommand ( move );
         board.SaveSpecialMove ( move );
      }
      else
      {
         UnmoveInfo unmove;
         board.MakeMove ( move, unmove );
      }
   }
}


static void SaveGameToFile ( 
    FILE *outfile,
    const ChessBoard  &board )
{
    int ply = board.GetCurrentPlyNumber();

    for ( int p=0; p < ply; p++ )
    {
        Move move = board.GetPastMove(p);
        fwrite ( &move, sizeof(Move), 1, outfile );
    }
}


cBOOLEAN ProcessChessCommands ( ChessBoard &board, int &source, int &dest )
{
    char msg [1024];
    cBOOLEAN whiteMove = board.WhiteToMove();
    Move edit;
    edit.score = 0;

    cBOOLEAN itIsTimeToCruise = cFALSE;
    if ( Global_TacticalBenchmarkFlag && Global_UI )
    {
		const char *explainText =
		"The tactical benchmark will run a series of pre-programmed chess positions "
		"to determine how quickly your computer can perform chess calculations. "
		"After the benchmark is finished, the current chess position will be restored "
		"and you will be able to resume your game. "
		"However, this test may take a long time (6 minutes on a 100 MHz Pentium).\n\n"
		"Do you really want to run the tactical benchmark?";

        Global_TacticalBenchmarkFlag = cFALSE;
        int choice = MessageBox ( HwndMain,
            explainText,
            CHESS_PROGRAM_NAME,
            MB_ICONQUESTION | MB_YESNO );

        if ( choice == IDYES )
        {
            double timeElapsed = Global_UI->runTacticalBenchmark();
			if ( timeElapsed > 0.0 ) 
			{
				const char *resultFormat = 
				"Benchmark completed in %0.2lf seconds\n\n"
				"Your computer is %0.2lf times as fast as the author's 100 MHz Pentium.\n\n"
#ifdef CHENARD_PROFILER
				"time data: {%d,%d,%d,%d,%d,%d,%d,%d,%d,%d}\n"
				"call data: {%d,%d,%d,%d,%d,%d,%d,%d,%d,%d}\n\n"
#endif
				"If you feel like it, send email to me <dcross@intersrv.com> with your computer's "
				"benchmark time and your processor type (e.g. 486, Pentium) and speed (MHz). "
				"I want to start collecting these statistics to publish them on the Chenard web page "
				"<http://www.intersrv.com/~dcross/chenard.html>.";

	            sprintf ( msg, resultFormat, 
					timeElapsed, 
					291.0 / timeElapsed
#ifdef CHENARD_PROFILER
					, ProfilerHitCount[0],
					ProfilerHitCount[1],
					ProfilerHitCount[2],
					ProfilerHitCount[3],
					ProfilerHitCount[4],
					ProfilerHitCount[5],
					ProfilerHitCount[6],
					ProfilerHitCount[7],
					ProfilerHitCount[8],
					ProfilerHitCount[9],
					
					ProfilerCallCount[0],
					ProfilerCallCount[1],
					ProfilerCallCount[2],
					ProfilerCallCount[3],
					ProfilerCallCount[4],
					ProfilerCallCount[5],
					ProfilerCallCount[6],
					ProfilerCallCount[7],
					ProfilerCallCount[8],
					ProfilerCallCount[9] 
#endif
				);

				MessageBox ( HwndMain, msg, CHESS_PROGRAM_NAME, MB_OK );
				TheBoardDisplayBuffer.update (board);
				TheBoardDisplayBuffer.freshenBoard();
			}
        }
    }
    else if ( Global_BoardEditRequest.isValid )
    {
        if ( Global_GameOverFlag )
        {
            MessageBox ( HwndMain,
                "Please start a new game (File | New) before editing the board",
                CHESS_PROGRAM_NAME,
                MB_ICONEXCLAMATION | MB_OK );
        }
        else
        {
            SQUARE square = Global_BoardEditRequest.square;
            int    x = Global_BoardEditRequest.x;
            int    y = Global_BoardEditRequest.y;

            board.SetSquareContents ( square, x, y );
            edit.dest = dest = SPECIAL_MOVE_EDIT | ConvertSquareToNybble(square);
            edit.source = source = OFFSET(x+2,y+2);
            board.SaveSpecialMove ( edit );

            Global_GameModifiedFlag = cTRUE;
            itIsTimeToCruise = cTRUE;
        }

        Global_BoardEditRequest.isValid = cFALSE;
    }
    else if ( Global_ClearBoardFlag )
    {
        if ( Global_GameOverFlag )
        {
            MessageBox ( HwndMain,
                "Please start a new game (File | New) before clearing the board",
                CHESS_PROGRAM_NAME,
                MB_ICONEXCLAMATION | MB_OK );
        }
        else
        {
            int choice = MessageBox
               ( HwndMain,
                 "Do you really want to clear the board?",
                 CHESS_PROGRAM_NAME,
                 MB_YESNO | MB_ICONQUESTION | MB_APPLMODAL );

            if ( choice == IDYES )
            {
                board.ClearEverythingButKings();
                itIsTimeToCruise = cTRUE;
                edit.source = source = 0;        // signal to clear everything!
                edit.dest = dest = SPECIAL_MOVE_EDIT;
                board.SaveSpecialMove ( edit );
            }
        }

        Global_ClearBoardFlag = cFALSE;
    }
    else if ( Global_ResetGameFlag )
    {
        Global_ResetGameFlag = cFALSE;

        int choice = IDYES;
        if ( Global_GameModifiedFlag )
        {
            choice = MessageBox ( 
                HwndMain,
                "Do you really want to start a new game?",
                CHESS_PROGRAM_NAME,
                MB_YESNO | MB_ICONQUESTION | MB_APPLMODAL );
        }

        if ( choice == IDYES )
        {
            board.Init();
            itIsTimeToCruise = cTRUE;
            dest = SPECIAL_MOVE_NULL;
            source = 0;
            Global_HaveFilename = cFALSE;
            Global_GameModifiedFlag = cFALSE;
        }
    }
	else if ( Global_RedoMoveFlag )
	{
		Global_RedoMoveFlag = cFALSE;
		Global_MoveUndoPath.redo ( board );
		TheBoardDisplayBuffer.update ( board );
		if ( Global_UI )
		{
			Global_UI->DrawBoard ( board );
		}
	}
	else if ( Global_UndoMoveFlag )
	{
		Global_UndoMoveFlag = cFALSE;
		Global_MoveUndoPath.undo ( board );
		TheBoardDisplayBuffer.update ( board );
		if ( Global_UI )
		{
			Global_UI->DrawBoard ( board );
		}
	}
    else if ( Global_GameSaveFlag )
    {
         Global_GameSaveFlag = cFALSE;
         FILE *outfile = fopen ( Global_GameFilename, "wb" );
         if ( outfile )
         {
             SaveGameToFile ( outfile, board );
             fclose (outfile);
             Global_HaveFilename = cTRUE;
             Global_GameModifiedFlag = cFALSE;
         }
         else
         {
             sprintf ( msg, "Cannot save file '%s'", Global_GameFilename );
             MessageBox ( HwndMain, msg, "File Open Error", MB_OK | MB_ICONERROR );
         }
    }
    else if ( Global_GameOpenFlag )
    {
        Global_GameOpenFlag = cFALSE;
        FILE *infile = fopen ( Global_GameFilename, "rb" );
        if ( infile )
        {
            LoadGameFromFile ( infile, board );
            fclose (infile);
            Global_HaveFilename = cTRUE;
            SetWindowText ( 
                HwndMain, 
                board.WhiteToMove() ? TitleBarText_WhitesTurn : TitleBarText_BlacksTurn );

            TheBoardDisplayBuffer.update (board);
            TheBoardDisplayBuffer.freshenBoard();
            Global_GameModifiedFlag = cFALSE;

            // We need to make a NULL move so that legal move list gets regenerated,
            // *and* the correct player will move,
            // *and* we will detect end-of-game properly.

            source = 0;
            dest = SPECIAL_MOVE_NULL;
            itIsTimeToCruise = cTRUE;
        }
        else
        {
            sprintf ( msg, "Cannot open file '%s'", Global_GameFilename );
            MessageBox ( HwndMain, msg, "File Open Error", MB_OK | MB_ICONERROR );
        }
    }

    return itIsTimeToCruise;
}


cBOOLEAN ChessUI_win32_gui::ReadMove ( 
    ChessBoard &board,
    int        &source,
    int        &dest )
{
   // Let the client window know we are starting to read
   // a human player's move...
   SendMessage ( hwnd, WM_DDC_ENTER_HUMAN_MOVE, 0, 0 );

   TheBoardDisplayBuffer.startReadingMove ( board.WhiteToMove() );

   while ( TheBoardDisplayBuffer.isReadingMove() )
   {
      Sleep (100);

      if ( Global_AbortFlag )
      {
          return cFALSE;   // simulate resignation to abort
      }

	  if ( Global_UserResign )
	  {
	      Global_UserResign = cFALSE;
	      return cFALSE;
	  }

      cBOOLEAN itIsTimeToCruise = ProcessChessCommands (board, source, dest);

      if ( itIsTimeToCruise )
      {
         DrawBoard ( board );
         SendMessage ( hwnd, WM_DDC_LEAVE_HUMAN_MOVE, 0, 0 );
         return cTRUE;
      }
   }

   TheBoardDisplayBuffer.copyMove ( source, dest );
   SendMessage ( hwnd, WM_DDC_LEAVE_HUMAN_MOVE, 0, 0 );
   return cTRUE;
}


SQUARE ChessUI_win32_gui::PromotePawn ( int, ChessSide side )
{
   static volatile SQUARE prom;
   prom = EMPTY;
   PostMessage ( hwnd, WM_DDC_PROMOTE_PAWN, WPARAM(side), LPARAM(&prom) );
   while ( prom == EMPTY )
   {
      Sleep ( 100 );
   }

   return prom;
}


void ChessUI_win32_gui::DrawBoard ( const ChessBoard &board )
{
   TheBoardDisplayBuffer.update ( board );
   TheBoardDisplayBuffer.deselectSquare();
   TheBoardDisplayBuffer.freshenBoard();
   // SaveGameToFile ( GameSave_filename, board, cFALSE );
   Sleep ( 100 );   // Helps the board to "breathe"
}


void ChessUI_win32_gui::Resign ( ChessSide iGiveUp )
{
   static gameReport report;
   report.winner = (iGiveUp == SIDE_WHITE) ? SIDE_BLACK : SIDE_WHITE;
   report.resignation = cTRUE;
   PostMessage ( hwnd, WM_DDC_GAME_RESULT, WPARAM(0), LPARAM(&report) );
   SetWindowText ( hwnd, TitleBarText_GameOver );
}


void ChessUI_win32_gui::ReportEndOfGame ( ChessSide winner )
{
   static gameReport report;
   report.winner = winner;
   report.resignation = cFALSE;
   PostMessage ( hwnd, WM_DDC_GAME_RESULT, WPARAM(0), LPARAM(&report) );
   SetWindowText ( hwnd, TitleBarText_GameOver );
}


void ChessUI_win32_gui::PredictMate ( int n )
{
   if ( !enableMateAnnounce )
       return;

   Sleep ( 200 );
   static volatile int messageProcessed;

   messageProcessed = 0;

   PostMessage ( hwnd,
                 WM_DDC_PREDICT_MATE,
                 WPARAM(n),
                 LPARAM(&messageProcessed) );

   while ( !messageProcessed )
   {
      Sleep ( 200 );
   }
}


static void FlashSquare ( int ofs, SQUARE before, SQUARE after )
{
   const int x = XPART(ofs) - 2;
   const int y = YPART(ofs) - 2;

   TheBoardDisplayBuffer.setSquareContents ( x, y, after );
   TheBoardDisplayBuffer.freshenSquare ( x, y );

   for ( int i=0; i < 2; i++ )
   {
      Sleep ( 180 );

      TheBoardDisplayBuffer.setSquareContents ( x, y, before );
      TheBoardDisplayBuffer.freshenSquare ( x, y );
      Sleep ( 180 );

      TheBoardDisplayBuffer.setSquareContents ( x, y, after );
      TheBoardDisplayBuffer.freshenSquare ( x, y );
   }
}


void ChessUI_win32_gui::DisplayMove ( ChessBoard &board, Move move )
{
	if ( tacticalBenchmarkMode )
		return;		// keep move display from messing up our timing

	ChessBeep ( 1400, 50 );

	int ofs1, ofs2;
	move.actualOffsets ( board, ofs1, ofs2 );

	// Show it being picked up...
	FlashSquare ( ofs1, board.GetSquareContents(ofs1), EMPTY );

	// Temporarily make move in board to show it being put down...
	UnmoveInfo unmove;
	board.MakeMove ( move, unmove );
	TheBoardDisplayBuffer.update ( board );
	TheBoardDisplayBuffer.freshenSquare ( XPART(ofs1)-2, YPART(ofs1)-2 );
	FlashSquare ( ofs2, EMPTY, board.GetSquareContents(ofs2) );

	// Undo the move because it's done officially by the ChessGame object.
	board.UnmakeMove ( move, unmove );
}


const unsigned MaxVectorDepth = 3;


void ChessUI_win32_gui::DebugPly ( int depth, ChessBoard &board, Move move )
{
	static int counter = 0;
	static ChessBoard TopBoard, TempBoard;
	static Move CurrentPath [MAX_BESTPATH_DISPLAY];

	if ( depth >= MAX_BESTPATH_DISPLAY )
		return;

	CurrentPath[depth] = move;
	if ( depth == 0 )
	{
		// The parameter 'board' contains the top-level position in the search.
		// Copy it into TempBoard so we have it whenever we need to display
		// move notation for the current path.
		TopBoard = board;
	}

	if ( Global_AnalysisType == 2 )
	{
		const int limit = 5000;
		if ( ++counter > limit )
		{
			// Now it's time to refresh the current path display.

			TempBoard = TopBoard;

			for ( int i=0; i<=depth; i++ )
			{
				char moveString [64];
				UnmoveInfo unmove;
				FormatChessMove ( TempBoard, CurrentPath[i], moveString );
				TempBoard.MakeMove ( CurrentPath[i], unmove );
				ChessDisplayTextBuffer::SetText ( STATIC_ID_BESTPATH(i), moveString );
			}

			for ( ; i < MAX_BESTPATH_DISPLAY; i++ )
			{
				ChessDisplayTextBuffer::SetText ( STATIC_ID_BESTPATH(i), "" );
			}

			counter = 0;
		}
	}
}


void ChessUI_win32_gui::DebugExit ( int depth, ChessBoard &board, SCORE score )
{
}



void ChessUI_win32_gui::DisplayBestPath ( 
    const ChessBoard &_board,
    const BestPath &path )
{
	if ( Global_AnalysisType != 1 )
		return;

    // We change the board here, but we put it back the way we find it!
    ChessBoard &board = (ChessBoard &) _board;

    int n = path.depth;
    if ( n > MAX_BESTPATH_DISPLAY )
        n = MAX_BESTPATH_DISPLAY;

    UnmoveInfo unmove [MAX_BESTPATH_DISPLAY];

    for ( int i=0; i <= MAX_BESTPATH_DISPLAY; i++ )
    {
        char moveString [64];
        if ( i > n )
        {
            moveString[0] = '\0';
        }
        else
        {
            Move move = path.m[i];
            FormatChessMove ( board, move, moveString );
            board.MakeMove ( move, unmove[i] );
            if ( i == 0 )
            {
                char temp [64];
                sprintf ( temp, "%-12s %6d", moveString, int(move.score) );
                strcpy ( moveString, temp );
            }
        }
        ChessDisplayTextBuffer::SetText ( STATIC_ID_BESTPATH(i), moveString );
    }

    for ( i=n; i>=0; i-- )
    {
        Move move = path.m[i];
        board.UnmakeMove ( move, unmove[i] );
    }
}



void ChessUI_win32_gui::RecordMove ( 
    ChessBoard  &board,
    Move         move,
    INT32        thinkTime )
{
	char moveString [MAX_MOVE_STRLEN + 1];
	char displayString [30 + MAX_MOVE_STRLEN + 1];
	int  wm = board.WhiteToMove();

	FormatChessMove ( board, move, moveString );

	UnmoveInfo unmove;
	board.MakeMove ( move, unmove );
	Global_MoveUndoPath.resetFromBoard ( board );
	board.UnmakeMove ( move, unmove );

	if ( Global_SpeakMovesFlag )
	{
		Speak ( board, move, moveString );
	}

	if ( wm )
	{
		sprintf ( displayString, "%d. %s: %s (%0.2lf)", 
			1 + board.GetCurrentPlyNumber()/2,
			"White",
			moveString,
			double(thinkTime) / 100.0 );

		ChessDisplayTextBuffer::SetText ( 
			STATIC_ID_WHITES_MOVE,
			displayString );

		// It will be black's turn real soon now.
		SetWindowText ( hwnd, TitleBarText_BlacksTurn );
	}
	else
	{
		sprintf ( displayString,
			"%s: %s (%0.2lf)",
			"Black",
			moveString,
			double(thinkTime) / 100.0 );

		ChessDisplayTextBuffer::SetText ( 
			STATIC_ID_BLACKS_MOVE,
			displayString );

		// It will be white's turn real soon now.
		SetWindowText ( hwnd, TitleBarText_WhitesTurn );
	}

	Global_GameModifiedFlag = cTRUE;
}


void ChessUI_win32_gui::DisplayBestMoveSoFar ( const ChessBoard &board,
                                               Move              bestMove,
                                               int )
{
	if ( Global_AnalysisType == 2 )
	{
		char moveString [MAX_MOVE_STRLEN + 1];
		char displayString [20 + MAX_MOVE_STRLEN + 1];

		FormatChessMove ( board, bestMove, moveString );
		sprintf ( displayString,
				  "%6d [%s]",
				  int(bestMove.score),
				  moveString );

		ChessDisplayTextBuffer::SetText ( STATIC_ID_CURRENT_MOVE, displayString );
	}

    if ( tacticalBenchmarkMode )
    {
        if ( bestMove == benchmarkExpectedMove )
        {
            benchmarkPlayer->AbortSearch();
        }
    }
}


void ChessUI_win32_gui::DisplayCurrentMove ( const ChessBoard &board,
                                          Move              move,
                                          int               level )
{
   if ( Global_AnalysisType != 0 )
   {
      char moveString [MAX_MOVE_STRLEN + 1];
      char displayString [20 + MAX_MOVE_STRLEN + 1];
   
      FormatChessMove ( board, move, moveString );
      sprintf ( displayString,
                "%2d: %6d [%s]",
                level,
                int ( move.score ),
                moveString );
   
      if ( Global_AnalysisType == 1 )
	  {
			ChessDisplayTextBuffer::SetText ( STATIC_ID_CURRENT_MOVE, displayString );
	  }
   }
}


void ChessUI_win32_gui::ReportSpecial ( const char *msg )
{
    ChessDisplayTextBuffer::SetText ( STATIC_ID_CURRENT_MOVE, msg );
}


void ChessUI_win32_gui::ReportComputerStats ( 
    INT32   thinkTime,
    UINT32  nodesVisited,
    UINT32  nodesEvaluated,
    UINT32  /*nodesGenerated*/,
    int     /*fwSearchDepth*/,
    UINT32  /*nvis*/ [NODES_ARRAY_SIZE],
    UINT32  /*ngen*/ [NODES_ARRAY_SIZE] )
{
    char displayString [80];

    sprintf ( displayString, "visited=%lu", long(nodesVisited) );
    ChessDisplayTextBuffer::SetText ( STATIC_ID_NODES_VISITED, displayString );

    sprintf ( displayString, "evaluated=%lu", long(nodesEvaluated) );
    ChessDisplayTextBuffer::SetText ( STATIC_ID_NODES_EVALUATED, displayString );

	extern cBOOLEAN Global_AllowResignFlag;
	if ( !whiteIsHuman )
		((ComputerChessPlayer *)whitePlayer)->setResignFlag ( Global_AllowResignFlag );

	if ( !blackIsHuman )
		((ComputerChessPlayer *)blackPlayer)->setResignFlag ( Global_AllowResignFlag );
}


void ChessUI_win32_gui::forceMove()
{
    if ( !whiteIsHuman )
        ((ComputerChessPlayer *)whitePlayer)->AbortSearch();

    if ( !blackIsHuman )
        ((ComputerChessPlayer *)blackPlayer)->AbortSearch();
}


void ChessUI_win32_gui::setWhiteThinkTime ( INT32 hundredthsOfSeconds ) 
{ 
	if ( !whiteIsHuman && whitePlayer && hundredthsOfSeconds >= 100 )
	{
		ComputerChessPlayer *puter = (ComputerChessPlayer *)whitePlayer;
		puter->SetTimeLimit ( hundredthsOfSeconds );
	}
}

void ChessUI_win32_gui::setBlackThinkTime ( INT32 hundredthsOfSeconds ) 
{ 
	if ( !blackIsHuman && blackPlayer && hundredthsOfSeconds >= 100 )
	{
		ComputerChessPlayer *puter = (ComputerChessPlayer *)blackPlayer;
		puter->SetTimeLimit ( hundredthsOfSeconds );
	}
}


INT32 ChessUI_win32_gui::queryWhiteThinkTime() const
{
	if ( !whiteIsHuman && whitePlayer )
	{
		const ComputerChessPlayer *puter = (const ComputerChessPlayer *)whitePlayer;
		return puter->QueryTimeLimit();
	}

	return 0;
}


INT32 ChessUI_win32_gui::queryBlackThinkTime() const
{
	if ( !blackIsHuman && blackPlayer )
	{
		const ComputerChessPlayer *puter = (const ComputerChessPlayer *)blackPlayer;
		return puter->QueryTimeLimit();
	}

	return 0;
}


//-------------------------- benchmark stuff ------------------------------


/*
   hardcode varok.gam varok.cpp VarokGame -U -x
*/

static unsigned long VarokGame[] =
{0x4129L,0x5B74L,0x3720L,0xFFFD586FL,0x3226L,0x244D65L,0x362AL,0xFFFD5C68L,
0x291BL,0xFFED6570L,0x3F27L,0xFFE74A62L,0x4B3FL,0xFFED6258L,0x3E32L,
0xFFD86873L,0x4028L,0xFFD5B072L,0x321CL,0xFFDD5864L,0x584BL,0xFFC75862L,
0x331DL,0xFFCF7065L,0x4D40L,0xFFF54D5BL,0x4B1FL,0xFFFB5A70L,0x5037L,
0xC6571L,0xB01EL,0x147173L,0x1B1AL,0x156F6EL};



double ChessUI_win32_gui::runTacticalBenchmark()
{
	StartProfiler();
    tacticalBenchmarkMode = cTRUE;
    enableMateAnnounce = cFALSE;

    // Create a temporary chess board.
    ChessBoard tboard;

    // Create a temporary chess player to think about the position
    benchmarkPlayer = new ComputerChessPlayer (*this);

    INT32 timeSpent;
    double totalTime = 0.0;
    Move move;
    UnmoveInfo unmove;

    LoadGameFromHardcode (
        VarokGame, sizeof(VarokGame)/sizeof(VarokGame[0]), tboard );
    benchmarkExpectedMove.source = 0;
    benchmarkExpectedMove.dest = 0;
    TheBoardDisplayBuffer.update (tboard);
    TheBoardDisplayBuffer.freshenBoard();
    benchmarkPlayer->SetSearchDepth (4);
    benchmarkPlayer->ResetHistoryBuffers();
    benchmarkPlayer->GetMove ( tboard, move, timeSpent );
    totalTime += (0.01 * timeSpent);

    // Now just search a neutral but very complex midgame position to a fixed depth.
    // We do not expect a particular move; we just want to see how long a fixed depth takes.
    tboard.Init();
    tboard.ClearEverythingButKings();
    tboard.SetSquareContents ( WKING, 6, 0 );
    tboard.SetSquareContents ( BKING, 6, 7 );
    tboard.SetSquareContents ( WROOK, 0, 0 );
    tboard.SetSquareContents ( WROOK, 5, 0 );
    tboard.SetSquareContents ( WPAWN, 0, 1 );
    tboard.SetSquareContents ( WPAWN, 1, 1 );
    tboard.SetSquareContents ( WPAWN, 2, 1 );
    tboard.SetSquareContents ( WQUEEN, 4, 1 );
    tboard.SetSquareContents ( WPAWN, 5, 1 );
    tboard.SetSquareContents ( WPAWN, 6, 1 );
    tboard.SetSquareContents ( WPAWN, 7, 1 );
    tboard.SetSquareContents ( WBISHOP, 1, 2 );
    tboard.SetSquareContents ( WKNIGHT, 2, 2 );
    tboard.SetSquareContents ( WPAWN, 3, 2 );
    tboard.SetSquareContents ( WKNIGHT, 5, 2 );
    tboard.SetSquareContents ( WPAWN, 4, 3 );
    tboard.SetSquareContents ( BBISHOP, 2, 4 );
    tboard.SetSquareContents ( BPAWN, 4, 4 );
    tboard.SetSquareContents ( WBISHOP, 6, 4 );
    tboard.SetSquareContents ( BPAWN, 0, 5 );
    tboard.SetSquareContents ( BKNIGHT, 2, 5 );
    tboard.SetSquareContents ( BPAWN, 3, 5 );
    tboard.SetSquareContents ( BKNIGHT, 5, 5 );
    tboard.SetSquareContents ( BPAWN, 1, 6 );
    tboard.SetSquareContents ( BPAWN, 2, 6 );
    tboard.SetSquareContents ( BBISHOP, 3, 6 );
    tboard.SetSquareContents ( BPAWN, 5, 6 );
    tboard.SetSquareContents ( BPAWN, 6, 6 );
    tboard.SetSquareContents ( BPAWN, 7, 6 );
    tboard.SetSquareContents ( BROOK, 0, 7 );
    tboard.SetSquareContents ( BQUEEN, 3, 7 );
    tboard.SetSquareContents ( BROOK, 4, 7 );
    benchmarkExpectedMove.source = 0;
    benchmarkExpectedMove.dest = 0;
    TheBoardDisplayBuffer.update (tboard);
    TheBoardDisplayBuffer.freshenBoard();
    benchmarkPlayer->SetSearchDepth (4);
    benchmarkPlayer->ResetHistoryBuffers();
    benchmarkPlayer->GetMove ( tboard, move, timeSpent );
    totalTime += (0.01 * timeSpent);

    // Now do a little endgame position
    tboard.Init();
    tboard.ClearEverythingButKings();
    tboard.SetSquareContents ( WPAWN, 0, 1 );
    tboard.SetSquareContents ( WPAWN, 1, 1 );
    tboard.SetSquareContents ( WKING, 5, 3 );
    tboard.SetSquareContents ( WKNIGHT, 6, 3 );
    tboard.SetSquareContents ( BPAWN, 3, 4 );
    tboard.SetSquareContents ( WPAWN, 4, 4 );
    tboard.SetSquareContents ( BKING, 4, 5 );
    tboard.SetSquareContents ( BPAWN, 0, 6 );
    tboard.SetSquareContents ( BPAWN, 1, 6 );
    tboard.SetSquareContents ( BBISHOP, 3, 6 );
    TheBoardDisplayBuffer.update (tboard);
    TheBoardDisplayBuffer.freshenBoard();
    benchmarkExpectedMove.source = 0;
    benchmarkExpectedMove.dest = 0;
    benchmarkPlayer->SetSearchDepth (7);
    benchmarkPlayer->ResetHistoryBuffers();
    benchmarkPlayer->GetMove ( tboard, move, timeSpent );
    totalTime += (0.01 * timeSpent);

    // zugzwang test:  Smyslov v Kasparov, Vilnius 1984, game 12.
    tboard.Init();
    tboard.ClearEverythingButKings();
    tboard.SetSquareContents ( WKING, 7, 1 );
    tboard.SetSquareContents ( BKING, 6, 6 );
    tboard.SetSquareContents ( WBISHOP, 4, 1 );
    tboard.SetSquareContents ( WPAWN, 0, 2 );
    tboard.SetSquareContents ( WKNIGHT, 3, 2 );
    tboard.SetSquareContents ( BQUEEN, 4, 2 );
    tboard.SetSquareContents ( WPAWN, 6, 2 );
    tboard.SetSquareContents ( WPAWN, 1, 3 );
    tboard.SetSquareContents ( WPAWN, 4, 3 );
    tboard.SetSquareContents ( WQUEEN, 6, 3 );
    tboard.SetSquareContents ( BPAWN, 1, 4 );
    tboard.SetSquareContents ( BPAWN, 6, 4 );
    tboard.SetSquareContents ( WPAWN, 7, 4 );
    tboard.SetSquareContents ( BPAWN, 0, 5 );
    tboard.SetSquareContents ( WPAWN, 4, 5 );
    tboard.SetSquareContents ( BPAWN, 7, 5 );
    tboard.SetSquareContents ( BKNIGHT, 4, 6 );
    tboard.SetSquareContents ( BROOK, 5, 7 );
    move.source = OFFSET(9,3);
    move.dest = OFFSET(8,3);
    tboard.MakeMove ( move, unmove );
    benchmarkExpectedMove.source = 0;
    benchmarkExpectedMove.dest = 0;
    benchmarkPlayer->SetSearchDepth (4);
    TheBoardDisplayBuffer.update (tboard);
    TheBoardDisplayBuffer.freshenBoard();
    benchmarkPlayer->ResetHistoryBuffers();
    benchmarkPlayer->GetMove ( tboard, move, timeSpent );
    totalTime += (0.01 * timeSpent);

    // Late midgame, with highly mobile pieces
    tboard.Init();
    tboard.ClearEverythingButKings();
    tboard.SetSquareContents ( WKING, 6, 0 );
    tboard.SetSquareContents ( WROOK, 3, 0 );
    tboard.SetSquareContents ( WROOK, 4, 0 );
    tboard.SetSquareContents ( WPAWN, 0, 1 );
    tboard.SetSquareContents ( WPAWN, 1, 1 );
    tboard.SetSquareContents ( WBISHOP, 2, 1 );
    tboard.SetSquareContents ( WPAWN, 5, 1 );
    tboard.SetSquareContents ( WPAWN, 6, 1 );
    tboard.SetSquareContents ( WPAWN, 2, 2 );
    tboard.SetSquareContents ( WQUEEN, 6, 2 );
    tboard.SetSquareContents ( WPAWN, 7, 2 );
    tboard.SetSquareContents ( WKNIGHT, 3, 3 );
    tboard.SetSquareContents ( BPAWN, 1, 5 );
    tboard.SetSquareContents ( BPAWN, 4, 5 );
    tboard.SetSquareContents ( BQUEEN, 5, 5 );
    tboard.SetSquareContents ( BPAWN, 6, 5 );
    tboard.SetSquareContents ( BPAWN, 0, 6 );
    tboard.SetSquareContents ( BBISHOP, 1, 6 );
    tboard.SetSquareContents ( BPAWN, 5, 6 );
    tboard.SetSquareContents ( BBISHOP, 6, 6 );
    tboard.SetSquareContents ( BKING, 0, 7 );
    tboard.SetSquareContents ( BROOK, 2, 7 );
    tboard.SetSquareContents ( BROOK, 4, 7 );
    benchmarkExpectedMove.source = 0;
    benchmarkExpectedMove.dest = 0;
    benchmarkPlayer->SetSearchDepth (4);
    TheBoardDisplayBuffer.update (tboard);
    TheBoardDisplayBuffer.freshenBoard();
    benchmarkPlayer->ResetHistoryBuffers();
    benchmarkPlayer->GetMove ( tboard, move, timeSpent );
    totalTime += (0.01 * timeSpent);

    // Complex queen's gambit declined opening position
    tboard.Init();
    tboard.ClearEverythingButKings();
    tboard.SetSquareContents ( WKING, 6, 0 );
    tboard.SetSquareContents ( WROOK, 0, 0 );
    tboard.SetSquareContents ( WQUEEN, 3, 0 );
    tboard.SetSquareContents ( WROOK, 5, 0 );
    tboard.SetSquareContents ( WPAWN, 1, 1 );
    tboard.SetSquareContents ( WPAWN, 5, 1 );
    tboard.SetSquareContents ( WPAWN, 6, 1 );
    tboard.SetSquareContents ( WPAWN, 7, 1 );
    tboard.SetSquareContents ( WPAWN, 0, 2 );
    tboard.SetSquareContents ( WKNIGHT, 2, 2 );
    tboard.SetSquareContents ( WBISHOP, 3, 2 );
    tboard.SetSquareContents ( WPAWN, 4, 2 );
    tboard.SetSquareContents ( WKNIGHT, 5, 2 );
    tboard.SetSquareContents ( WPAWN, 2, 3 );
    tboard.SetSquareContents ( WPAWN, 3, 3 );
    tboard.SetSquareContents ( BPAWN, 3, 4 );
    tboard.SetSquareContents ( WBISHOP, 6, 4 );
    tboard.SetSquareContents ( BPAWN, 1, 5 );
    tboard.SetSquareContents ( BKNIGHT, 2, 5 );
    tboard.SetSquareContents ( BPAWN, 4, 5 );
    tboard.SetSquareContents ( BKNIGHT, 5, 5 );
    tboard.SetSquareContents ( BPAWN, 0, 6 );
    tboard.SetSquareContents ( BBISHOP, 1, 6 );
    tboard.SetSquareContents ( BPAWN, 2, 6 );
    tboard.SetSquareContents ( BBISHOP, 4, 6 );
    tboard.SetSquareContents ( BPAWN, 5, 6 );
    tboard.SetSquareContents ( BPAWN, 6, 6 );
    tboard.SetSquareContents ( BPAWN, 7, 6 );
    tboard.SetSquareContents ( BKING, 6, 7 );
    tboard.SetSquareContents ( BROOK, 4, 7 );
    tboard.SetSquareContents ( BQUEEN, 3, 7 );
    tboard.SetSquareContents ( BROOK, 0, 7 );
    benchmarkExpectedMove.source = 0;
    benchmarkExpectedMove.dest = 0;
    benchmarkPlayer->SetSearchDepth (4);
    TheBoardDisplayBuffer.update (tboard);
    TheBoardDisplayBuffer.freshenBoard();
    benchmarkPlayer->ResetHistoryBuffers();
    benchmarkPlayer->GetMove ( tboard, move, timeSpent );
    totalTime += (0.01 * timeSpent);

    tacticalBenchmarkMode = cFALSE;
    enableMateAnnounce = cTRUE;
    delete benchmarkPlayer;
    benchmarkPlayer = 0;

	StopProfiler();

    return totalTime;
}


/*--- end of file uiwin32.cpp ---*/
