/*============================================================================

      unmove.cpp  -  Copyright (C) 1993-1996 by Don Cross
      email: dcross@intersrv.com
      WWW:   http://www.intersrv.com/~dcross/

      Contains code for unmaking a move on a ChessBoard.

      Revision history:

1993 May 29 [Don Cross]:
     Fixed bug with not updating bk_offset and wk_offset while unmaking
     non-special moves.

1993 August 30 [Don Cross]
     Changing pointers to references in the interfaces where
     appropriate.

1993 October 19 [Don Cross]
     Removed unnecessary updates of wmaterial and bmaterial.
     They are unnecessary because we just restore them from
     unmove.wmaterial and unmove.bmaterial, respectively.

1993 October 22 [Don Cross]
     Fixed a bug in pawn promotion code:  when a pawn promotion was
     also a capture, I was forgetting to initialize prom_piece.
     I have changed the code from getting prom_piece from the dest
     square in this case to just using the move.dest itself to
     figure out the promoted piece.

1994 February 10 [Don Cross]
     Adding indexed pieces.

============================================================================*/

#include "chess.h"
#include "profiler.h"


void ChessBoard::UnmakeWhiteMove ( Move move, UnmoveInfo &unmove )
{
	PROFILER_ENTER(PX_UNMOVE);

   int      source = move.source & BOARD_OFFSET_MASK;
   int      dest;
   SQUARE   capture = unmove.capture;
   int      prom_piece_index;

   if ( move.dest > OFFSET(9,9) )
   {
      // This is a "special" move...

      switch ( move.dest & 0xF0 )
      {
         case SPECIAL_MOVE_PROMOTE_NORM:
              prom_piece_index = (move.dest & PIECE_MASK) | WHITE_IND;
              dest = source + NORTH;
              board [source] = WPAWN;
              board [dest] = EMPTY;
              ++inventory [WP_INDEX];
              --inventory [prom_piece_index];
              break;

         case SPECIAL_MOVE_PROMOTE_CAP_EAST:
              prom_piece_index = (move.dest & PIECE_MASK) | WHITE_IND;
              dest = source + NORTHEAST;
              board [source] = WPAWN;
              board [dest] = capture;
              ++inventory [WP_INDEX];
              --inventory [prom_piece_index];
              break;

         case SPECIAL_MOVE_PROMOTE_CAP_WEST:
              prom_piece_index = (move.dest & PIECE_MASK) | WHITE_IND;
              dest = source + NORTHWEST;
              board [source] = WPAWN;
              board [dest] = capture;
              ++inventory [WP_INDEX];
              --inventory [prom_piece_index];
              break;

         case SPECIAL_MOVE_KCASTLE:
              wk_offset = OFFSET(6,2);
              board [ OFFSET(6,2) ] = WKING;
              board [ OFFSET(9,2) ] = WROOK;
              board [ OFFSET(7,2) ] = board [ OFFSET(8,2) ] = EMPTY;
              break;

         case SPECIAL_MOVE_QCASTLE:
              wk_offset = OFFSET(6,2);
              board [ OFFSET(6,2) ] = WKING;
              board [ OFFSET(2,2) ] = WROOK;
              board [ OFFSET(4,2) ] = board [ OFFSET(5,2) ] = EMPTY;
              break;

         case SPECIAL_MOVE_EP_EAST:
              board [source] = board [dest = source + NORTHEAST];
              board [dest] = EMPTY;
              board [source + EAST] = capture;
              break;

         case SPECIAL_MOVE_EP_WEST:
              board [source] = board [dest = source + NORTHWEST];
              board [dest] = EMPTY;
              board [source + WEST] = capture;
              break;
      }
   }
   else
   {
      // This is a "normal" move...
      dest = move.dest;
      SQUARE move_piece = board[source] = board[dest];
      board[dest] = capture;

      if ( move_piece & WK_MASK )
      {
         wk_offset = source;
      }
   }

   if ( capture != EMPTY )
   {
      ++inventory [SPIECE_INDEX(capture)];
   }

   --hashCount [hashCode];

   flags            =  unmove.flags;
   bmaterial        =  unmove.bmaterial;
   wmaterial        =  unmove.wmaterial;
   prev_move        =  unmove.prev_move;
   lastCapOrPawn    =  unmove.lastCapOrPawn;
   hashCode         =  unmove.hashCode;

   --ply_number;
   white_to_move = cTRUE;

	PROFILER_EXIT();
}


//------------------------------------------------------------------------


void ChessBoard::UnmakeBlackMove ( Move move, UnmoveInfo &unmove )
{
	PROFILER_ENTER(PX_UNMOVE)

   int      source = move.source & BOARD_OFFSET_MASK;
   int      dest;
   SQUARE   capture = unmove.capture;
   SQUARE   prom_piece_index;

   if ( move.dest > OFFSET(9,9) )
   {
      // This is a "special" move...

      switch ( move.dest & 0xF0 )
      {
         case SPECIAL_MOVE_PROMOTE_NORM:
              prom_piece_index = (move.dest & PIECE_MASK) | BLACK_IND;
              dest = source + SOUTH;
              board [source] = BPAWN;
              board [dest] = EMPTY;
              ++inventory [BP_INDEX];
              --inventory [prom_piece_index];
              break;

         case SPECIAL_MOVE_PROMOTE_CAP_EAST:
              prom_piece_index = (move.dest & PIECE_MASK) | BLACK_IND;
              dest = source + SOUTHEAST;
              board [source] = BPAWN;
              board [dest] = capture;
              ++inventory [BP_INDEX];
              --inventory [prom_piece_index];
              break;

         case SPECIAL_MOVE_PROMOTE_CAP_WEST:
              prom_piece_index = (move.dest & PIECE_MASK) | BLACK_IND;
              dest = source + SOUTHWEST;
              board [source] = BPAWN;
              board [dest] = capture;
              ++inventory [BP_INDEX];
              --inventory [prom_piece_index];
              break;

         case SPECIAL_MOVE_KCASTLE:
              bk_offset = OFFSET(6,9);
              board [ OFFSET(6,9) ] = BKING;
              board [ OFFSET(9,9) ] = BROOK;
              board [ OFFSET(7,9) ] = board [ OFFSET(8,9) ] = EMPTY;
              break;

         case SPECIAL_MOVE_QCASTLE:
              bk_offset = OFFSET(6,9);
              board [ OFFSET(6,9) ] = BKING;
              board [ OFFSET(2,9) ] = BROOK;
              board [ OFFSET(4,9) ] = board [ OFFSET(5,9) ] = EMPTY;
              break;

         case SPECIAL_MOVE_EP_EAST:
              board [source] = board [dest = source + SOUTHEAST];
              board [dest] = EMPTY;
              board [source + EAST] = capture;
              break;

         case SPECIAL_MOVE_EP_WEST:
              board [source] = board [dest = source + SOUTHWEST];
              board [dest] = EMPTY;
              board [source + WEST] = capture;
              break;
      }
   }
   else
   {
      // This is a "normal" move...
      dest = move.dest;
      SQUARE move_piece = board[source] = board[dest];
      board [dest] = capture;

      if ( move_piece & BK_MASK )
      {
         bk_offset = source;
      }
   }

   if ( capture != EMPTY )
   {
      ++inventory [SPIECE_INDEX(capture)];
   }

   --hashCount [hashCode];

   flags            =  unmove.flags;
   bmaterial        =  unmove.bmaterial;
   wmaterial        =  unmove.wmaterial;
   prev_move        =  unmove.prev_move;
   lastCapOrPawn    =  unmove.lastCapOrPawn;
   hashCode         =  unmove.hashCode;

   --ply_number;
   white_to_move = cFALSE;

	PROFILER_EXIT()
}


/*--- end of file unmove.cpp ---*/