/*==========================================================================

     wbrdbuf.cpp  -  Copyright (C) 1993-1996 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Contains BoardDisplayBuffer for Win32 GUI version of Chenard.

     Revision history:

1994 February 9 [Don Cross]
     Adding visual feedback for selecting squares.

1996 July 29 [Don Cross]
     Started porting from OS/2 to Win32.

1997 January 30 [Don Cross]
     Adding "Skak" font.

==========================================================================*/

#include "chess.h"
#include "winchess.h"
#include "winguich.h"
#include "resource.h"

int RawBitmapDX = 40;
int RawBitmapDY = 40;
int BitmapScaleFactor = 1;
int BitmapScaleDenom = 1;
int ChessBoardSize = 0;    // 0=small, 1=medium, 2=large

int PieceFont_DX [NUM_PIECE_FONTS] = {40, 48, 48};
int PieceFont_DY [NUM_PIECE_FONTS] = {40, 48, 48};


void NewBoardSize ( int newSize )
{
    if ( newSize < 0 || newSize > 2 )
	newSize = 0;

    ChessBoardSize = newSize;
    switch ( ChessBoardSize )
    {
		case 1:   BitmapScaleFactor = 3;  BitmapScaleDenom = 2;  break;
        case 2:   BitmapScaleFactor = 2;  BitmapScaleDenom = 1;  break;
        default:  BitmapScaleFactor = 1;  BitmapScaleDenom = 1;  break;
    }
}


BoardDisplayBuffer::BoardDisplayBuffer():
    bitmapsLoadedFlag ( 0 ),
    whiteViewFlag ( 1 ),
    readingMoveFlag ( cFALSE ),
    gotSource ( cFALSE ),
    moveSource ( 0 ),
    moveDest ( 0 ),
    selX ( -1 ),
    selY ( -1 ),
    pieceFont ( PIECE_FONT_TILBURG ),
	tempHDC ( 0 )
{
    for ( int x=0; x < 8; x++ )
    {
        for ( int y=0; y < 8; y++ )
        {
            board[x][y] = EMPTY;
            changed[x][y] = cFALSE;
        }
    }

    RawBitmapDX = PieceFont_DX [pieceFont];
    RawBitmapDY = PieceFont_DY [pieceFont];
}


BoardDisplayBuffer::~BoardDisplayBuffer()
{
    unloadBitmaps();
}


void BoardDisplayBuffer::unloadBitmaps()
{
    if ( bitmapsLoadedFlag )
    {
        bitmapsLoadedFlag = 0;
        if ( wpw != NULL )  {  DeleteObject ( wpw );  wpw = NULL;  }
        if ( wnw != NULL )  {  DeleteObject ( wnw );  wnw = NULL;  }
        if ( wbw != NULL )  {  DeleteObject ( wbw );  wbw = NULL;  }
        if ( wrw != NULL )  {  DeleteObject ( wrw );  wrw = NULL;  }
        if ( wqw != NULL )  {  DeleteObject ( wqw );  wqw = NULL;  }
        if ( wkw != NULL )  {  DeleteObject ( wkw );  wkw = NULL;  }
        if ( wpb != NULL )  {  DeleteObject ( wpb );  wpb = NULL;  }
        if ( wnb != NULL )  {  DeleteObject ( wnb );  wnb = NULL;  }
        if ( wbb != NULL )  {  DeleteObject ( wbb );  wbb = NULL;  }
        if ( wrb != NULL )  {  DeleteObject ( wrb );  wrb = NULL;  }
        if ( wqb != NULL )  {  DeleteObject ( wqb );  wqb = NULL;  }
        if ( wkb != NULL )  {  DeleteObject ( wkb );  wkb = NULL;  }
        if ( bpw != NULL )  {  DeleteObject ( bpw );  bpw = NULL;  }
        if ( bnw != NULL )  {  DeleteObject ( bnw );  bnw = NULL;  }
        if ( bbw != NULL )  {  DeleteObject ( bbw );  bbw = NULL;  }
        if ( brw != NULL )  {  DeleteObject ( brw );  brw = NULL;  }
        if ( bqw != NULL )  {  DeleteObject ( bqw );  bqw = NULL;  }
        if ( bkw != NULL )  {  DeleteObject ( bkw );  bkw = NULL;  }
        if ( bpb != NULL )  {  DeleteObject ( bpb );  bpb = NULL;  }
        if ( bnb != NULL )  {  DeleteObject ( bnb );  bnb = NULL;  }
        if ( bbb != NULL )  {  DeleteObject ( bbb );  bbb = NULL;  }
        if ( brb != NULL )  {  DeleteObject ( brb );  brb = NULL;  }
        if ( bqb != NULL )  {  DeleteObject ( bqb );  bqb = NULL;  }
        if ( bkb != NULL )  {  DeleteObject ( bkb );  bkb = NULL;  }
        if ( qmw != NULL )  {  DeleteObject ( qmw );  qmw = NULL;  }
        if ( qmb != NULL )  {  DeleteObject ( qmb );  qmb = NULL;  }
        if ( ew != NULL )   {  DeleteObject ( ew  );  ew = NULL;   }
        if ( eb != NULL )   {  DeleteObject ( eb  );  eb = NULL;   }
    }
}



void BoardDisplayBuffer::loadBitmaps ( HINSTANCE progInstance )
{
    if ( bitmapsLoadedFlag )
        return;

    if ( pieceFont == PIECE_FONT_TILBURG )
    {
        // White pieces on white background...
        wpw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WPawn_W) );
        wnw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WKnight_W) );
        wbw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WBishop_W) );
        wrw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WRook_W) );
        wqw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WQueen_W) );
        wkw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WKing_W) );

        // White pieces on black background...
        wpb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WPawn_B) );
        wnb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WKnight_B) );
        wbb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WBishop_B) );
        wrb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WRook_B) );
        wqb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WQueen_B) );
        wkb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_WKing_B) );

        // Black pieces on white background...
        bpw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BPawn_W) );
        bnw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BKnight_W) );
        bbw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BBishop_W) );
        brw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BRook_W) );
        bqw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BQueen_W) );
        bkw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BKing_W) );

        // Black pieces on black background...
        bpb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BPawn_B) );
        bnb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BKnight_B) );
        bbb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BBishop_B) );
        brb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BRook_B) );
        bqb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BQueen_B) );
        bkb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_BKing_B) );

        // Special bitmaps...
        qmw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_Qmark_W) );
        qmb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_Qmark_B) );
        ew  = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_Empty_W) );
        eb  = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_Empty_B) );
    }
	else if ( pieceFont == PIECE_FONT_SKAK )
	{
        // White pieces on white background...
        wpw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WPawn_W) );
        wnw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WKnight_W) );
        wbw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WBishop_W) );
        wrw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WRook_W) );
        wqw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WQueen_W) );
        wkw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WKing_W) );

        // White pieces on black background...
        wpb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WPawn_B) );
        wnb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WKnight_B) );
        wbb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WBishop_B) );
        wrb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WRook_B) );
        wqb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WQueen_B) );
        wkb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_WKing_B) );

        // Black pieces on white background...
        bpw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BPawn_W) );
        bnw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BKnight_W) );
        bbw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BBishop_W) );
        brw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BRook_W) );
        bqw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BQueen_W) );
        bkw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BKing_W) );

        // Black pieces on black background...
        bpb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BPawn_B) );
        bnb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BKnight_B) );
        bbb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BBishop_B) );
        brb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BRook_B) );
        bqb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BQueen_B) );
        bkb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A2_BKing_B) );

        // Special bitmaps...
        qmw = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_Qmark_W) );
        qmb = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_Qmark_B) );
        ew  = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_Empty_W) );
        eb  = LoadBitmap ( progInstance, MAKEINTRESOURCE(A1_Empty_B) );
	}
    else
    {
        pieceFont = PIECE_FONT_ORIGINAL;
        // White pieces on white background...
        wpw = LoadBitmap ( progInstance, MAKEINTRESOURCE(WPawn_W) );
        wnw = LoadBitmap ( progInstance, MAKEINTRESOURCE(WKnight_W) );
        wbw = LoadBitmap ( progInstance, MAKEINTRESOURCE(WBishop_W) );
        wrw = LoadBitmap ( progInstance, MAKEINTRESOURCE(WRook_W) );
        wqw = LoadBitmap ( progInstance, MAKEINTRESOURCE(WQueen_W) );
        wkw = LoadBitmap ( progInstance, MAKEINTRESOURCE(WKing_W) );

        // White pieces on black background...
        wpb = LoadBitmap ( progInstance, MAKEINTRESOURCE(WPawn_B) );
        wnb = LoadBitmap ( progInstance, MAKEINTRESOURCE(WKnight_B) );
        wbb = LoadBitmap ( progInstance, MAKEINTRESOURCE(WBishop_B) );
        wrb = LoadBitmap ( progInstance, MAKEINTRESOURCE(WRook_B) );
        wqb = LoadBitmap ( progInstance, MAKEINTRESOURCE(WQueen_B) );
        wkb = LoadBitmap ( progInstance, MAKEINTRESOURCE(WKing_B) );

        // Black pieces on white background...
        bpw = LoadBitmap ( progInstance, MAKEINTRESOURCE(BPawn_W) );
        bnw = LoadBitmap ( progInstance, MAKEINTRESOURCE(BKnight_W) );
        bbw = LoadBitmap ( progInstance, MAKEINTRESOURCE(BBishop_W) );
        brw = LoadBitmap ( progInstance, MAKEINTRESOURCE(BRook_W) );
        bqw = LoadBitmap ( progInstance, MAKEINTRESOURCE(BQueen_W) );
        bkw = LoadBitmap ( progInstance, MAKEINTRESOURCE(BKing_W) );

        // Black pieces on black background...
        bpb = LoadBitmap ( progInstance, MAKEINTRESOURCE(BPawn_B) );
        bnb = LoadBitmap ( progInstance, MAKEINTRESOURCE(BKnight_B) );
        bbb = LoadBitmap ( progInstance, MAKEINTRESOURCE(BBishop_B) );
        brb = LoadBitmap ( progInstance, MAKEINTRESOURCE(BRook_B) );
        bqb = LoadBitmap ( progInstance, MAKEINTRESOURCE(BQueen_B) );
        bkb = LoadBitmap ( progInstance, MAKEINTRESOURCE(BKing_B) );

        // Special bitmaps...
        qmw = LoadBitmap ( progInstance, MAKEINTRESOURCE(Qmark_W) );
        qmb = LoadBitmap ( progInstance, MAKEINTRESOURCE(Qmark_B) );
        ew  = LoadBitmap ( progInstance, MAKEINTRESOURCE(Empty_W) );
        eb  = LoadBitmap ( progInstance, MAKEINTRESOURCE(Empty_B) );
    }

    RawBitmapDX = PieceFont_DX [pieceFont];
    RawBitmapDY = PieceFont_DY [pieceFont];
    bitmapsLoadedFlag = 1;
}


void BoardDisplayBuffer::changePieceFont ( int newPieceFont )
{
    if ( newPieceFont != pieceFont )
    {
        unloadBitmaps();
        pieceFont = newPieceFont;
        loadBitmaps (global_hInstance);
    }
}


// The following function comes from Chapter 13 of Charles Petzold's
// book "Programming Windows 3.1".

void DrawBitmap (
    HDC hdc,
    HBITMAP hBitmap,
    short xStart,
    short yStart )
{
    HDC hdcMem = CreateCompatibleDC(hdc);
    SelectObject ( hdcMem, hBitmap );
    SetMapMode ( hdcMem, GetMapMode(hdc) );

    BITMAP bm;
    GetObject ( hBitmap, sizeof(BITMAP), &bm );
    POINT ptSize;
    ptSize.x = bm.bmWidth;
    ptSize.y = bm.bmHeight;
    DPtoLP ( hdc, &ptSize, 1 );

    POINT ptOrg;
    ptOrg.x = 0;
    ptOrg.y = 0;
    DPtoLP ( hdcMem, &ptOrg, 1 );

    StretchBlt ( 
        hdc, xStart, yStart, 
        (BitmapScaleFactor * ptSize.x) / BitmapScaleDenom, 
        (BitmapScaleFactor * ptSize.y) / BitmapScaleDenom,
        hdcMem, ptOrg.x, ptOrg.y, 
        ptSize.x, ptSize.y,
        SRCCOPY );

    DeleteDC (hdcMem);
}


void DrawOffboard ( HDC hdc, int x, int y, cBOOLEAN whiteViewFlag )
{
    if ( !whiteViewFlag )
    {
        x = 7 - x;
        y = 7 - y;
    }

    RECT rect;
    rect.left   = SQUARE_SCREENX1(x);
    rect.right  = SQUARE_SCREENX2(x)+1;
    rect.top    = SQUARE_SCREENY1(y);
    rect.bottom = SQUARE_SCREENY2(y)+1;

    HBRUSH hbrush = CreateSolidBrush (CHESS_BACKGROUND_COLOR);
    HBRUSH oldBrush = SelectObject ( hdc, hbrush );
    FillRect ( hdc, &rect, hbrush );
    SelectObject ( hdc, oldBrush );
    DeleteObject ( hbrush );
}


void BoardDisplayBuffer::drawSquare (
    HDC hdc,
    SQUARE square,
    int x, int y )
{
   HBITMAP hbm;
   int whiteSquare = (x + y) & 1;

   switch ( square )
   {
      case EMPTY:       hbm = whiteSquare ? ew  : eb;   break;

      case WPAWN:       hbm = whiteSquare ? wpw : wpb;  break;
      case WKNIGHT:     hbm = whiteSquare ? wnw : wnb;  break;
      case WBISHOP:     hbm = whiteSquare ? wbw : wbb;  break;
      case WROOK:       hbm = whiteSquare ? wrw : wrb;  break;
      case WQUEEN:      hbm = whiteSquare ? wqw : wqb;  break;
      case WKING:       hbm = whiteSquare ? wkw : wkb;  break;

      case BPAWN:       hbm = whiteSquare ? bpw : bpb;  break;
      case BKNIGHT:     hbm = whiteSquare ? bnw : bnb;  break;
      case BBISHOP:     hbm = whiteSquare ? bbw : bbb;  break;
      case BROOK:       hbm = whiteSquare ? brw : brb;  break;
      case BQUEEN:      hbm = whiteSquare ? bqw : bqb;  break;
      case BKING:       hbm = whiteSquare ? bkw : bkb;  break;

      case OFFBOARD:    DrawOffboard(hdc,x,y,whiteViewFlag);  return;

      default:          hbm = whiteSquare ? qmw : qmb;
   }

   short xStart, yStart;

   if ( whiteViewFlag )
   {
       xStart = SQUARE_SCREENX1 ( x );
       yStart = SQUARE_SCREENY1 ( y );
   }
   else
   {
       xStart = SQUARE_SCREENX1 ( 7 - x );
       yStart = SQUARE_SCREENY1 ( 7 - y );
   }

   DrawBitmap ( hdc, hbm, xStart, yStart );

   if ( selX == x && selY == y )
   {
      const int inward = 4;
      const int curve = 10;

      int xLeft = xStart + inward;
      int yTop  = yStart + inward;

      int xRight = xStart + CHESS_BITMAP_DX - inward;
      int yBottom = yStart + CHESS_BITMAP_DY - inward;

      HBRUSH oldbrush = SelectObject ( hdc, GetStockObject(NULL_BRUSH) );
      HPEN hpen = CreatePen ( PS_DOT, 0, RGB(127,0,0) );
      HPEN oldpen = SelectObject ( hdc, hpen );
      RoundRect ( hdc, xLeft, yTop, xRight, yBottom, curve, curve );
      SelectObject ( hdc, oldpen );
      SelectObject ( hdc, oldbrush );
   }

   changed[x][y] = cFALSE;
}


void BoardDisplayBuffer::selectSquare ( int x, int y )
{
   if ( selX>=0 && selX<=7 && selY>=0 && selY<=7 )
   {
      changed[selX][selY] = cTRUE;
      freshenSquare ( selX, selY );
   }

   selX = x;
   selY = y;

   if ( selX>=0 && selX<=7 && selY>=0 && selY<=7 )
   {
      changed[selX][selY] = cTRUE;
      freshenSquare ( selX, selY );
   }
}



void BoardDisplayBuffer::drawVector ( HDC hdc )
{
	if ( vx1 != -1 )
	{
		MoveToEx ( hdc, vx1, vy1, NULL );
		LineTo ( hdc, vx2, vy2 );
	}
}



void BoardDisplayBuffer::draw ( HDC hdc,
                                int minx, int maxx,
                                int miny, int maxy )
{
   if ( maxx < 0 || minx > 7 || maxy < 0 || miny > 7 )  return;
   if ( minx < 0 )  minx = 0;
   if ( maxx > 7 )  maxx = 7;
   if ( miny < 0 )  miny = 0;
   if ( maxy > 7 )  maxy = 7;

   if ( !whiteViewFlag )
   {
      int t;

      minx = 7 - minx;
      maxx = 7 - maxx;
      miny = 7 - miny;
      maxy = 7 - maxy;

      // swap 'em

      t = minx;
      minx = maxx;
      maxx = t;

      t = miny;
      miny = maxy;
      maxy = t;
   }

   loadBitmaps (global_hInstance);

   for ( int y=miny; y <= maxy; y++ )
   {
      for ( int x=minx; x <= maxx; x++ )
      {
         // Put the bitmap in the presentation space given.
         drawSquare ( hdc, board[x][y], x, y );
      }
   }

   HBRUSH oldbrush = SelectObject ( hdc, GetStockObject(NULL_BRUSH) );
   HPEN oldpen = SelectObject ( hdc, GetStockObject(BLACK_PEN) );
   Rectangle ( 
           hdc, 
           SQUARE_SCREENX1(0)-1, SQUARE_SCREENY1(7)-1,
           SQUARE_SCREENX2(7)+2, SQUARE_SCREENY2(0)+2 );
   SelectObject ( hdc, oldpen );
   SelectObject ( hdc, oldbrush );
}


void BoardDisplayBuffer::setSquareContents ( int x, int y, SQUARE s )
{
   if ( x>=0 && x<=7 && y>=0 && y<=7 )
   {
      board[x][y] = s;
      changed[x][y] = cTRUE;
   }
}


void BoardDisplayBuffer::update ( const ChessBoard &b )
{
   for ( int x=0; x < 8; x++ )
   {
      for ( int y=0; y < 8; y++ )
      {
         SQUARE s = b.GetSquareContents ( x, y );

         if ( s != board[x][y] )
         {
            board[x][y] = s;
            changed[x][y] = cTRUE;
         }
      }
   }
}


void BoardDisplayBuffer::startReadingMove ( cBOOLEAN whiteIsMoving )
{
   readingMoveFlag = cTRUE;
   gotSource = cFALSE;
   moverIsWhite = whiteIsMoving;
}


cBOOLEAN BoardDisplayBuffer::isReadingMove() const
{
   return readingMoveFlag;
}


void BoardDisplayBuffer::squareSelectedNotify ( int x, int y )
{
   int beepFreq;

   if ( x<0 || x>7 || y<0 || y>7 )
   {
      return;   // It's off the Vulcan board!!!!
   }

   int ofs = OFFSET ( 2+x, 2+y );

   if ( readingMoveFlag )
   {
      int sideMask = moverIsWhite ? WHITE_MASK : BLACK_MASK;
      cBOOLEAN looksValid = cFALSE;

      if ( !gotSource )
      {
         if ( board[x][y] & sideMask )
         {
            looksValid = cTRUE;
            beepFreq = 1000;
            gotSource = cTRUE;
            moveSource = ofs;
            selectSquare ( x, y );
         }
      }
      else
      {
         looksValid = cTRUE;
         beepFreq = 1100;
         moveDest = ofs;
         readingMoveFlag = cFALSE;
         deselectSquare();
      }

      if ( looksValid )
      {
         // Put stuff here to do for all valid-looking square selections!

         ChessBeep ( beepFreq, 50 );
      }
   }
}


void BoardDisplayBuffer::copyMove ( int &source, int &dest )
{
   source = moveSource;
   dest = moveDest;
}


void BoardDisplayBuffer::setView ( int newWhiteView )
{
   whiteViewFlag = newWhiteView;
}


void BoardDisplayBuffer::freshenSquare ( int x, int y )
{
   RECT rect;

   if ( !whiteViewFlag )
   {
      x = 7 - x;
      y = 7 - y;
   }

   rect.left   = SQUARE_SCREENX1(x);
   rect.top    = SQUARE_SCREENY1(y);
   rect.right  = SQUARE_SCREENX2(x);
   rect.bottom = SQUARE_SCREENY2(y);

   InvalidateRect ( HwndMain, &rect, FALSE );
}


void BoardDisplayBuffer::freshenBoard()
{
#if 0
   RECT rect;

   rect.left   = SQUARE_SCREENX1(0);
   rect.top    = SQUARE_SCREENY1(0);
   rect.right  = SQUARE_SCREENX2(7);
   rect.bottom = SQUARE_SCREENY2(7);
#endif

   InvalidateRect ( HwndMain, NULL, FALSE );
}


BoardDisplayBuffer TheBoardDisplayBuffer;


/*--- end of file wbrdbuf.cpp ---*/