/*=========================================================================

     wingui.cpp  -  Copyright (C) 1993-1997 by Don Cross
     email: dcross@intersrv.com
     WWW:   http://www.intersrv.com/~dcross/

     Contains WinMain() for Win32 GUI version of Chenard.

     Revision history:

1997 June 10 [Don Cross]
	Adding an execution profiler so I can figure out where my
	code needs to be optimized.

1997 May 10 [Don Cross]
	Making Edit|Copy Game Listing work when it is the computer's 
	turn to move.

1997 March 8 [Don Cross]
	Adding Game|Allow computer to resign, Game|Resign.

1997 March 3 [Don Cross]
	Changed cBOOLEAN Global_ViewDebugFlag into int Global_AnalysisType.
	Now the allowed values are 0=none, 1=bestpath, 2=currentpath.

1997 February 6 [Don Cross]
     I have decided to move the WAV files in the WinChenard
	 distribution to their own zip file which can be downloaded
	 independently of the executable and readme file.  Therefore,
	 I am making Chenard check for the existence of the necessary
	 *.WAV files before going into View|Speak mode.  If they are not
	 present, I will present a dialog box which tells the user how
	 to get them from the Chenard web page.

1997 January 25 [Don Cross]
     Made Edit|Copy Game Listing self-explanatory by displaying
	 an informational dialog box after the operation completes
	 successfully.  This was in response to an email I received
	 from someone who couldn't figure out what the command was doing.
	 Also added better error checking for anomalous conditions.

1997 January 30 [Don Cross]
     Adding support for Edit|Undo Move and Edit|Redo Move.
	 Adding new Skak font.

1997 January 3 [Don Cross]
     Added support for editing the think time limit for computer
	 players while the game is in progress.
	 Also, put the View|Freshen menu command back in, because
	 Lyle says he saw the display get screwed up once.

1996 August 24 [Don Cross]
     Replaced memory-based learning tree with disk-based tree.

1996 July 29 [Don Cross]
     Copied code from OS/2 Presentation Manager version of Chenard
     as baseline for Win32 GUI version.

1994 February 9 [Don Cross]
     Adding visual feedback for selecting squares.

1994 January 31 [Don Cross]
     Fixed RestoreBoardFromFile so that it calls
     ChessBoard::SaveSpecialMove when it finds an edit command
     in the game file.

1994 February 4 [Don Cross]
     Fixed bug with editing board when viewed from Black's side.
     Fixed bug with Empty side not disabling pieces in edit dialog.

1994 January 12 [Don Cross]:
     Started adding application menu for Game, Edit, View.
     Here are the menu commands I intend to support:

     Game:
        New:   start over at the beginning of the game.
        Open:  read a game from a file and continue it.
        Save, Save As:  save the game to a file.

     Edit:  (most/all options will be disabled if not human's turn)
        Clear board:  make the board blank???
        Set up pieces:  go into "edit mode"???

1993 December 31 [Don Cross]:
     Fixed bug: Title bar text used to say it was White's turn
     when we restored a game from a file and it was either side's
     turn to move.

=========================================================================*/

#include <string.h>
#include <stdio.h>
#include <process.h>
#include <io.h>

#include "chess.h"
#include "winchess.h"
#include "winguich.h"
#include "resource.h"
#include "profiler.h"

const char ChessProgramName[] = CHESS_PROGRAM_NAME;

char Global_GameFilename [256] = "*.GAM";

cBOOLEAN Global_ResetGameFlag    = cFALSE;
cBOOLEAN Global_ClearBoardFlag   = cFALSE;
int		 Global_AnalysisType     = 0;   //0=none, 1=best path, 2=current path
cBOOLEAN Global_GameOpenFlag     = cFALSE;
cBOOLEAN Global_GameSaveFlag     = cFALSE;
cBOOLEAN Global_HaveFilename     = cFALSE;
cBOOLEAN Global_GameModifiedFlag = cFALSE;
cBOOLEAN Global_TacticalBenchmarkFlag = cFALSE;
cBOOLEAN Global_GameOverFlag     = cTRUE;   // It's over before it begins!
cBOOLEAN Global_UndoMoveFlag     = cFALSE;
cBOOLEAN Global_RedoMoveFlag     = cFALSE;
cBOOLEAN Global_AllowResignFlag	 = cFALSE;

ChessUI_win32_gui *Global_UI = 0;
ChessBoard Global_Board;

BoardEditRequest Global_BoardEditRequest;

static char WhiteTimeLimit [64] = "10";
static char BlackTimeLimit [64] = "10";

char TitleBarText_WhitesTurn[] = CHESS_PROGRAM_NAME " - White's turn";
char TitleBarText_BlacksTurn[] = CHESS_PROGRAM_NAME " - Black's turn";
char TitleBarText_GameOver[]   = CHESS_PROGRAM_NAME " - game over";

char GameSave_filename[] = "chess.gam";

LRESULT CALLBACK ChessWndProc ( HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam );


cBOOLEAN Global_AbortFlag = cFALSE;
static cBOOLEAN globalImmedQuitFlag = cFALSE;
static int SoundEnableFlag = 1;
HWND   HwndMain;
HMENU  HmenuMain;
HINSTANCE global_hInstance = NULL;
DWORD global_windowFlags = WS_MINIMIZEBOX | WS_CAPTION | WS_SYSMENU | WS_OVERLAPPED;

DefPlayerInfo DefPlayer;


static void EnableEditPieces ( HWND hwnd, cBOOLEAN enable )
{
    EnableWindow ( GetDlgItem(hwnd,IDC_EDIT_PAWN), enable );
    EnableWindow ( GetDlgItem(hwnd,IDC_EDIT_KNIGHT), enable );
    EnableWindow ( GetDlgItem(hwnd,IDC_EDIT_BISHOP), enable );
    EnableWindow ( GetDlgItem(hwnd,IDC_EDIT_ROOK), enable );
    EnableWindow ( GetDlgItem(hwnd,IDC_EDIT_QUEEN), enable );
    EnableWindow ( GetDlgItem(hwnd,IDC_EDIT_KING), enable );
}


BOOL CALLBACK About_DlgProc (
    HWND hwnd,  // handle to dialog box
    UINT msg,           // message
    WPARAM wparam,  // first message parameter
    LPARAM lparam ) // second message parameter
{
    BOOL result = FALSE;

    switch ( msg )
    {
        case WM_COMMAND:
		{
			switch ( wparam )
			{
				case IDOK:
				{
					EndDialog ( hwnd, IDOK );
					result = TRUE;
				}
				break;
			}
		}
		break;
    }

    return result;
}


BOOL CALLBACK EditThinkTimes_DlgProc (
	HWND hwnd,
	UINT msg,
	WPARAM wparam,
	LPARAM lparam )
{
	BOOL result = FALSE;
	char buffer [128];

	switch ( msg )
	{
		case WM_INITDIALOG:
		{
			// Figure out which players are computer players.
			// Disable think time edit boxes for non-computer players.
			if ( Global_UI->queryWhiteIsHuman() )
			{
				SetWindowText ( GetDlgItem(hwnd,IDC_EDIT_WHITE_TIME), "n/a" );
			    EnableWindow ( GetDlgItem(hwnd,IDC_EDIT_WHITE_TIME), FALSE );
			}
			else
			{
				INT32 timeLimit = Global_UI->queryWhiteThinkTime();
				sprintf ( buffer, "%ld", long(timeLimit/100) );
				SetWindowText ( GetDlgItem(hwnd,IDC_EDIT_WHITE_TIME), buffer );
			}

			if ( Global_UI->queryBlackIsHuman() )
			{
				SetWindowText ( GetDlgItem(hwnd,IDC_EDIT_BLACK_TIME), "n/a" );
				EnableWindow ( GetDlgItem(hwnd,IDC_EDIT_BLACK_TIME), FALSE );
			}
			else
			{
				INT32 timeLimit = Global_UI->queryBlackThinkTime();
				sprintf ( buffer, "%ld", long(timeLimit/100) );
				SetWindowText ( GetDlgItem(hwnd,IDC_EDIT_BLACK_TIME), buffer );
			}
		}
		break;

		case WM_COMMAND:
		{
			switch ( wparam )
			{
				case IDOK:
				{
					if ( Global_UI )
					{
						int timeLimit = 0;
						if ( GetWindowText ( GetDlgItem(hwnd,IDC_EDIT_WHITE_TIME), buffer, sizeof(buffer) ) &&
							(timeLimit = atoi(buffer)) >= 1 )
						{
							Global_UI->setWhiteThinkTime ( INT32(timeLimit) * 100 );
						}

						if ( GetWindowText ( GetDlgItem(hwnd,IDC_EDIT_BLACK_TIME), buffer, sizeof(buffer) ) &&
							(timeLimit = atoi(buffer)) >= 1 )
						{
							Global_UI->setBlackThinkTime ( INT32(timeLimit) * 100 );
						}

						EndDialog ( hwnd, IDOK );
						result = TRUE;
					}
				}
				break;

				case IDCANCEL:
				{
					EndDialog ( hwnd, IDCANCEL );
					result = TRUE;
				}
				break;
			}
		}
		break;
	}

	return result;
}


BOOL CALLBACK EditSquare_DlgProc (
    HWND hwnd,  // handle to dialog box
    UINT msg,           // message
    WPARAM wparam,  // first message parameter
    LPARAM lparam ) // second message parameter
{
   static SQUARE piece = EMPTY;
   static SQUARE side = EMPTY;
   BOOL result = TRUE;
   static SQUARE *squarePtr = 0;
   static SHORT lastPiece = IDC_EDIT_PAWN;
   static SHORT lastSide = IDC_EDIT_EMPTY;

   switch ( msg )
   {
      case WM_INITDIALOG:
      {
         squarePtr = (SQUARE *) (lparam);
         CheckRadioButton ( hwnd, IDC_EDIT_PAWN, IDC_EDIT_KING, lastPiece );
         CheckRadioButton ( hwnd, IDC_EDIT_EMPTY, IDC_EDIT_BLACK, lastSide );
         EnableEditPieces ( hwnd, lastSide != IDC_EDIT_EMPTY );
      }
      break;

      case WM_COMMAND:
      {
         switch ( wparam )
         {
            case IDOK:
            {
               if ( squarePtr )
               {
                  if ( side == EMPTY )
                     *squarePtr = EMPTY;
                  else
                     *squarePtr = PieceLookupTable.sval [ piece | side ];
               }

               EndDialog ( hwnd, IDOK );           
            }
            break;

            case IDCANCEL:
            {
               EndDialog ( hwnd, IDCANCEL );           
            }
            break;

            case IDC_EDIT_PAWN:    piece = P_INDEX;   lastPiece = wparam;   break;
            case IDC_EDIT_KNIGHT:  piece = N_INDEX;   lastPiece = wparam;   break;
            case IDC_EDIT_BISHOP:  piece = B_INDEX;   lastPiece = wparam;   break;
            case IDC_EDIT_ROOK:    piece = R_INDEX;   lastPiece = wparam;   break;
            case IDC_EDIT_QUEEN:   piece = Q_INDEX;   lastPiece = wparam;   break;
            case IDC_EDIT_KING:    piece = K_INDEX;   lastPiece = wparam;   break;

            case IDC_EDIT_EMPTY:
            {
               side = EMPTY;
               lastSide = wparam;
               EnableEditPieces ( hwnd, FALSE );            
            }
            break;

            case IDC_EDIT_WHITE:
            {
               side = WHITE_IND;
               lastSide = wparam;
               EnableEditPieces ( hwnd, TRUE );         
            }
            break;

            case IDC_EDIT_BLACK:
            {
               side = BLACK_IND;
               lastSide = wparam;
               EnableEditPieces ( hwnd, TRUE );         
            }
            break;
         }
      }
      break;

      default:
          result = FALSE;
      break;
   }

   return result;
}


static cBOOLEAN EditSquareDialog ( HWND hwnd, SQUARE &square )
{
   int result = DialogBoxParam ( 
       global_hInstance,
       MAKEINTRESOURCE(IDD_EDIT_SQUARE),
       hwnd,
       DLGPROC(EditSquare_DlgProc),
       LPARAM(&square) );

   cBOOLEAN choseSomething = cFALSE;

   switch ( result )
   {
      case IDOK:
         choseSomething = cTRUE;
         break;
   }

   return choseSomething;
}

BOOL CALLBACK PromotePawn_DlgProc (
    HWND hwnd,          // handle to dialog box
    UINT msg,           // message
    WPARAM wparam,  // first message parameter
    LPARAM lparam ) // second message parameter
{
   BOOL result = FALSE;
   BOOL rook_rb, bishop_rb, knight_rb;
   static SQUARE *promPtr;

   switch ( msg )
   {
      case WM_INITDIALOG:
      {
         // Turn on the 'queen' radio button...
         CheckRadioButton ( hwnd, RB_QUEEN, RB_KNIGHT, RB_QUEEN );

         // Place to put promoted pawn data...
         promPtr = (SQUARE *) lparam;
         result = TRUE;
      }
      break;

      case WM_COMMAND:
      {
         switch ( wparam )
         {
            case IDOK:
            {
               rook_rb   = IsDlgButtonChecked ( hwnd, RB_ROOK );
               bishop_rb = IsDlgButtonChecked ( hwnd, RB_BISHOP );
               knight_rb = IsDlgButtonChecked ( hwnd, RB_KNIGHT );

               if ( rook_rb )
                  *promPtr = R_INDEX;
               else if ( bishop_rb )
                  *promPtr = B_INDEX;
               else if ( knight_rb )
                  *promPtr = N_INDEX;
               else // assume the queen radio button is selected
                  *promPtr = Q_INDEX;

               EndDialog ( hwnd, IDOK );
            }
            break;
         }
         break;
      }
   }

   return result;
}


BOOL CALLBACK DefineChessPlayers_DlgProc (
    HWND hwnd,          // handle to dialog box
    UINT msg,           // message
    WPARAM wparam,      // first message parameter
    LPARAM lparam )     // second message parameter
{
   BOOL result = TRUE;
   const char *defaultTimeLimit = "10";

   switch ( msg )
   {
      case WM_INITDIALOG:
      {
          if ( ChessRandom(2) )
          {
              // Turn on the 'human' radio button for white player...
              CheckRadioButton ( hwnd, RB_WHUMAN, RB_WCOMPUTER, RB_WHUMAN );
     
              // Disable think time for White computer player:
              EnableWindow ( GetDlgItem(hwnd,TB_WTIME), FALSE );
     
              // Turn on the 'computer' radio button for black player...
              CheckRadioButton ( hwnd, RB_BHUMAN, RB_BCOMPUTER, RB_BCOMPUTER );
          }
          else
          {
              // Turn on the 'human' radio button for black player...
              CheckRadioButton ( hwnd, RB_BHUMAN, RB_BCOMPUTER, RB_BHUMAN );
    
              // Disable think time for Black computer player:
              EnableWindow ( GetDlgItem(hwnd,TB_BTIME), FALSE );
    
              // Turn on the 'computer' radio button for white player...
              CheckRadioButton ( hwnd, RB_WHUMAN, RB_WCOMPUTER, RB_WCOMPUTER );
          }

          // Put a default of 10 seconds in each time limit edit box:
          SetWindowText ( GetDlgItem(hwnd,TB_WTIME), WhiteTimeLimit );
          SetWindowText ( GetDlgItem(hwnd,TB_BTIME), BlackTimeLimit );
      }
      break;

      case WM_COMMAND:
      {
         switch ( wparam )
         {
            case RB_WHUMAN:
                 // Disable the white computer player's think time box
                 EnableWindow ( GetDlgItem(hwnd,TB_WTIME), FALSE );
                 break;

            case RB_WCOMPUTER:
                 // Enable the white computer player's think time box
                 EnableWindow ( GetDlgItem(hwnd,TB_WTIME), TRUE );
                 break;

            case RB_BHUMAN:
                 // Disable the black computer player's think time box
                 EnableWindow ( GetDlgItem(hwnd,TB_BTIME), FALSE );
                 break;

            case RB_BCOMPUTER:
                 // Enable the black computer player's think time box
                 EnableWindow ( GetDlgItem(hwnd,TB_BTIME), TRUE );
                 break;

            case IDOK:
            {
               // Get the noisy flag...
               SoundEnableFlag = TRUE;

               // Figure out whether white is human or computer
               int checked = IsDlgButtonChecked ( hwnd, RB_WHUMAN );

               DefPlayer.whiteType = checked ?
                                     DefPlayerInfo::humanPlayer :
                                     DefPlayerInfo::computerPlayer;

               // Think time for white
               int value;

               if ( !GetWindowText ( GetDlgItem(hwnd,TB_WTIME), WhiteTimeLimit, sizeof(WhiteTimeLimit) ) ||
                    (value = atoi(WhiteTimeLimit)) <= 0 )
               {
                   strcpy ( WhiteTimeLimit, defaultTimeLimit );
                   value = atoi(defaultTimeLimit);
               }

               DefPlayer.whiteThinkTime = value;
               DefPlayer.whiteUseSeconds = 1;
               DefPlayer.whiteSearchBias = 1;

               // Figure out whether black is human or computer

               checked = IsDlgButtonChecked ( hwnd, RB_BHUMAN );

               DefPlayer.blackType = checked ?
                                     DefPlayerInfo::humanPlayer :
                                     DefPlayerInfo::computerPlayer;

               // Think time for black

               if ( !GetWindowText ( GetDlgItem(hwnd,TB_BTIME), BlackTimeLimit, sizeof(BlackTimeLimit) ) ||
                    (value = atoi(BlackTimeLimit)) <= 0 )
               {
                   strcpy ( BlackTimeLimit, defaultTimeLimit );
                   value = atoi(defaultTimeLimit);
               }

               DefPlayer.blackThinkTime = value;
               DefPlayer.blackUseSeconds = 1;
               DefPlayer.blackSearchBias = 1;

               // See whether we should restore game from file...
               checked = IsDlgButtonChecked ( hwnd, CB_RESTORE_GAME );
               DefPlayer.restoreGameFlag = 0;
               EndDialog ( hwnd, IDOK );
            }
            break;

            case IDCANCEL:
            {
               EndDialog ( hwnd, IDCANCEL );
            }
            break;
         }
      }
      break;

      default:
         result = FALSE;
         break;
   }

   return result;
}


static cBOOLEAN DefinePlayerDialog ( HWND hwnd )
{
   ULONG fred = DialogBox (
       global_hInstance,
       MAKEINTRESOURCE(IDD_DEFINE_PLAYERS),
       hwnd,
       DLGPROC(DefineChessPlayers_DlgProc) );

   return (fred == IDOK) ? cTRUE : cFALSE;
}


static BOOL SetChessWindowSize ( HWND hwnd, cBOOLEAN showDebugInfo )
{
   RECT rectDesktop;
   GetWindowRect ( GetDesktopWindow(), &rectDesktop );

   int newDX = CHESS_WINDOW_DX;
   int newDY = CHESS_WINDOW_DY;
   int debugDX = 0;

   if ( showDebugInfo )
   {
       debugDX = DEBUG_WINDOW_DX;
       newDX += debugDX;
   }

   RECT newFrameRect;
   newFrameRect.left = (rectDesktop.right - newDX) / 2;
   newFrameRect.right = newFrameRect.left + newDX;
   newFrameRect.top = (rectDesktop.bottom - newDY) / 2;
   newFrameRect.bottom = newFrameRect.top + newDY;

   if ( AdjustWindowRect ( &newFrameRect, global_windowFlags, TRUE ) )
   {
       if ( newFrameRect.left < 0 )
       {
           newFrameRect.right -= newFrameRect.left;
           newFrameRect.left = 0;
       }

       if ( newFrameRect.top < 0 )
       {
           newFrameRect.bottom -= newFrameRect.top;
           newFrameRect.top = 0;
       }

       return SetWindowPos (
           hwnd, HWND_TOP,
           newFrameRect.left,
           newFrameRect.top,
           newFrameRect.right - newFrameRect.left,
           newFrameRect.bottom - newFrameRect.top,
           SWP_SHOWWINDOW );
   }

   return FALSE;   // some error occurred in AdjustWindowRect
}


const char Profile_ViewDebugInfo[]	= "ViewDebugInfo";
const char Profile_PieceStyle[]		= "PieceStyle";
const char Profile_ChessBoardSize[] = "ChessBoardSize";
const char Profile_WhiteTimeLimit[] = "WhiteTimeLimit";
const char Profile_BlackTimeLimit[] = "BlackTimeLimit";
const char Profile_SpeakMovesFlag[] = "SpeakMovesFlag";
const char Profile_ResignFlag[]		= "ResignFlag";


static void LoadChessOptions()
{
    char opt[128];

    GetProfileString ( ChessProgramName, Profile_ViewDebugInfo, "0", opt, sizeof(opt) );
    Global_AnalysisType = atoi(opt);

    GetProfileString ( ChessProgramName, Profile_PieceStyle, "1", opt, sizeof(opt) );
    int temp = atoi(opt);
    TheBoardDisplayBuffer.changePieceFont (temp);

    GetProfileString ( ChessProgramName, Profile_ChessBoardSize, "0", opt, sizeof(opt) );
    temp = atoi(opt);
    NewBoardSize (temp);

    GetProfileString ( ChessProgramName, Profile_WhiteTimeLimit, "10", WhiteTimeLimit, sizeof(WhiteTimeLimit) );
    GetProfileString ( ChessProgramName, Profile_BlackTimeLimit, "10", BlackTimeLimit, sizeof(BlackTimeLimit) );

	GetProfileString ( ChessProgramName, Profile_SpeakMovesFlag, "0", opt, sizeof(opt) );
	Global_SpeakMovesFlag = atoi(opt);

	GetProfileString ( ChessProgramName, Profile_ResignFlag, "0", opt, sizeof(opt) );
	Global_AllowResignFlag = atoi(opt);
}


static void ApplyChessOptions()
{
    // Set the analysis checkmarks
    UINT uCheck = (Global_AnalysisType==0) ? MF_CHECKED : MF_UNCHECKED;
    CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_NONE, uCheck );
    uCheck = (Global_AnalysisType==1) ? MF_CHECKED : MF_UNCHECKED;
	CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_BESTPATH, uCheck );
    uCheck = (Global_AnalysisType==2) ? MF_CHECKED : MF_UNCHECKED;
	CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_CURRENTPATH, uCheck );

	// Set the checkmark for speaking moves...
	uCheck = Global_SpeakMovesFlag ? MF_CHECKED : MF_UNCHECKED;
	CheckMenuItem ( HmenuMain, ID_VIEW_SPEAKMOVESTHROUGHSOUNDCARD, uCheck );

    // Set the checkmarks for the piece styles
    int pfont = TheBoardDisplayBuffer.queryPieceFont();
    uCheck = (pfont == PIECE_FONT_ORIGINAL) ? MF_CHECKED : MF_UNCHECKED;
    CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_ORIGINAL, uCheck );

    uCheck = (pfont == PIECE_FONT_TILBURG) ? MF_CHECKED : MF_UNCHECKED;
    CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_TILBURG, uCheck );

	uCheck = (pfont == PIECE_FONT_SKAK) ? MF_CHECKED : MF_UNCHECKED;
	CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_SKAK, uCheck );

    // Set the checkmarks for board size
    uCheck = (ChessBoardSize==0) ? MF_CHECKED : MF_UNCHECKED;
    CheckMenuItem ( HmenuMain, ID_VIEW_SMALLBOARD, uCheck );

    uCheck = (ChessBoardSize==1) ? MF_CHECKED : MF_UNCHECKED;
    CheckMenuItem ( HmenuMain, ID_VIEW_MEDIUMBOARD, uCheck );

    uCheck = (ChessBoardSize==2) ? MF_CHECKED : MF_UNCHECKED;
    CheckMenuItem ( HmenuMain, ID_VIEW_LARGEBOARD, uCheck );

	// set checkmark for resign flag
	uCheck = Global_AllowResignFlag ? MF_CHECKED : MF_UNCHECKED;
	CheckMenuItem ( HmenuMain, ID_GAME_ALLOWCOMPUTERTORESIGN, uCheck );
}


static void SaveChessOptions()
{
    char opt [64];

    sprintf ( opt, "%d", Global_AnalysisType );
    WriteProfileString ( ChessProgramName, Profile_ViewDebugInfo, opt );

    sprintf ( opt, "%d", TheBoardDisplayBuffer.queryPieceFont() );
    WriteProfileString ( ChessProgramName, Profile_PieceStyle, opt );

    sprintf ( opt, "%d", ChessBoardSize );
    WriteProfileString ( ChessProgramName, Profile_ChessBoardSize, opt );

    WriteProfileString ( ChessProgramName, Profile_WhiteTimeLimit, WhiteTimeLimit );
    WriteProfileString ( ChessProgramName, Profile_BlackTimeLimit, BlackTimeLimit );

	sprintf ( opt, "%d", int(Global_SpeakMovesFlag) );
	WriteProfileString ( ChessProgramName, Profile_SpeakMovesFlag, opt );

	sprintf ( opt, "%d", int(Global_AllowResignFlag) );
	WriteProfileString ( ChessProgramName, Profile_ResignFlag, opt );
}


void ChessThreadFunc ( void * )
{
   Global_Board.Init();
   ChessUI_win32_gui  theUserInterface ( HwndMain );
   Global_UI = &theUserInterface;
   TheBoardDisplayBuffer.update ( Global_Board );
   ChessGame theGame ( Global_Board, theUserInterface );

   for(;;)
   {
       Global_GameOverFlag = cFALSE;
       theGame.Play();
       PostMessage ( HwndMain, UINT(WM_DDC_GAME_OVER), WPARAM(0), LPARAM(0) );

       for(;;)
       {
           Sleep(100);
           int dummySource, dummyDest;
           if ( ProcessChessCommands ( Global_Board, dummySource, dummyDest ) )
               break;
       }
   }

   globalImmedQuitFlag = cTRUE;
   Global_UI = 0;
}


int WINAPI WinMain(
    HINSTANCE hInstance,    // handle to current instance
    HINSTANCE hPrevInstance,    // handle to previous instance
    LPSTR lpCmdLine,        // pointer to command line
    int nCmdShow )      // show state of window
{
   static CHAR chessWindowClass[] = "DDC_Chenard";
   WNDCLASS wndclass;

   global_hInstance = hInstance;

   LoadChessOptions();

   if ( !hPrevInstance )
   {
       wndclass.style       =  CS_HREDRAW | CS_VREDRAW;
       wndclass.lpfnWndProc =  ChessWndProc;
       wndclass.cbClsExtra  =  0;
       wndclass.cbWndExtra  =  0;
       wndclass.hInstance   =  hInstance;
       wndclass.hIcon       =  LoadIcon ( NULL, IDI_APPLICATION );
       wndclass.hCursor     =  LoadCursor ( NULL, IDC_ARROW );
       wndclass.hbrBackground   =  GetStockObject ( WHITE_BRUSH );
       wndclass.lpszMenuName    =  NULL;
       wndclass.lpszClassName   =  chessWindowClass;
       RegisterClass ( &wndclass );
   }

   HmenuMain = LoadMenu ( hInstance, MAKEINTRESOURCE(999) );

   HwndMain = CreateWindow (
       chessWindowClass,
       TitleBarText_WhitesTurn,
       global_windowFlags,
       CW_USEDEFAULT,
       CW_USEDEFAULT,
       CW_USEDEFAULT,
       CW_USEDEFAULT,
       NULL,
       HmenuMain,
       hInstance,
       NULL );

   ApplyChessOptions();
   SetChessWindowSize ( HwndMain, (Global_AnalysisType != 0) );

   HACCEL hAccel = LoadAccelerators ( hInstance, MAKEINTRESOURCE(IDR_ACCELERATOR1) );

   MSG msg;
   msg.wParam = 0;

   unsigned long chessThreadID = _beginthread ( ChessThreadFunc, 0, NULL );
   if ( chessThreadID != (unsigned long)(-1) )
   {
       while ( GetMessage ( &msg, NULL, 0, 0 ) && !Global_AbortFlag )
       {
           if ( !TranslateAccelerator(HwndMain,hAccel,&msg) )
           {
               TranslateMessage (&msg);
               DispatchMessage (&msg);
           }
       }
   }
   else
   {
      // Put error code here!
      ChessBeep ( 880, 200 );
   }

   SaveChessOptions();

   ChessDisplayTextBuffer::DeleteAll();
   ExitProcess(0);   // This is needed to make sure the chess thread is killed
   return 0;
}


static void SetStaticColors ( HWND hwnd, LONG fg, LONG bg )
{
   // WinSetPresParam ( hwnd, PP_FOREGROUNDCOLORINDEX, sizeof(LONG), &fg );
   // WinSetPresParam ( hwnd, PP_BACKGROUNDCOLORINDEX, sizeof(LONG), &bg );
}


static void CheckHumanMoveState ( cBOOLEAN insideHumanMove )
{
   UINT uEnableHuman, uEnableComputer;
   if ( insideHumanMove )
   {
       uEnableHuman     =  MF_BYCOMMAND | MF_ENABLED;
       uEnableComputer  =  MF_BYCOMMAND | MF_GRAYED;
   }
   else
   {
       uEnableHuman     =  MF_BYCOMMAND | MF_GRAYED;
       uEnableComputer  =  MF_BYCOMMAND | MF_ENABLED;
   }

   // Things to enable only when it is a human player's move...
   EnableMenuItem ( HmenuMain, ID_FILE_NEW, uEnableHuman );
   EnableMenuItem ( HmenuMain, ID_FILE_OPEN, uEnableHuman );
   EnableMenuItem ( HmenuMain, ID_FILE_SAVE, uEnableHuman );
   EnableMenuItem ( HmenuMain, ID_FILE_SAVEAS, uEnableHuman );
   EnableMenuItem ( HmenuMain, ID_EDIT_CLEARBOARD, uEnableHuman );
   EnableMenuItem ( HmenuMain, ID_EDIT_EDITMODE, uEnableHuman );
   EnableMenuItem ( HmenuMain, ID_EDIT_UNDOMOVE, uEnableHuman );
   EnableMenuItem ( HmenuMain, ID_EDIT_REDOMOVE, uEnableHuman );

   // Things to enable only when it is a computer player's move...
   EnableMenuItem ( HmenuMain, ID_GAME_FORCEMOVE, uEnableComputer );
}


static void SendEditRequest ( int x, int y, SQUARE s )
{
   // If the following flag is set, it means that an edit
   // is still pending.  We don't want to get confused, so
   // we ignore mouse clicks until then.

   if ( !Global_BoardEditRequest.isValid )
   {
      Global_BoardEditRequest.x = x;
      Global_BoardEditRequest.y = y;
      Global_BoardEditRequest.square = s;
      Global_BoardEditRequest.isValid = cTRUE;
   }
}


static cBOOLEAN CopyGameListing()
{
	const char *copyError = "Copy Game Listing Error";
    if ( Global_Board.HasBeenEdited() )
    {
        MessageBox (
            HwndMain,
            "Cannot get game listing because board has been edited",
            copyError,
            MB_ICONERROR | MB_OK );
        return cFALSE;
    }

    if ( !OpenClipboard(HwndMain) )
	{
		MessageBox (
			HwndMain,
			"Cannot open the Windows clipboard!",
			copyError,
			MB_ICONERROR | MB_OK );
        return cFALSE;
	}

    const unsigned bufSize = 4096;
    char buffer [bufSize];
    GetGameListing ( Global_Board, buffer, bufSize );

    HGLOBAL hGlobalMemory = GlobalAlloc ( GHND, strlen(buffer)+1 );
    if ( hGlobalMemory )
    {
        char far *globalPtr = (char far *) GlobalLock (hGlobalMemory);
        if ( globalPtr )
        {
            char far *a = globalPtr;
            const char far *b = buffer;
            do { *a++ = *b; } while ( *b++ );
            GlobalUnlock (hGlobalMemory);
            EmptyClipboard();
            SetClipboardData ( CF_TEXT, hGlobalMemory );
        }
		else
		{
			GlobalDiscard ( hGlobalMemory );
			MessageBox (
				HwndMain,
				"Error allocating global memory handle!",
				copyError,
				MB_ICONERROR | MB_OK );
			CloseClipboard();
			return cFALSE;
		}
    }

    CloseClipboard();
	return cTRUE;
}


static cBOOLEAN CopyWebPageURL()
{
	const char *copyError = "Copy Web Page URL Error";
    if ( !OpenClipboard(HwndMain) )
	{
		MessageBox (
			HwndMain,
			"Cannot open the Windows clipboard!",
			copyError,
			MB_ICONERROR | MB_OK );
        return cFALSE;
	}

	const char *buffer = "http://www.intersrv.com/~dcross/chenard.html";
    HGLOBAL hGlobalMemory = GlobalAlloc ( GHND, strlen(buffer)+1 );
    if ( hGlobalMemory )
    {
        char far *globalPtr = (char far *) GlobalLock (hGlobalMemory);
        if ( globalPtr )
        {
            char far *a = globalPtr;
            const char far *b = buffer;
            do { *a++ = *b; } while ( *b++ );
            GlobalUnlock (hGlobalMemory);
            EmptyClipboard();
            SetClipboardData ( CF_TEXT, hGlobalMemory );
        }
		else
		{
			GlobalDiscard ( hGlobalMemory );
			MessageBox (
				HwndMain,
				"Error allocating global memory handle!",
				copyError,
				MB_ICONERROR | MB_OK );
			CloseClipboard();
			return cFALSE;
		}
    }

    CloseClipboard();
	return cTRUE;
}


void Chenard_FileSaveAs()
{
    static char filenameBuffer [256];
    strcpy ( filenameBuffer, "*.gam" );

    OPENFILENAME ofn;
    ofn.lStructSize = sizeof(ofn);
    ofn.hwndOwner = HwndMain;
    ofn.hInstance = NULL;
    ofn.lpstrFilter = "Chess Games (*.gam)\0.gam\0";
    ofn.lpstrCustomFilter = NULL;
    ofn.nMaxCustFilter = 0;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = filenameBuffer;
    ofn.nMaxFile = sizeof(filenameBuffer);
    ofn.lpstrFileTitle = NULL;
    ofn.nMaxFileTitle = 0;
    ofn.lpstrInitialDir = NULL;
    ofn.lpstrTitle = "Save Game As";
    ofn.Flags = OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST | OFN_HIDEREADONLY;
    ofn.nFileOffset = 0;
    ofn.nFileExtension = 0;
    ofn.lpstrDefExt = "gam";
    ofn.lCustData = 0;
    ofn.lpfnHook = NULL;
    ofn.lpTemplateName = NULL;

    if ( GetSaveFileName(&ofn) )
    {
        strcpy ( Global_GameFilename, filenameBuffer );
        Global_GameSaveFlag = cTRUE;
    }
}


void Chenard_FileSave()
{
    if ( Global_HaveFilename )
        Global_GameSaveFlag = cTRUE;
    else
        Chenard_FileSaveAs();
}


void Chenard_FileOpen()
{
    static char filenameBuffer [256];
    strcpy ( filenameBuffer, "*.gam" );

    OPENFILENAME ofn;
    ofn.lStructSize = sizeof(ofn);
    ofn.hwndOwner = HwndMain;
    ofn.hInstance = NULL;
    ofn.lpstrFilter = "Chess Games (*.gam)\0.gam\0";
    ofn.lpstrCustomFilter = NULL;
    ofn.nMaxCustFilter = 0;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = filenameBuffer;
    ofn.nMaxFile = sizeof(filenameBuffer);
    ofn.lpstrFileTitle = NULL;
    ofn.nMaxFileTitle = 0;
    ofn.lpstrInitialDir = NULL;
    ofn.lpstrTitle = "Open Game";
    ofn.Flags = OFN_PATHMUSTEXIST | OFN_HIDEREADONLY;
    ofn.nFileOffset = 0;
    ofn.nFileExtension = 0;
    ofn.lpstrDefExt = "gam";
    ofn.lCustData = 0;
    ofn.lpfnHook = NULL;
    ofn.lpTemplateName = NULL;

    if ( GetOpenFileName(&ofn) )
    {
        strcpy ( Global_GameFilename, filenameBuffer );
        Global_GameOpenFlag = cTRUE;
    }
}


static cBOOLEAN WaveFilesExist()
{
	char filename [256];

	// See if 1.WAV through 8.WAV exist...

	for ( char c='1'; c<='8'; c++ )
	{
		sprintf ( filename, "%c.wav", c );
		if ( _access ( filename, 4 ) != 0 )  return cFALSE;
	}

	// See if A.WAV through H.WAV exist...
	for ( c='a'; c<='h'; c++ )
	{
		sprintf ( filename, "%c.wav", c );
		if ( _access ( filename, 4 ) != 0 )  return cFALSE;
	}

	// Check on the WAV files with the piece names

	if ( _access ( "knight.wav", 4 ) != 0 )		return cFALSE;
	if ( _access ( "bishop.wav", 4 ) != 0 )		return cFALSE;
	if ( _access ( "rook.wav", 4 ) != 0 )		return cFALSE;
	if ( _access ( "queen.wav", 4 ) != 0 )		return cFALSE;
	if ( _access ( "king.wav", 4 ) != 0 )		return cFALSE;

	// Other WAV files...

	if ( _access ( "check.wav", 4 ) != 0 )		return cFALSE;
	if ( _access ( "ep.wav", 4 ) != 0 )			return cFALSE;
	if ( _access ( "mate.wav", 4 ) != 0 )		return cFALSE;
	if ( _access ( "movesto.wav", 4 ) != 0 )	return cFALSE;
	if ( _access ( "oo.wav", 4 ) != 0 )			return cFALSE;
	if ( _access ( "ooo.wav", 4 ) != 0 )		return cFALSE;
	if ( _access ( "prom.wav", 4 ) != 0 )		return cFALSE;
	if ( _access ( "stale.wav", 4 ) != 0 )		return cFALSE;
	if ( _access ( "takes.wav", 4 ) != 0 )		return cFALSE;

	return cTRUE;
}


LRESULT CALLBACK ChessWndProc (
    HWND hwnd,
    UINT msg,
    WPARAM wparam,
    LPARAM lparam )
{
   BOOL result = FALSE;
   char buffer [128];
   static cBOOLEAN insideHumanMove = cFALSE;

   switch ( msg )
   {
      case WM_INITMENU:
          CheckHumanMoveState ( insideHumanMove );
          break;

      case WM_TIMER:
	  {
#ifdef CHENARD_PROFILER
	      if ( wparam == 1 )   // make sure it's really our timer
		  {
		      ++ProfilerHitCount[ProfilerIndex];
		  }
#endif
      }
      break;

      case WM_CLOSE:
          if ( Global_GameModifiedFlag )
          {
              int choice = MessageBox (
                  hwnd,
                  "Changes to game will be lost - do you really want to quit?",
                  ChessProgramName,
                  MB_ICONQUESTION | MB_YESNO );

              if ( choice == IDNO )
              {
                  break;   // Don't exit the program after all.
              }
          }
          PostMessage ( hwnd, WM_QUIT, 0, 0 );
          break;

      case WM_CREATE:
      {
          if ( !DefinePlayerDialog(hwnd) )
          {
             Global_AbortFlag = cTRUE;
          }
          else
          {
             const int textSep = 2;
             const int textHeight = 12;   // ???

             ChessDisplayTextBuffer *tb = new ChessDisplayTextBuffer (
                 hwnd, STATIC_ID_WHITES_MOVE, CHESS_BOARD_BORDER_DX, 0,
                 ChessDisplayTextBuffer::margin_bottom, ANSI_VAR_FONT );

             tb = new ChessDisplayTextBuffer (
                 hwnd, STATIC_ID_BLACKS_MOVE, 170, 0,
                 ChessDisplayTextBuffer::margin_bottom, ANSI_VAR_FONT );

             int textY = 8;

             for ( int textID = FIRST_RIGHT_TEXTID, depth=0;
                   textID <= LAST_RIGHT_TEXTID;
                   textID++, depth++ )
             {
                 // Creating the object puts it in the class's list of instances
                 tb = new ChessDisplayTextBuffer (
                     hwnd, textID, 0, textY,
                     ChessDisplayTextBuffer::margin_right, ANSI_FIXED_FONT,
					 (textID>=STATIC_ID_BESTPATH(0)) ? (depth & 1) : 0 );

                 textY += (textHeight + textSep);
             }

             ChessDisplayTextBuffer::RepositionAll();
          }
      }
      break;

      case WM_COMMAND:   // Mostly handles program's menus
      {
         switch ( LOWORD(wparam) )
         {
            case ID_FILE_NEW:
               Global_ResetGameFlag = cTRUE;
               break;

            case ID_FILE_OPEN:
               Chenard_FileOpen();
               break;
    
            case ID_FILE_SAVE:
               Chenard_FileSave();
               break;
    
            case ID_FILE_SAVEAS:
               Chenard_FileSaveAs();
               break;

            case ID_FILE_EXIT:
               SendMessage ( hwnd, WM_CLOSE, 0, 0 );
               break;

            case ID_EDIT_CLEARBOARD:
               Global_ClearBoardFlag = cTRUE;
               break;

			case ID_EDIT_REDOMOVE:
				Global_RedoMoveFlag = cTRUE;
				break;

			case ID_EDIT_UNDOMOVE:
				Global_UndoMoveFlag = cTRUE;
				break;

            case ID_EDIT_EDITMODE:
            {
               int editMode = GetMenuState ( HmenuMain, ID_EDIT_EDITMODE, MF_BYCOMMAND );
               editMode = (editMode ^ MF_CHECKED) & MF_CHECKED;
               CheckMenuItem ( HmenuMain, ID_EDIT_EDITMODE, editMode | MF_BYCOMMAND );
            }
            break;

            case ID_EDIT_COPYGAMELISTING:
			{
               if ( CopyGameListing() )
			   {
			       MessageBox ( 
				      hwnd,
                      "A text listing of the game has been copied to the clipboard.\n"
					  "You can now paste the listing into a text editor, word processor, etc.",
                      CHESS_PROGRAM_NAME,
                      MB_ICONINFORMATION | MB_OK );
			   }
			}
            break;

			case ID_EDIT_COPYCHENARDWEBPAGEURL:
			{
				if ( CopyWebPageURL() )
				{
					MessageBox (
						hwnd,
						"The address (URL) of the Chenard web page has been copied to the Windows clipboard. "
						"You can paste this address into your favorite browser to get the latest "
						"version of this program and read the latest news about Chenard.",
						CHESS_PROGRAM_NAME,
						MB_ICONINFORMATION | MB_OK );
				}
			}
			break;

            case ID_VIEW_ROTATEBOARD:
               TheBoardDisplayBuffer.toggleView();
               TheBoardDisplayBuffer.freshenBoard();
               break;

            case ID_VIEW_FRESHEN:
               TheBoardDisplayBuffer.freshenBoard();
               break;

            case ID_VIEW_PIECESTYLE_ORIGINAL:
            {
                CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_ORIGINAL, MF_CHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_TILBURG,  MF_UNCHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_SKAK,     MF_UNCHECKED );
                if ( TheBoardDisplayBuffer.queryPieceFont() != PIECE_FONT_ORIGINAL )
                {
                    TheBoardDisplayBuffer.changePieceFont (PIECE_FONT_ORIGINAL);
                    ChessDisplayTextBuffer::RepositionAll();
                    SetChessWindowSize ( hwnd, (Global_AnalysisType != 0) );
					TheBoardDisplayBuffer.freshenBoard();
                }
            }
            break;

            case ID_VIEW_PIECESTYLE_TILBURG:
            {
                CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_ORIGINAL, MF_UNCHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_TILBURG,  MF_CHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_SKAK,     MF_UNCHECKED );
                if ( TheBoardDisplayBuffer.queryPieceFont() != PIECE_FONT_TILBURG )
                {
                    TheBoardDisplayBuffer.changePieceFont (PIECE_FONT_TILBURG);
                    ChessDisplayTextBuffer::RepositionAll();
                    SetChessWindowSize ( hwnd, (Global_AnalysisType != 0) );
					TheBoardDisplayBuffer.freshenBoard();
                }
            }
            break;

			case ID_VIEW_PIECESTYLE_SKAK:
			{
                CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_ORIGINAL, MF_UNCHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_TILBURG,  MF_UNCHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_PIECESTYLE_SKAK,     MF_CHECKED );
                if ( TheBoardDisplayBuffer.queryPieceFont() != PIECE_FONT_SKAK )
                {
                    TheBoardDisplayBuffer.changePieceFont (PIECE_FONT_SKAK);
                    ChessDisplayTextBuffer::RepositionAll();
                    SetChessWindowSize ( hwnd, (Global_AnalysisType != 0) );
					TheBoardDisplayBuffer.freshenBoard();
                }
			}
			break;

            case ID_VIEW_SMALLBOARD:
            {
                CheckMenuItem ( HmenuMain, ID_VIEW_SMALLBOARD,  MF_CHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_MEDIUMBOARD, MF_UNCHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_LARGEBOARD,  MF_UNCHECKED );
                NewBoardSize ( 0 );
                ChessDisplayTextBuffer::RepositionAll();
                SetChessWindowSize ( hwnd, (Global_AnalysisType != 0) );
            }
            break;

            case ID_VIEW_MEDIUMBOARD:
            {
                CheckMenuItem ( HmenuMain, ID_VIEW_SMALLBOARD,  MF_UNCHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_MEDIUMBOARD, MF_CHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_LARGEBOARD,  MF_UNCHECKED );
                NewBoardSize ( 1 );
                ChessDisplayTextBuffer::RepositionAll();
                SetChessWindowSize ( hwnd, (Global_AnalysisType != 0) );
            }
            break;

            case ID_VIEW_LARGEBOARD:
            {
                CheckMenuItem ( HmenuMain, ID_VIEW_SMALLBOARD,  MF_UNCHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_MEDIUMBOARD, MF_UNCHECKED );
                CheckMenuItem ( HmenuMain, ID_VIEW_LARGEBOARD,  MF_CHECKED );
                NewBoardSize ( 2 );
                ChessDisplayTextBuffer::RepositionAll();
                SetChessWindowSize ( hwnd, (Global_AnalysisType != 0) );
            }
            break;

            case ID_VIEW_ANALYSIS_NONE:
			{
				CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_NONE, MF_CHECKED | MF_BYCOMMAND );
				CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_BESTPATH, MF_UNCHECKED | MF_BYCOMMAND );
				CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_CURRENTPATH, MF_UNCHECKED | MF_BYCOMMAND );
				Global_AnalysisType = 0;
                ChessDisplayTextBuffer::RepositionAll();
                SetChessWindowSize ( hwnd, (Global_AnalysisType != 0) );
			}
			break;

			case ID_VIEW_ANALYSIS_BESTPATH:
			{
				CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_NONE, MF_UNCHECKED | MF_BYCOMMAND );
				CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_BESTPATH, MF_CHECKED | MF_BYCOMMAND );
				CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_CURRENTPATH, MF_UNCHECKED | MF_BYCOMMAND );
				Global_AnalysisType = 1;
                ChessDisplayTextBuffer::RepositionAll();
                SetChessWindowSize ( hwnd, (Global_AnalysisType != 0) );
			}
			break;

			case ID_VIEW_ANALYSIS_CURRENTPATH:
            {
				CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_NONE, MF_UNCHECKED | MF_BYCOMMAND );
				CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_BESTPATH, MF_UNCHECKED | MF_BYCOMMAND );
				CheckMenuItem ( HmenuMain, ID_VIEW_ANALYSIS_CURRENTPATH, MF_CHECKED | MF_BYCOMMAND );
				Global_AnalysisType = 2;
                ChessDisplayTextBuffer::RepositionAll();
                SetChessWindowSize ( hwnd, (Global_AnalysisType != 0) );
            }
            break;

			case ID_VIEW_SPEAKMOVESTHROUGHSOUNDCARD:
			{
                int speak = GetMenuState ( HmenuMain, ID_VIEW_SPEAKMOVESTHROUGHSOUNDCARD, MF_BYCOMMAND );
                speak = (speak ^ MF_CHECKED) & MF_CHECKED;
				if ( speak )
				{
					if ( WaveFilesExist() )
					{
						Global_SpeakMovesFlag = cTRUE;
					}
					else
					{
						Global_SpeakMovesFlag = cFALSE;
						speak = 0;
						CopyWebPageURL();
						
						MessageBox (
							HwndMain,

"Chenard is unable to speak because "
"one or more WAV files (audio recordings) are missing. "
"Note that these files are distributed separately from "
"WinChenard due to their size and the fact that some people "
"will not want to use this feature.\n\n"
"Please download the file 'chenwav.zip' from the following web page "
"and unzip it into the directory where Chenard is installed.\n\n"
"http://www.intersrv.com/~dcross/chenard.html\n\n"
"As a convenience, this URL has been copied to your Windows clipboard "
"so that you can paste it into your favorite web browser.",

							"Chenard: Cannot speak!",	// address of title of message box  
							MB_ICONERROR | MB_OK );
					}
				}
				else
				{
					Global_SpeakMovesFlag = cFALSE;
				}
				CheckMenuItem ( HmenuMain, ID_VIEW_SPEAKMOVESTHROUGHSOUNDCARD, speak | MF_BYCOMMAND );
			}
			break;
    
            case ID_GAME_FORCEMOVE:
            {
                if ( Global_UI )
                {
                    Global_UI->forceMove();
                }
            }
            break;
    
            case ID_GAME_TACTICALBENCHMARK:
            {
                Global_TacticalBenchmarkFlag = cTRUE;
            }
            break;

			case ID_GAME_ALLOWCOMPUTERTORESIGN:
			{
				int flag = GetMenuState ( HmenuMain, ID_GAME_ALLOWCOMPUTERTORESIGN, MF_BYCOMMAND );
				flag = (flag ^ MF_CHECKED) & MF_CHECKED;
				Global_AllowResignFlag = flag ? cTRUE : cFALSE;
			}
			break;

			case ID_GAME_RESIGN:
			{
				if ( MessageBox ( 
					HwndMain, 
					"Do you really want to resign?", 
					CHESS_PROGRAM_NAME,
					MB_ICONQUESTION | MB_YESNO ) == IDYES )
				{
					extern cBOOLEAN Global_UserResign;
					Global_UserResign = cTRUE;
				}
			}
			break;

			case ID_EDIT_THINK_TIMES:
			{
				if ( Global_UI )
				{
					DialogBox (
						global_hInstance,
						MAKEINTRESOURCE(IDD_EDIT_THINK_TIMES),
						hwnd,
						DLGPROC(EditThinkTimes_DlgProc) );
					break;
				}
			}
			break;

            case ID_HELP_ABOUT:
                DialogBox (
                    global_hInstance,
                    MAKEINTRESOURCE(IDD_ABOUT),
                    hwnd,
                    DLGPROC(About_DlgProc) );
                break;
         }
      }
      break;

      case WM_ERASEBKGND:
      {
         if ( !Global_AbortFlag )
         {
            RECT crect;
            GetClientRect ( hwnd, &crect );
            // Do not erase the background behind the board because the bitmaps
            // are opaque anyway.  Break the remaining area into four rectangles.
            HBRUSH hbrush = CreateSolidBrush ( CHESS_BACKGROUND_COLOR );

            // Erase above the chess board
            RECT rect;
            rect.left = crect.left;
            rect.top = crect.top;
            rect.right = SQUARE_SCREENX2(7)+1;
            rect.bottom = SQUARE_SCREENY1(7)-1;
            FillRect ( HDC(wparam), &rect, hbrush );

            // Erase below the chess board
            rect.top = SQUARE_SCREENY2(0)+1;
            rect.bottom = crect.bottom;
            FillRect ( HDC(wparam), &rect, hbrush );

            // Erase to the left of the chess board
            rect.left = crect.left;
            rect.right = SQUARE_SCREENX1(0)-1;
            rect.top = SQUARE_SCREENY1(7)-1;
            rect.bottom = SQUARE_SCREENY2(0)+1;
            FillRect ( HDC(wparam), &rect, hbrush );

            // Erase to the right of the chess board
            rect.left = SQUARE_SCREENX2(7)+1;
            rect.right = crect.right;
            rect.top = crect.top;
            rect.bottom = crect.bottom;
            FillRect ( HDC(wparam), &rect, hbrush );
            DeleteObject ( hbrush );
            result = TRUE;
         }
      }
      break;

      case WM_PAINT:
      {
         if ( !Global_AbortFlag )
         {
            PAINTSTRUCT ps;
            HDC hdc = BeginPaint ( hwnd, &ps );
            int x1 = SQUARE_CHESSX (ps.rcPaint.left);
            int x2 = SQUARE_CHESSX (ps.rcPaint.right);
            int y1 = SQUARE_CHESSY (ps.rcPaint.bottom);
            int y2 = SQUARE_CHESSY (ps.rcPaint.top);
            TheBoardDisplayBuffer.draw ( hdc, x1, x2, y1, y2 );
            ChessDisplayTextBuffer::DrawAll ( hdc );
            EndPaint ( hwnd, &ps );
            result = TRUE;
         }
      }
      break;

      case WM_LBUTTONUP:
      {
         if ( insideHumanMove )
         {
            int mouseX = LOWORD(lparam);
            int mouseY = HIWORD(lparam);

            int x = SQUARE_CHESSX ( mouseX );
            int y = SQUARE_CHESSY ( mouseY );
            if ( !TheBoardDisplayBuffer.queryWhiteView() )
            {
               // Need to rotate the coords...

               x = 7 - x;
               y = 7 - y;
            }

            int editMode = MF_CHECKED &
            GetMenuState ( HmenuMain, ID_EDIT_EDITMODE, MF_BYCOMMAND );
            if ( editMode )
            {
               SQUARE square = EMPTY;

               if ( Global_GameOverFlag || EditSquareDialog ( hwnd, square ) )
               {
                   SendEditRequest ( x, y, square );
               }
            }
            else
            {
               TheBoardDisplayBuffer.squareSelectedNotify ( x, y );
            }
         }
      }
      break;

      case WM_RBUTTONUP:
      {
         if ( insideHumanMove )
         {
            int mouseX = LOWORD(lparam);
            int mouseY = HIWORD(lparam);

            int editMode = MF_CHECKED &
            GetMenuState ( HmenuMain, ID_EDIT_EDITMODE, MF_BYCOMMAND );

            if ( editMode )
            {
               int x = SQUARE_CHESSX ( mouseX );
               int y = SQUARE_CHESSY ( mouseY );

               if ( !TheBoardDisplayBuffer.queryWhiteView() )
               {
                  // Need to rotate the coords...

                  x = 7 - x;
                  y = 7 - y;
               }

               SendEditRequest ( x, y, Global_BoardEditRequest.square );
            }
         }
      }
      break;

      case WM_DDC_ENTER_HUMAN_MOVE:
      {
         // Getting here means that we have just begun reading
         // a human player's move.  There are certain menu options
         // whose enable state should be changed now.

         insideHumanMove = cTRUE;
         CheckHumanMoveState ( insideHumanMove );
      }
      break;

      case WM_DDC_LEAVE_HUMAN_MOVE:
      {
         // Getting here means we are just now finishing with
         // a human player's move.

         insideHumanMove = cFALSE;
         CheckHumanMoveState ( insideHumanMove );
      }
      break;

      case WM_DDC_FATAL:
      {
         char *errorText = (char *)(lparam);

         MessageBox ( hwnd,
                      errorText,
                      CHESS_PROGRAM_NAME " fatal error",
                      MB_ICONEXCLAMATION | MB_OK );

         PostMessage ( hwnd, WM_QUIT, ULONG(0), ULONG(0) );
      }
      break;

      case WM_DDC_PREDICT_MATE:
	  if ( !Global_SpeakMovesFlag )
      {
         sprintf ( buffer, "Mate in %d", int(wparam) );
         MessageBox ( hwnd,
                      buffer,
                      ChessProgramName,
                      MB_OK );
      }
      *(int *)(lparam) = 1;   // signals that message box is finished
      break;

      case WM_DDC_PROMOTE_PAWN:
      {
         SQUARE *promPtr = (SQUARE *)lparam;
         SQUARE localProm = EMPTY;

         DialogBoxParam ( 
             global_hInstance,
             MAKEINTRESOURCE(IDD_PROMOTE_PAWN),
             hwnd,
             DLGPROC(PromotePawn_DlgProc),
             LPARAM(&localProm) );

         if ( localProm != R_INDEX &&
              localProm != B_INDEX &&
              localProm != N_INDEX &&
              localProm != Q_INDEX )
         {
            localProm = Q_INDEX;
         }

         *promPtr = localProm;
      }
      break;

      case WM_DDC_GAME_RESULT:
      {
         ChessUI_win32_gui::gameReport *report =
            (ChessUI_win32_gui::gameReport *)(lparam);

         char *message;

         switch ( report->winner )
         {
            case SIDE_WHITE:
               message = report->resignation ? "Black resigns" : "White wins";
               break;

            case SIDE_BLACK:
               message = report->resignation ? "White resigns" : "Black wins";
               break;

            case SIDE_NEITHER:
               message = "This game is a draw";
               break;

            default:
               message = "??? I'm confused ???";
               break;
         }

         MessageBox ( hwnd, message, ChessProgramName, MB_OK );
      }
      break;

      case WM_DDC_GAME_OVER:
      {
          insideHumanMove = cTRUE;
          Global_GameOverFlag = cTRUE;
          CheckHumanMoveState ( insideHumanMove );
      }
      break;

	  case WM_DDC_DRAW_VECTOR:
	  {
	      int vofs1 = int(wparam);
		  int vofs2 = int(lparam);
		  int x1 = SQUARE_MIDX ( XPART(vofs1) - 2 );
		  int y1 = SQUARE_MIDY ( YPART(vofs1) - 2 );
		  int x2 = SQUARE_MIDX ( XPART(vofs2) - 2 );
		  int y2 = SQUARE_MIDY ( YPART(vofs2) - 2 );
	      HDC hdc = GetDC ( hwnd );
		  HGDIOBJ oldpen = SelectObject ( hdc, GetStockObject(WHITE_PEN) );
	      int prev = SetROP2 ( hdc, R2_XORPEN );
		  MoveToEx ( hdc, x1, y1, NULL );
		  LineTo ( hdc, x2, y2 );
		  SetROP2 ( hdc, prev );
		  SelectObject ( hdc, oldpen );
		  ReleaseDC ( hwnd, hdc );
	  }
	  break;

      default:
         result = DefWindowProc ( hwnd, msg, wparam, lparam );
         break;
   }

   return result;
}


void ChessFatal ( const char *message )
{
    PostMessage ( HwndMain, UINT(WM_DDC_FATAL), WPARAM(0), LPARAM(message) );
}


void ChessBeep ( int freq, int duration )
{
//    if ( SoundEnableFlag )
//    {
//        Beep ( freq, duration );
//    }
}


//------------------------------------------------------------------

ChessDisplayTextBuffer *ChessDisplayTextBuffer::All = 0;



ChessDisplayTextBuffer::ChessDisplayTextBuffer ( 
    HWND _hwnd, int _id, int _x, int _y, marginType _margin, int _textFont, int _highlight ):
    id ( _id ),
    hwnd ( _hwnd ),
    text ( new char [MAX_CHESS_TEXT+1] ),
    next ( ChessDisplayTextBuffer::All ),
    x ( _x ),
    y ( _y ),
    margin ( _margin ),
    textFont ( _textFont ),
	highlight ( _highlight )
{
    text[0] = '\0';
    All = this;
}

ChessDisplayTextBuffer::~ChessDisplayTextBuffer()
{
    if ( text )
    {
        delete[] text;
        text = 0;
    }
}


void ChessDisplayTextBuffer::reposition()
{
    switch ( margin )
    {
        case margin_right:
            x = CHESS_BITMAP_DX*8 + 10;
            break;

        case margin_bottom:
            y = CHESS_BITMAP_DY*8 + CHESS_BOARD_BORDER_DY + 2;
            break;
    }
}


void ChessDisplayTextBuffer::calcRegion ( RECT &rect ) const
{
    rect.left = x;
    rect.top = y;

    HDC hdc = GetDC(hwnd);
    TEXTMETRIC tm;
    HFONT oldFont = SelectObject ( hdc, GetStockObject(textFont) );
    GetTextMetrics ( hdc, &tm );
    SelectObject ( hdc, oldFont );
    ReleaseDC ( hwnd, hdc );

    int numChars = text ? strlen(text) : 0;
    rect.right = x + numChars * tm.tmAveCharWidth + 3;
    rect.bottom = y + tm.tmHeight + tm.tmExternalLeading + 3;
}


void ChessDisplayTextBuffer::draw ( HDC hdc ) const
{
    HFONT oldFont = SelectObject ( hdc, GetStockObject(textFont) );
    char *p = text ? text : "";
	COLORREF color = highlight ? CHESS_TEXT_COLOR2 : CHESS_TEXT_COLOR;
    COLORREF oldTextColor = SetTextColor ( hdc, color );
    COLORREF oldBkColor   = SetBkColor ( hdc, CHESS_BACKGROUND_COLOR );
    TextOut ( hdc, x, y, p, strlen(p) );
    if ( oldBkColor != CLR_INVALID ) 
        SetBkColor ( hdc, oldBkColor );
    if ( oldTextColor != CLR_INVALID )
        SetTextColor ( hdc, oldTextColor );
    SelectObject ( hdc, oldFont );
}


void ChessDisplayTextBuffer::setText ( const char *newText )
{
    RECT before;
    calcRegion ( before );
    strncpy ( text, newText, MAX_CHESS_TEXT );
    RECT rect;
    calcRegion ( rect );
    if ( before.right > rect.right )    rect.right = before.right;
    if ( before.bottom > rect.bottom )  rect.bottom = before.bottom;
    InvalidateRect ( hwnd, &rect, TRUE );
}

ChessDisplayTextBuffer *ChessDisplayTextBuffer::Find ( int _id )
{
    for ( ChessDisplayTextBuffer *p=All; p; p = p->next )
    {
        if ( p->id == _id )
        {
            return p;
        }
    }

    return 0;
}


void ChessDisplayTextBuffer::DrawAll ( HDC hdc )
{
    for ( ChessDisplayTextBuffer *p=All; p; p=p->next )
    {
        p->draw ( hdc );
    }
}

void ChessDisplayTextBuffer::RepositionAll ()
{
    for ( ChessDisplayTextBuffer *p=All; p; p=p->next )
    {
        p->reposition();
    }
}

void ChessDisplayTextBuffer::DeleteAll()
{
    while ( All )
    {
        ChessDisplayTextBuffer *next = All->next;
        delete All;
        All = next;
    }
}


void ChessDisplayTextBuffer::SetText ( int _id, const char *newText )
{
    ChessDisplayTextBuffer *p = Find(_id);
    if ( p )
    {
        p->setText ( newText );
    }
}


/*--- end of file wingui.cpp ---*/
